/*
  Type0 Font (aka Composite Font)

  from PDF Reference 2nd. ed., p 311
   PDF 1.2 introduces a general architecture for composite fonts that
   theoretically allows a Type 0 font to have multiple descendants, which
   might themselves be Type 0 fonts. However, in versions up to and
   including PDF 1.3, only a single descendant is allowed, which must be
   a CIDFont (not a font). This restriction may be relaxed in a future
   PDF version.
 
  Shunsaku Hirata <hira4scp@mbox.nc.kyushu-u.ac.jp>
*/
#include "config.h"
#include <string.h>
#include <stdio.h>
#include "mem.h"
#include "error.h"
#include "pdfobj.h"
#include "numbers.h"
#include "tfm.h"
#include "pdflimits.h"
#include "cid.h"
#include "type0.h"

static unsigned char verbose = 0;

void type0_set_verbose(void)
{
  cid_set_verbose();
  if (verbose < 255) {
    verbose += 1;
  }
}

int num_type0_fonts = 0, max_type0_fonts;

/* Composite fonts are not supported */
struct a_type0_font 
{
  pdf_obj *indirect;
  char *encoding;
  char *used_chars;
  int cid_id; /* wrong, but still only single descendant is allowed */
  int wmode; /* writing mode, horizontal: 0, vertical: 1 */
} *type0_fonts = NULL;

static void init_a_type0_font (struct a_type0_font *this_type0_font)
{
  this_type0_font->indirect = NULL;
  this_type0_font->encoding = NULL;
  this_type0_font->used_chars = NULL;
  this_type0_font->cid_id = -1;
  this_type0_font->wmode = 0;
}

pdf_obj *type0_font_resource (int type0_id)
{
  if (type0_id>=0 && type0_id<max_type0_fonts) {
    return pdf_link_obj(type0_fonts[type0_id].indirect);
  } else {
    fprintf (stderr, "type0_font_resource(): type0_id=%d", type0_id);
    ERROR ("Invalid font id in type1_font_resource()");
    return NULL;
  }
}

char *type0_font_encoding (int type0_id)
{
  char *result;
  if (type0_id>=0 && type0_id<max_type0_fonts) {
    result = type0_fonts[type0_id].encoding;
  } else {
    fprintf (stderr, "type0_font_used(): type0_id=%d", type0_id);
    ERROR ("Invalid font id in type0_font_encoding()");
  }
  return result;
}

char *type0_font_used (int type0_id)
{
  char *result;
  if (type0_id>=0 && type0_id<max_type0_fonts) {
    result = type0_fonts[type0_id].used_chars;
  } else {
    fprintf (stderr, "type0_font_used(): type0_id=%d", type0_id);
    ERROR ("Invalid font id in type0_font_used()");
  }
  return result;
}

/* Adobe Type0 Font Resources */
int type0_font (const char *map_name, int tfm_font_id, char
		*resource_name, int encoding_id, int remap)
{
  int result = -1;
  int cid_id;
  int wmode;
  char *fullname, *fontname, *encoding;
  int len;
  pdf_obj *font_resource, *tmp1;
#ifdef MEM_DEBUG
  MEM_START
#endif

  /* TODO: more encodings */
  if (encoding_id >= 0) {
    /* ERROR("unexpected encoding id. encording field shoud be `default' or `none'."); */
    return result;
  } else { /* only JIS encoding is supported */
    encoding = NEW(2, char);
    if ((wmode = is_vertical(tfm_font_id)) == 1) {
      sprintf(encoding, "V");
    } else {
      sprintf(encoding, "H");
    }
  }

  if (map_name && (result = cid_find_parent(map_name, encoding)) >= 0) {
  } else if (map_name && ((cid_id = cid_get_id(map_name)) >= 0)) {
    /*
      BaseFont name may contain encoding name,
      while CID-keyed fonts do not contain any encodings.
     */
    if (num_type0_fonts >= max_type0_fonts) {
      max_type0_fonts = MAX (max_type0_fonts+MAX_FONTS, num_type0_fonts+1);
      type0_fonts = RENEW (type0_fonts, max_type0_fonts, struct a_type0_font);
    }

    init_a_type0_font (type0_fonts+num_type0_fonts);
    type0_fonts[num_type0_fonts].cid_id = cid_id;

    font_resource = pdf_new_dict ();
    pdf_add_dict (font_resource,
		  pdf_new_name ("Type"),
		  pdf_new_name ("Font"));
    pdf_add_dict (font_resource,
		  pdf_new_name ("Subtype"),
		  pdf_new_name ("Type0"));
  /*
    Name:
     Name used in font resource dictionary - required only in PDF-1.0
    Note: This entry is obsolescent and its use is no longer recommended.
    (See implementation note 39 in Appendix H., PDF Reference v1.3, 2nd ed.)
  */
    pdf_add_dict (font_resource,
		  pdf_new_name ("Name"),
		  pdf_new_name (resource_name));
    /* No FontDescritor, instead DescendantFonts appears here */
    tmp1 = pdf_new_array ();
    pdf_add_array(tmp1, cid_call_descendant(num_type0_fonts, cid_id));
    pdf_add_dict (font_resource, pdf_new_name ("DescendantFonts"), tmp1);
    /* Future PDF spec. will allow DesecendantFonts to refer multiple fonts */
    /* PostScript Font name */
    fontname = cid_fontname(cid_id);
    if (verbose) fprintf(stderr, "(%s:%s)", "CID", fontname);
    len = strlen(fontname)+strlen(encoding)+2; /* don't forget `-' */
    fullname = NEW (len, char);
    sprintf(fullname, "%s-%s", fontname, encoding);
    pdf_add_dict (font_resource, 
		  pdf_new_name ("BaseFont"),
		  pdf_new_name (fullname));
    RELEASE(fullname);
    pdf_add_dict (font_resource,
		  pdf_new_name ("Encoding"),
		  pdf_new_name(encoding));

    type0_fonts[num_type0_fonts].indirect = pdf_ref_obj(font_resource);
    type0_fonts[num_type0_fonts].encoding = NEW(strlen(encoding)+1, char);
    strcpy(type0_fonts[num_type0_fonts].encoding, encoding) ;
    type0_fonts[num_type0_fonts].wmode = wmode;
    pdf_release_obj (font_resource);
    result = num_type0_fonts;
    num_type0_fonts += 1;
  } else { /* not a CID font */
    result = -1;
  }

  RELEASE(encoding);

#ifdef MEM_DEBUG
  MEM_END
#endif /* MEM_DEBUG */
  return result;
}

void type0_close_all (void)
{
  int i;

  cid_flush_all();
  for (i=0; i<num_type0_fonts; i++) {
    pdf_release_obj(type0_fonts[i].indirect);
    if (type0_fonts[i].encoding) RELEASE(type0_fonts[i].encoding);
    if (type0_fonts[i].used_chars) RELEASE(type0_fonts[i].used_chars);
  }
  if (type0_fonts) RELEASE (type0_fonts);
}
