---
myst:
  html_meta:
    keywords: LaTeX, programmation, commandes, macros, argument optionnel, plusieurs arguments optionnels
---

# Comment obtenir une commande ayant plus d'un argument optionnel ?

Vous savez sans doute qu'on ne peut définir (au plus) qu'un argument optionnel et que celui-ci est nécessairement le premier. L'objet de la présente astuce est de contourner cette limitation.

Toutefois, il faut faire ici attention : une commande avec deux arguments optionnels semble la limite de ce qui est raisonnable en la matière. Si vous souhaitez avoir plus d'arguments optionnels, vous devriez plutôt considérer la méthode décrite à la question "[](arguments-cle-valeur.md)" pour disposer d'une méthode bien plus pratique à programmer et à utiliser.


## Avec les commandes de base

### Commandes enchaînées

Commençons par une remarque assez fondamentale sur la façon dont TeX gère les commandes. D'abord, il faut bien comprendre qu'une commande n'est pas une fonction : il ne s'agit pas d'exécuter à part le code de la commande et de renvoyer un résultat, mais juste de remplacer le nom de la commande par son texte de définition. En particulier, TeX n'opère aucun contrôle sur le texte de définition d'une commande, qui peut contenir des commandes non définies ou n'ayant pas le bon nombre d'arguments.

On peut alors méditer sur le cas d'une commande définie comme `\newcommand\latin{\textit}`. Techniquement, elle semble être une commande sans argument. En pratique, l'utilisateur écrira `\latin{id est}` comme si la commande admettait un argument : celui-ci sera en fait passé à `\textit`. Vous êtes maintenant prêts à comprendre l'exemple suivant illustrant des commandes enchaînées.

```
\documentclass{article}
  \usepackage{lmodern}   % Caractères plus lisibles
  \pagestyle{empty}      % N'affiche pas de numéro de page

  \newcommand*\xvec[1][0]{x_{#1},\ldots,\xvecint}
  \newcommand*\xvecint[1][n]{x_{#1}}

\begin{document}
\begin{enumerate}
\item $\xvec$       % donne x_0,\ldots,x_n
\item $\xvec[1]$    % donne x_1,\ldots,x_n
\item $\xvec[1][m]$ % donne x_1,\ldots,x_m
\item $\xvec[m]$    % donne x_m,\ldots,x_n (attention)
\end{enumerate}
\end{document}
```

Si le dernier argument est optionnel et que vous ne l'utilisez pas, votre commande va manger les espaces qui la suivent, soyez prudent hors du mode mathématique ;

### Commandes emboîtées

Si vous avez déjà lu la question "[](definir_une_macro_a_plus_de_9_arguments.md)", vous pouvez probablement deviner la solution un peu plus générale à ce problème : les commandes emboîtees (ou relais de commande). Voici un exemple de cette technique :

```{noedit}
\newcommand{\maCommande}[1][défaut1]{%
  \def\ArgI{{#1}}%
  \maCommandeRelais
}
\newcommand\maCommandeRelais[1][défaut2]{%
  % le premier argument optionnel est maintenant dans \ArgI
  % le second est dans #1
  ... %
}
```

Bien sûr, `\maCommandeRelais` peut avoir autant d'arguments obligatoires qu'autorisé, après avoir pris en compte celui qui est pris avec son propre argument optionnel, autrement dit jusqu'à 8 arguments. 

Notez également que vous pouvez évidemment étendre la technique pour fournir autant d'arguments optionnels que votre imagination enfiévrée le permet... Mais, c'est à vos risques et périls et, surtout, à ceux de vos utilisateurs.


## Avec l'extension <ctanpkg:twoopt> 

Des variantes de `\newcommand` (et proches parents), avec des noms comme `\newcommandtwoopt`, sont disponibles dans l'extension <ctanpkg:twoopt>. Cependant, si vous le pouvez, il est probablement préférable d'apprendre à écrire les commandes vous-même, juste pour voir pourquoi ce n'est pas forcément une bonne idée du point de vue de la programmation.


## Avec l'extension <ctanpkg:optparams> 

L'extension <ctanpkg:optparams>, de Jonathan Sauer, fournit une commande `\optparams` que vous utilisez comme intermédiaire pour définir des commandes avec jusqu'à neuf arguments optionnels. Sa [documentation](texdoc:optparams) montre des exemples de commandes avec quatre arguments optionnels (alors que cet auteur a créé sa propre extension "clé-valeur"!).


## Avec l'extension <ctanpkg:xargs>

L'extension <ctanpkg:xargs>, de Manuel Pégourié-Gonnard, utilise une extension clé-valeur (<ctanpkg:xkeyval>) pour *définir* la disposition des arguments optionnels. En voici un exemple :

```{noedit}
\usepackage{xargs}
...
\newcommandx{\maCommande}[3][1=1, 3=n]{...}
```

Ce code définit une commande `\maCommande` qui a un premier argument optionnel (par défaut 1), un deuxième argument obligatoire, et un troisième argument optionnel (par défaut n).


## Avec le programme `newcommand`

Une approche alternative est proposée par le [programme](ctanpkg:newcommand) `newcommand` de Scott Pakin : il prend un nom de commande et une définition d'un ensemble d'arguments de commande (dans un langage assez facilement compréhensible) et restitue des macros (La)TeX qui permettent à la commande d'être définie. La commande nécessite qu'un interpréteur Python soit installé sur votre ordinateur.

:::{sources}
[Gérer astucieusement ses arguments](https://www.elzevir.fr/imj/latex/tips.html#argh), par Manuel Pégourié-Gonnard
[More than one optional argument](faquk:FAQ-twooptarg)
:::
