/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains various routines for manipulating raster
 *	data encoded as bits.
 *
 *	There is a bug in the V2.2 C compiler involving optimization of
 *	expressions within loops. When the expression involves a result
 *	that is char or short, and the result cannot be put into a register,
 *	the stack allocator reserves the right length amount on the stack,
 *	but a longword is actually written by the code, resulting in an
 *	over-write of another variable that causes a crash. I have fixed
 *	the ones that my tests have ferreted out, but there may be others
 *	in unexercised code. The solution is to statically allocate a
 *	variable for the invariant expression and use it directly within
 *	the loop instead of the expression, or to reduce the complexity of
 *	the routine so that the value can reside in a register. This
 *	problem may have been fixed in V2.3, which I do not yet have.
 */

#include <ctype.h>

#include "arith.p"
#include "strng.p"

/*
 *	Routine Raster_Word_to_Byte compresses word-aligned
 *	pixel arrays from the PXL file into byte-aligned arrays:
 */

Raster_Word_to_Byte (Width, Height, In, Out)
unsigned short Width, Height;
unsigned long *In;
unsigned char *Out;
{
	auto   int w, h;
	auto   unsigned long *In_Ptr;
	auto   unsigned char *Out_Ptr;
	auto   unsigned long Value;

	In_Ptr = In;
	Out_Ptr = Out;
	for (h = Height; h > 0; h--) {
		for (w = Width; w > 24; w -= 32) {
			Value = *In_Ptr++;
			*Out_Ptr++ = (Value >> 24) & 0xFF;
			*Out_Ptr++ = (Value >> 16) & 0xFF;
			*Out_Ptr++ = (Value >> 8) & 0xFF;
			*Out_Ptr++ = Value & 0xFF;
		}
		if (w > 0) {
			Value = *In_Ptr++;
			*Out_Ptr++ = (Value >> 24) & 0xFF;
			if ((w -= 8) > 0)
				*Out_Ptr++ = (Value >> 16) & 0xFF;
			if ((w -= 8) > 0)
				*Out_Ptr++ = (Value >> 8) & 0xFF;
		}
	}
}

/*
 *	Routine Compress_Raster compresses a byte-aligned raster array
 *	into an equivalent, but smaller, form. Currently this form
 *	allows for the specification of repeated rows, and each row
 *	compressed as follows: repeated bytes are indicated by a count
 *	byte with bit #7 equal to one; byte counts are indicated by
 *	a count byte with bit #7 equal to zero.
 */

unsigned int Compress_Raster (Width, Height, In, Out)
unsigned short Width, Height;
unsigned char *In, *Out;
{
	auto   unsigned char *In_Ptr, *Out_Ptr, *Ptr;
	auto   int Row_Size, Count;
	auto   unsigned short w, h;
	auto   unsigned char c;

	In_Ptr = In;
	Out_Ptr = Out;
	Row_Size = (Width + 7) >> 3;
	for (h = Height; h > 0; h--)
	if (In_Ptr == In || *Ptr == 255 || Compare_Memory_M (In_Ptr, &In_Ptr[-Row_Size], Row_Size) != 0) {
		Ptr = Out_Ptr;
		*Out_Ptr++ = 1;
		for (w = Row_Size; w > 0; ) {
			Count = 0;
			c = *In_Ptr++;
			for (; --w > 0 && Count < 127 && *In_Ptr == c; Count++)
				In_Ptr++;
			if (Count > 0) {
				*Out_Ptr++ = Count | 0x80;
				*Out_Ptr++ = c;
			} else {
				Out_Ptr++;
				*Out_Ptr++ = c;
				for (; Count < 127 && w > 0; w--, Count++) {
					if (*In_Ptr == c) {
						In_Ptr--;
						w++;
						Out_Ptr--;
						Count--;
						break;
					}
					c = *In_Ptr++;
					*Out_Ptr++ = c;
				}
				Out_Ptr[-(Count+2)] = Count;
			}
		}
		if (Out_Ptr - &Ptr[1] >= (Row_Size >> 7) + Row_Size) {	/* No advantage to compressing */
			Out_Ptr = &Ptr[1];
			In_Ptr = &In_Ptr[-Row_Size];
			for (w = Row_Size; w > 0; ) {
				if (w <= 128) {
					Count = w;
					w = 0;
				} else {
					Count = 128;
					w -= 128;
				}
				*Out_Ptr++ = Count - 1;
				Move_Memory (In_Ptr, Out_Ptr, Count);
				In_Ptr = &In_Ptr[Count];
				Out_Ptr = &Out_Ptr[Count];
			}
		}
	} else {
		In_Ptr = &In_Ptr[Row_Size];
		*Ptr += 1;
	}
	return (Out_Ptr - Out);
}

unsigned long Compressed_Size_Func (Width, Height)
unsigned short Width, Height;
{
	auto   unsigned long Size;

	Size = ((Width >> 7) + Width + 1) * (Height - 1) +
	       Width + ((Width + 2) / 3) + 1;
	return (Size);
}

/*
 *	Routine Unpack_Raster_Data unpacks the bits in a byte-aligned
 *	pixel array into individual 0 or 1 bytes:
 */

Unpack_Raster_Data (Width, Height, In, Out)
unsigned short Width, Height;
unsigned char *In, *Out;
{
	auto   unsigned char *In_Ptr, *Out_Ptr;
	auto   unsigned int w, h;
	auto   unsigned char Value, Mask;

	In_Ptr = &In[-1];
	Out_Ptr = Out;
	for (h = Height; h > 0; h--) {
		Mask = 0x01;
		for (w = Width; w > 0; w--) {
			if ((Mask >>= 1) == 0) {
				Mask = 0x80;
				Value = *++In_Ptr;
			}
			*Out_Ptr++ = ((Value & Mask) == 0) ? 0 : 1;
		}
	}
}

/*
 *	Routine Pack_Raster_Data converts an unpacked raster array
 *	to its equivalent packed form:
 */

Pack_Raster_Data (Width, Height, In, Out)
unsigned short Width, Height;
unsigned char *In, *Out;
{
	auto   unsigned char *In_Ptr, *Out_Ptr;
	auto   unsigned int w, h;
	auto   unsigned char Mask;

	In_Ptr = In;
	Out_Ptr = &Out[-1];
	for (h = Height; h > 0; h--) {
		Mask = 0x01;
		for (w = Width; w > 0; w--) {
			if ((Mask >>= 1) == 0) {
				Mask = 0x80;
				*++Out_Ptr = 0;
			}
			if (*In_Ptr++ != 0)
				*Out_Ptr |= Mask;
		}
	}
}

/*
 *	Routine Write_Raster_Column writes a column of bits to
 *	a specified raster:
 */

Write_Raster_Column (Width, Height, Column, Column_Data, Raster)
unsigned short Width, Height, Column;
unsigned char *Column_Data, *Raster;
{
	auto   unsigned char *In_Ptr, *Out_Ptr;
	auto   unsigned short Row_Size, h;
	auto   unsigned char In_Mask, Out_Mask;

	Row_Size = (Width + 7) >> 3;
	In_Ptr = &Column_Data[-1];
	Out_Ptr = &Raster[Column >> 3];
	Out_Mask = 0x80 >> (Column & 0x07);
	In_Mask = 0x01;
	for (h = Height; h > 0; h--) {
		if ((In_Mask >>= 1) == 0) {
			In_Mask = 0x80;
			In_Ptr++;
		}
		if ((*In_Ptr & In_Mask) == 0)
			*Out_Ptr &= ~Out_Mask;
		else
			*Out_Ptr |= Out_Mask;
		Out_Ptr = &Out_Ptr[Row_Size];
	}
}

/*
 *	Routines Scan_Raster_Horizontally and Scan_Raster_Vertically
 *	convert an unpacked pixel array into set of line segments, and
 *	return the number of dark line segments:
 */

unsigned int Scan_Raster_Horizontally (Width, Depth, Raster_Ptr, Segment_Ptr)
unsigned short Width, Depth, (*Segment_Ptr)[3];
unsigned char *Raster_Ptr;
{
	auto   int Wid;
	auto   unsigned char *Ptr, *Column0_Ptr, *ColumnW_Ptr, *End_Ptr;
	auto   unsigned short (*Seg_Ptr)[3], Row;

	Wid = Width;
	End_Ptr = &Raster_Ptr[Wid*Depth];
	Seg_Ptr = Segment_Ptr;
	Row = 0;
	for (Column0_Ptr = Raster_Ptr; Column0_Ptr < End_Ptr; Column0_Ptr = ColumnW_Ptr) {
		ColumnW_Ptr = &Column0_Ptr[Wid];
/*
 *	Scan forward on even numbered rows:
 */
		for (Ptr = Column0_Ptr; Ptr < ColumnW_Ptr; )
		if (*Ptr == 0)
			Ptr++;
		else {
			(*Seg_Ptr)[0] = Row;
			(*Seg_Ptr)[1] = Ptr - Column0_Ptr;
			while (++Ptr < ColumnW_Ptr && *Ptr != 0)
				;
			(*Seg_Ptr)[2] = &Ptr[-1] - Column0_Ptr;
			Seg_Ptr++;
		}
		Row++;
/*
 *	Scan backward on odd numbered rows. This results in a 'back-and
 *	forth' movement which should minimize the distance from the end
 *	of the current line segment to the start of the next line segment:
 */
		if ((Column0_Ptr = ColumnW_Ptr) < End_Ptr) {
			ColumnW_Ptr = &ColumnW_Ptr[Wid];
			for (Ptr = &ColumnW_Ptr[-1]; Ptr >= Column0_Ptr; )
			if (*Ptr == 0)
				Ptr--;
			else {
				(*Seg_Ptr)[0] = Row;
				(*Seg_Ptr)[1] = Ptr - Column0_Ptr;
				while (--Ptr >= Column0_Ptr && *Ptr != 0)
					;
				(*Seg_Ptr)[2] = &Ptr[1] - Column0_Ptr;
				Seg_Ptr++;
			}
			Row++;
		}
	}
	return (Seg_Ptr - Segment_Ptr);
}

unsigned int Scan_Raster_Vertically (Width, Depth, Raster_Ptr, Segment_Ptr)
unsigned short Width, Depth, (*Segment_Ptr)[3];
unsigned char *Raster_Ptr;
{
	auto   int Wid, NegWid;
	auto   unsigned char *Ptr, *Row0_Ptr, *RowD_Ptr, *End_Ptr;
	auto   unsigned short (*Seg_Ptr)[3], Column;

	Wid = Width; NegWid = -Wid;
	End_Ptr = &Raster_Ptr[Wid];
	Seg_Ptr = Segment_Ptr;
	Column = 0;
	RowD_Ptr = &Raster_Ptr[Wid*Depth];
	for (Row0_Ptr = Raster_Ptr; Row0_Ptr < End_Ptr; Row0_Ptr++) {
/*
 *	Scan Downward on even numbered columns:
 */
		for (Ptr = Row0_Ptr; Ptr < RowD_Ptr; )
		if (*Ptr == 0)
			Ptr = &Ptr[Wid];
		else {
			(*Seg_Ptr)[0] = Column;
			(*Seg_Ptr)[1] = (Ptr - Raster_Ptr) / Wid;
			while ((Ptr = &Ptr[Wid]) < RowD_Ptr && *Ptr != 0)
				;
			(*Seg_Ptr)[2] = (&Ptr[NegWid] - Raster_Ptr) / Wid;
			Seg_Ptr++;
		}
		Column++;
		RowD_Ptr++;
/*
 *	Scan Upward on odd numbered columns:
 */
		if (++Row0_Ptr < End_Ptr) {
			for (Ptr = &RowD_Ptr[NegWid]; Ptr >= Row0_Ptr; )
			if (*Ptr == 0)
				Ptr = &Ptr[NegWid];
			else {
				(*Seg_Ptr)[0] = Column;
				(*Seg_Ptr)[1] = (Ptr - Raster_Ptr) / Wid;
				while ((Ptr = &Ptr[NegWid]) >= Row0_Ptr && *Ptr != 0)
					;
				(*Seg_Ptr)[2] = (&Ptr[Wid] - Raster_Ptr) / Wid;
				Seg_Ptr++;
			}
			Column++;
			RowD_Ptr++;
		}
	}
	return (Seg_Ptr - Segment_Ptr);
}

Display_Unpacked_Raster (Width, Height, X_Origin, Y_Origin, Raster_Ptr)
unsigned short Width, Height;
short X_Origin, Y_Origin;
unsigned char *Raster_Ptr;
{
	auto   long W, H, W0, W1, H0, H1;
	auto   unsigned char *Ptr, Pixel, Mask;

	W0 = 0; if (X_Origin < 0) W0 = X_Origin;
	W1 = Width; if (X_Origin+1 > W1) W1 = X_Origin + 1;
	H0 = 0; if (Y_Origin < 0) H0 = Y_Origin;
	H1 = Height; if (Y_Origin+1 > H1) H1 = Y_Origin + 1;
	Ptr = Raster_Ptr;
	for (H = H0; H < H1; H++) {
		Mask = 0x80;
		printf ("%4d ", H);
		if (H >= (long) Height || H < 0) {
			if (H == (long) Y_Origin) {
				for (W = W0; W < (long) X_Origin; W++)
					printf (" ");
				printf ("+");
			}
		} else for (W = W0; W < W1; W++) {
			if (W >= 0 && W < (long) Width) {
				Pixel = *Ptr & Mask;
				if ((Mask >>= 1) == 0) {
					Ptr++;
					Mask = 0x80;
				}
				printf ((H == (long) Y_Origin && W == (long) X_Origin) ?
					((Pixel == 0) ? "+" : "*") : ((Pixel == 0) ? " " : "X"));
			} else
				printf ((H == (long) Y_Origin && W == (long) X_Origin) ? "+" : " ");
		}
		printf ("\n");
		if (Mask != 0x80)
			Ptr++;
	}
}

Show_Raster (Width, Height, Raster)
unsigned short Width, Height;
unsigned char *Raster;
{
	auto   unsigned char *Ptr;
	auto   unsigned short w, h;

	printf ("\n");
	Ptr = Raster;
	for (h = 0; h < Height; h++) {
		for (w = 0; w < Width; w++)
			printf ((*Ptr++ == 0) ? " " : "*");
		printf ("\n");
	}
}

/*
 *	Routine Overlay_Raster overlays two different sized raster
 *	arrays, producing an equivalent raster definition, but at a
 *	different size.
 *
 *	NOTE: the character array used as output is used to keep a count
 *	of the number of "mini-pixels" that are on within each pixel
 *	of the raster. The value is limited to 255, so the size
 *	of an overlay pixel, in terms of mini-pixels, cannot exceed
 *	that number. The guaranteed maximum of this number is the
 *	product of the width and height of the 'base' raster.
 */

#define BLACKNESS 425

Overlay_Raster (Base, Base_Wid, Base_Hgt, Overlay, Ovr_Wid, Ovr_Hgt)
unsigned char *Base, *Overlay;
unsigned short Base_Wid, Base_Hgt, Ovr_Wid, Ovr_Hgt;
{
	auto   unsigned char *Ptr, *Ovr_Ptr;
	auto   unsigned int Size;
	auto   unsigned short x, y, y0, y1;
	auto   unsigned short Base_X_Grid, Base_Y_Grid;
	auto   unsigned short Ovr_X_Grid, Ovr_Y_Grid;
	extern int Gcd();

	if (Ovr_Wid == 0 || Ovr_Hgt == 0)
		return;
/*
 *	Compute the size of the composite grid:
 */
	Size = Gcd (Base_Wid, Ovr_Wid);
	Base_X_Grid = Ovr_Wid / Size;
	Ovr_X_Grid = Base_Wid / Size;
	Size = Gcd (Base_Hgt, Ovr_Hgt);
	Base_Y_Grid = Ovr_Hgt / Size;
	Ovr_Y_Grid = Base_Hgt / Size;
/*
 *	Initialize output array:
 */
	Size = Ovr_Wid * Ovr_Hgt;
	Clear_Memory_M (Overlay, Size);
/*
 *	For each dark pixel of the base raster, determine the set of
 *	mini-pixels that are contained within that cell; increment the
 *	mini-pixel count for each cell in the overlay raster that one
 *	of these mini-pixels falls:
 */
	Ptr = Base;
	for (y = 0; y < Base_Hgt; y++)
	for (x = 0; x < Base_Wid; x++)
	if (*Ptr++ != 0) {
		y1 = ((y + 1) * Base_Y_Grid - 1) / Ovr_Y_Grid;
		Ovr_Ptr = &Overlay[y1*Ovr_Wid];
		Overlay_Row (Ovr_Ptr, x, Base_X_Grid, Ovr_X_Grid, (y + 1) * Base_Y_Grid - y1 * Ovr_Y_Grid);
		y0 = y * Base_Y_Grid / Ovr_Y_Grid;
		if (y0 != y1) {
			Ovr_Ptr = &Overlay[y0*Ovr_Wid];
			Overlay_Row (Ovr_Ptr, x, Base_X_Grid, Ovr_X_Grid, (y0 + 1) * Ovr_Y_Grid - y * Base_Y_Grid);
			while (++y0 < y1) {
				Ovr_Ptr = &Ovr_Ptr[Ovr_Wid];
				Overlay_Row (Ovr_Ptr, x, Base_X_Grid, Ovr_X_Grid, Ovr_Y_Grid);
			}
		}
	}
/*
 *	For each output pixel, determine which ones will be 'on' and
 *	which ones will be 'off'. If the number of 'on' mini-pixels
 *	is >= BLACKNESS per thousand times the number of mini-pixels,
 *	that cell goes 'on':
 */
	x = XN_Div_D_R_M (Ovr_X_Grid * Ovr_Y_Grid, BLACKNESS, 1000);
	for (Ptr = Overlay; Ptr < &Overlay[Size]; )
	if (*Ptr < x)
		*Ptr++ = 0;
	else
		*Ptr++ = 1;
}
#undef BLACKNESS

Overlay_Row (Row_Ptr, x, Base_X_Grid, Ovr_X_Grid, Depth)
unsigned char *Row_Ptr;
unsigned short x, Base_X_Grid, Ovr_X_Grid, Depth;
{
	auto   unsigned char *Ptr;
	auto   unsigned short x0, x1;

	Ptr = Row_Ptr;
	x1 = ((x + 1) * Base_X_Grid - 1) / Ovr_X_Grid;
	Ptr[x1] += ((x + 1) * Base_X_Grid - x1 * Ovr_X_Grid) * Depth;
	x0 = x * Base_X_Grid / Ovr_X_Grid;
	if (x0 != x1) {
		Ptr = &Ptr[x0];
		*Ptr += ((x0 + 1) * Ovr_X_Grid - x * Base_X_Grid) * Depth;
		while (++x0 < x1)
			*++Ptr += Ovr_X_Grid * Depth;
	}
}

/*
 *	Routine Reduce_Raster takes a raster definition and reduces
 *	it to the minimal raster required by trimming off all 'off'
 *	pixels from the top, bottom, left and right sides of the
 *	raster. The raster is reduced in place.
 */

Reduce_Raster (Raster, Width, Height, X_Origin, Y_Origin, New_Width_Ptr,
	       New_Height_Ptr, New_X_Origin_Ptr, New_Y_Origin_Ptr)
unsigned char *Raster;
unsigned short Width, Height, *New_Width_Ptr, *New_Height_Ptr;
short X_Origin, Y_Origin, *New_X_Origin_Ptr, *New_Y_Origin_Ptr;
{
	unsigned char *Ptr, *Ptr1, *Ptr2;
	unsigned short Top, Left, Bottom, Right, w, h;
/*
 *	Determine the (top,left) and (bottom,right) of the minimum
 *	raster:
 */
	Top = Height; Bottom = 0; Left = Width; Right = 0;
	Ptr = Raster;
	for (h = 0; h < Height; h++) {
		for (w = 0; w < Width; w++)
		if (*Ptr++ != 0) {
			if (w < Left) Left = w;
			if (w >= Right) Right = w+1;
			if (h < Top) Top = h;
			if (h >= Bottom) Bottom = h+1;
		}
	}
	if (Bottom < Top)	/* Null raster */
		Top = Left = 0;
	Right -= Left;
	Bottom -= Top;
/*
 *	Compress the raster in place, if necessary:
 */
	if (Top > 0 || Left > 0 || Right < Width || Bottom < Height) {
		Ptr = Raster;
		Ptr1 = &Raster[Top*Width+Left];
		for (h = Bottom; h > 0; h--) {
			Ptr2 = Ptr1;
			for (w = Right; w > 0; w--)
				*Ptr++ = *Ptr2++;
			Ptr1 = &Ptr1[Width];
		}
	}
/*
 *	Compute new width, height and origin:
 */
	*New_Width_Ptr = Right;
	*New_Height_Ptr = Bottom;
	*New_X_Origin_Ptr = X_Origin - Left;
	*New_Y_Origin_Ptr = Y_Origin - Top;
}

/*
 *	Routine 'Generate_Circle_Raster' generates a raster definition
 *	suitable to produce a circle with a given radius. The output
 *	raster must be large enough to hold a raster with width and height
 *	equal to 2*radius+1. The location of the center of the circle
 *	relative to the upper left corner of the raster is (radius,radius).
 *	A circle with radius zero is just a single point. The output raster
 *	is in 'unpacked' form.
 */

#define BLACKNESS 600	/* Per thousand of pixel black before 'on' */

Generate_Circle_Raster (Radius, Raster)
unsigned int Radius;
unsigned char *Raster;
{
	auto   char *Ptr;
	auto   unsigned short x, y, Width;
	auto   int Index, r, t1, t2, t3, t4;
	auto   unsigned long y2, d;
	auto   unsigned short Count;
	static unsigned char Radius_1[] = {
		0,1,0,1,1,1,0,1,0
	}, Radius_2[] = {
		0,1,1,1,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,1,1,1,0
	};
	static unsigned short Filled_Amount[101] = {
		107,109,111,114,118,122,127,131,137,142,
		148,154,161,167,174,181,188,196,203,211,
		219,227,235,244,252,261,270,278,287,296,
		305,315,324,333,343,352,362,371,381,391,
		401,410,420,430,440,450,460,470,480,490,
		500,510,520,530,540,550,560,570,580,590,
		599,609,619,629,638,648,657,667,676,685,
		695,704,713,722,730,739,748,756,765,773,
		781,789,797,804,812,819,826,833,839,846,
		852,858,863,869,873,878,882,886,889,891,
		893
	};
	extern double sqrt();

	r = Radius;
	Width = 2 * r + 1;
/*
 *	Return quickly on three special cases:
 */
	Ptr = Raster;
	if (r <= 2) {
		if (r == 0)
			*Ptr = 1;
		else if (r == 1)
			Move_Memory_M (Radius_1, Ptr, sizeof (Radius_1));
		else
			Move_Memory_M (Radius_2, Ptr, sizeof (Radius_2));
		return;
	}
/*
 *	First, set all pixels to 'on':
 */
	for (Ptr = Raster; Ptr < &Raster[Width*Width]; )
		*Ptr++ = 1;
/*
 *	Now determine which pixels are outside of the bounds of
 *	the given radius. Only 1/8 of the circle needs to be done,
 *	because of the symmetry of the circle:
 */
	Ptr = Raster;
	for (y = XN_Div_D_T_M (r, 707, 1000) + 1; y <= r; y++) {
		y2  = y * y;
		for (x = 0; x <= y; x++) {
			d = x * x + y2;
			if ((Index = (int) (((double) r - sqrt ((double) d) + 1.0) * 100.0)) < 0 ||
			    Index <= 100 && Filled_Amount[Index] < BLACKNESS) {
				t1 = y + r; t2 = x + r; t3 = r - y; t4 = r - x;
				Ptr[t1*Width+t2] = 0;
				Ptr[t1*Width+t4] = 0;
				Ptr[t2*Width+t1] = 0;
				Ptr[t2*Width+t3] = 0;
				Ptr[t3*Width+t2] = 0;
				Ptr[t3*Width+t4] = 0;
				Ptr[t4*Width+t1] = 0;
				Ptr[t4*Width+t3] = 0;
			}
		}
	}
}
#undef BLACKNESS

/*
 *	Routine Reflect_Raster creates a mirror-image raster. The
 *	transformation is done on a packed raster, in place.
 */

Reflect_Raster (Width, Height, Raster)
unsigned short Width, Height;
unsigned char *Raster;
{
	auto   unsigned char *Row_Ptr, *Ptr_A, *Ptr_B;
	auto   unsigned short Raster_Width, w, h;
	auto   unsigned char Mask_A, Mask_B, Bit, Temp;
	static unsigned short Kludge_1;	/* This because of 2.2 compiler bug */
	static unsigned char Kludge_2;	/* Same bug */
	static unsigned int Rev_Mask[16] = {
		0x0, 0x8, 0x4, 0xC, 0x2, 0xA, 0x6, 0xE,
		0x1, 0x9, 0x5, 0xD, 0x3, 0xB, 0x7, 0xF
	};

	if (Width <= 1)
		return;
	Raster_Width = (Width + 7) >> 3;
	Row_Ptr = Raster;
/*
 *	Some optimizations are possible if the width is a multiple of 8:
 */
	if ((Width & 0x07) == 0) {
		Kludge_1 = (Raster_Width + 1) >> 1;
		for (h = Height; h > 0; h--) {
			Ptr_A = Row_Ptr;
			Ptr_B = Row_Ptr = &Ptr_A[Raster_Width];
/*
 *	Start on the ends and converge toward the middle of the
 *	row; check for special cases of middle byte and trivial
 *	palindromes:
 */
			w = Kludge_1;
			do {
				Ptr_B--;
				Temp = *Ptr_A;
				if (Ptr_A != Ptr_B)
					*Ptr_A = (*Ptr_B == 0x00 || *Ptr_B == 0xFF) ? *Ptr_B :
						 (Rev_Mask[*Ptr_B & 0x0F] << 4) | Rev_Mask[(*Ptr_B & 0xF0) >> 4];
				*Ptr_B = (Temp == 0x00 || Temp == 0xFF) ? Temp :
					 (Rev_Mask[Temp & 0x0F] << 4) | Rev_Mask[(Temp & 0xF0) >> 4];
				Ptr_A++;
			} while (--w > 0);
		}
		return;
	}
/*
 *	Otherwise, use brute force:
 */
	Kludge_2 = 0x80 >> (Width & 0x07);
	for (h = Height; h > 0; h--) {
		Ptr_A = Row_Ptr;
		Row_Ptr = &Ptr_A[Raster_Width];
		Ptr_B = &Row_Ptr[-1];
		Mask_A = 0x80;
		Mask_B = Kludge_2;
		for (w = Width >> 1; w > 0; w--) {
			if ((Mask_B <<= 1) == 0) {
				Mask_B = 0x01;
				Ptr_B--;
			}
			Bit = *Ptr_A & Mask_A;
			if ((*Ptr_B & Mask_B) == 0)
				*Ptr_A &= ~Mask_A;
			else
				*Ptr_A |= Mask_A;
			if (Bit == 0)
				*Ptr_B &= ~Mask_B;
			else
				*Ptr_B |= Mask_B;
			if ((Mask_A >>= 1) == 0) {
				Mask_A = 0x80;
				Ptr_A++;
			}
		}
	}
}

/*
 *	Routine Rotate_Raster rotates an unpacked raster description
 *	into one of four possible orientations.
 */

Rotate_Raster (Width, Height, In, Out, Orientation)
unsigned short Width, Height;
unsigned char *In, *Out;
int Orientation;
{
	auto   unsigned char *In_Ptr, *Out_Ptr, *Ptr;
	auto   unsigned short h, w;

	Out_Ptr = Out;
	switch (Orientation) {
	case 0:		/* Normal, no change */
		In_Ptr = In;
		for (h = Height; h > 0; h--)
			for (w = Width; w > 0; w--)
				*Out_Ptr++ = *In_Ptr++;
		break;

	case 1:		/* Rotated 90 degrees counterclockwise */
		Ptr = In;
		for (w = Width; w > 0; w--) {
			In_Ptr = --Ptr;
			for (h = Height; h > 0; h--)
				*Out_Ptr++ = *(In_Ptr+=Width);
		}
		break;

	case 2:		/* Rotated 180 degrees */
		In_Ptr = &In[Width*Height];
		for (h = Height; h > 0; h--)
			for (w = Width; w > 0; w--)
				*Out_Ptr++ = *--In_Ptr;
		break;

	case 3:		/* Rotated 270 degrees counterclockwise */
		Ptr = &In[Width*Height];
		for (w = Width; w > 0; w--) {
			In_Ptr = Ptr++;
			for (h = Height; h > 0; h--)
				*Out_Ptr++ = *(In_Ptr-=Width);
		}
	}
}

/*
 *	Routine Combine_Raster combines a raster with a base raster.
 *	The operations supported are: 0 => turn off bits corresponding
 *	to 'on' bits in the raster, or 1 => turn on bits corresponding
 *	to 'on' bits in the raster. Both rasters are in packed form.
 *
 *	It is assumed the overlay raster is completely contained within
 *	the base raster. It is also assumed that any bits to the right
 *	of the raster required to fill out the last byte are all zero.
 */

Combine_Raster (Base, Base_Width, Base_Height, Raster, Raster_Width,
		Raster_Height, Trunc_Width, X_Offset, Y_Offset, Op)
unsigned char *Base, *Raster;
unsigned short Base_Width, Base_Height, Raster_Width, Raster_Height;
unsigned short Trunc_Width, X_Offset, Y_Offset;
int Op;
{
	auto   unsigned char *In_Ptr, *Out_Ptr, *Base_Ptr, *Raster_Ptr;
	auto   unsigned int Head_Shift, Tail_Shift;
	auto   unsigned short w, h, Base_Wid, Raster_Wid;
	auto   unsigned char Prev_Byte, Next_Byte;

	Base_Wid = (Base_Width + 7) >> 3;
	Base_Ptr = &Base[Y_Offset*Base_Wid+(X_Offset>>3)];
	Raster_Wid = (Raster_Width + 7) >> 3;
	Raster_Ptr = Raster;
	Tail_Shift = X_Offset & 0x07;
	Head_Shift = 8 - Tail_Shift;
	for (h = Raster_Height; h > 0; h--) {
		In_Ptr = Raster_Ptr;
		Out_Ptr = Base_Ptr;
		Prev_Byte = 0;
		for (w = (Trunc_Width + 7) >> 3; w > 0; w--) {
			Next_Byte = (*In_Ptr >> Tail_Shift) | Prev_Byte;
			Prev_Byte = *In_Ptr++ << Head_Shift;
			if (Op == 0)
				*Out_Ptr++ &= ~Next_Byte;
			else
				*Out_Ptr++ |= Next_Byte;
		}
		if (Prev_Byte != 0) {	/* Something left-over */
			if (Op == 0)
				*Out_Ptr &= ~Prev_Byte;
			else
				*Out_Ptr |= Prev_Byte;
		}
		Base_Ptr = &Base_Ptr[Base_Wid];
		Raster_Ptr = &Raster_Ptr[Raster_Wid];
	}
}

/*
 *	Routine Convert_Hex_to_Raster converts a raster description
 *	expressed as a hexadecimal string into the binary equivalent
 *	form. The function returns zero if there was a non-hex digit
 *	in the string, 1 otherwise.
 */

int Convert_Hex_to_Raster (Width, Hex_Str, Out)
unsigned short Width;
char *Hex_Str;
unsigned char *Out;
{
	auto   unsigned char *Out_Ptr;
	auto   char *In_Ptr;
	auto   int Bad, Flag;
	auto   unsigned short Count;
	auto   unsigned char Digit;
	auto   char c;
	extern unsigned char XOrd[];

	Bad = 0;
	Flag = 0;
	Out_Ptr = Out;
	for (In_Ptr = Hex_Str, Count = (Width + 3) >> 2; Count > 0; In_Ptr++, Count--) {
		if ((c = *In_Ptr) == '\0')
			Digit = 0;
		else if (isxdigit (c)) {
			if (isdigit (c))
				Digit = c - '0';
			else
				Digit = XOrd[_toupper(c)] - 55;
		} else {
			Bad++;
			Digit = 0;
		}
		if (Flag == 0)
			*Out_Ptr = Digit << 4;
		else
			*Out_Ptr++ |= Digit;
		Flag ^= 1;
	}
	return ((Bad == 0) ? 1 : 0);
}

/*
 *	Routine Magnify_Raster takes a packed raster description and
 *	generates a new raster where each bit of the original raster
 *	is now (scale X scale) bits in size.
 */

Magnify_Raster (Width, Height, Scale, In, Out)
unsigned short Width, Height;
unsigned int Scale;
unsigned char *In, *Out;
{
	auto   unsigned char *In_Ptr, *Out_Ptr, *Row_Ptr;
	auto   unsigned int Row_Size, Scaled_Row_Size, Count;
	auto   unsigned short w, h;

	In_Ptr = In;
	Out_Ptr = Out;
	Row_Size = (Width + 7) >> 3;
	if (Scale == 1) {
		Move_Memory_M (In_Ptr, Out_Ptr, Row_Size * Height);
		return;
	}
	Scaled_Row_Size = (Width * Scale + 7) >> 3;
	for (h = Height; h > 0; h--) {
		Magnify_Raster_Row (Width, Scale, In_Ptr, Out_Ptr);
		Row_Ptr = Out_Ptr;
		Out_Ptr = &Out_Ptr[Scaled_Row_Size];
		for (Count = Scale - 1; Count > 0; Count--) {
			Move_Memory_M (Row_Ptr, Out_Ptr, Scaled_Row_Size);
			Out_Ptr = &Out_Ptr[Scaled_Row_Size];
		}
		In_Ptr = &In_Ptr[Row_Size];
	}
}

Magnify_Raster_Row (Width, Scale, In, Out)
unsigned short Width;
unsigned int Scale;
unsigned char *In, *Out;
{
	auto   unsigned char *In_Ptr, *Out_Ptr;
	auto   unsigned int Count, Scaled_Row_Size;
	auto   unsigned short w;
	auto   unsigned char In_Mask, Out_Mask, Bit;

	In_Ptr = In;
	Out_Ptr = Out;
	Scaled_Row_Size = (Width * Scale + 7) >> 3;
	if (Scale == 1) {
		Move_Memory_M (In_Ptr, Out_Ptr, Scaled_Row_Size);
		return;
	}
	Clear_Memory_M (Out_Ptr, Scaled_Row_Size);
	In_Mask = Out_Mask = 0x80;
	for (w = Width; w > 0; w--) {
		Bit = *In_Ptr & In_Mask;
		if ((Scale & 0x07) == 0) {	/* Exact multiple of 8 */
			if (Bit == 0)
				Out_Ptr = &Out_Ptr[Scale>>3];
			else for (Count = Scale >> 3; Count > 0; Count--)
				*Out_Ptr++ = 0xFF;
		} else if (Bit == 0) {
			Out_Ptr = &Out_Ptr[Scale>>3];
			for (Count = Scale & 0x07; Count > 0; Count--)
			if ((Out_Mask >>= 1) == 0) {
				Out_Mask = 0x80;
				Out_Ptr++;
			}
		} else for (Count = Scale; Count > 0; Count--) {
			*Out_Ptr |= Out_Mask;
			if ((Out_Mask >>= 1) == 0) {
				Out_Mask = 0x80;
				Out_Ptr++;
			}
		}
		if ((In_Mask >>= 1) == 0) {
			In_Mask = 0x80;
			In_Ptr++;
		}
	}
}

/*
 *	Routine Replicate_Raster takes a raster description and copies
 *	it to the output raster the number of times specified by the
 *	horizontal and vertical replication factors.
 */

Replicate_Raster (Width, Height, In, W_Repl, H_Repl, Out)
unsigned short Width, Height;
unsigned char *In, *Out;
unsigned int W_Repl, H_Repl;
{
	auto   unsigned char *In_Ptr, *Out_Ptr, *Ptr;
	auto   unsigned int Row_Size, Count;
	auto   unsigned short w, h;
	auto   unsigned char In_Mask, Out_Mask;

	Row_Size = (Width * W_Repl + 7) >> 3;
	In_Ptr = In;
	Out_Ptr = Out;
	for (h = Height; h > 0; h--) {
		if ((Width & 0x07) == 0) {	/* Exact multiple of 8 */
			for (Count = W_Repl; Count > 0; Count--) {
				Move_Memory_M (In_Ptr, Out_Ptr, Width>>3);
				Out_Ptr = &Out_Ptr[Width>>3];
			}
		} else {
			Clear_Memory_M (Out_Ptr, Row_Size);
			Out_Mask = 0x80;
			for (Count = W_Repl; Count > 0; Count--) {
				In_Mask = 0x80;
				Ptr = In_Ptr;
				for (w = Width; w > 0; w--) {
					if ((*Ptr & In_Mask) != 0)
						*Out_Ptr |= Out_Mask;
					if ((In_Mask >>= 1) == 0) {
						In_Mask = 0x80;
						Ptr++;
					}
					if ((Out_Mask >>= 1) == 0) {
						Out_Mask = 0x80;
						Out_Ptr++;
					}
				}
			}
			if (Out_Mask != 0x80)
				Out_Ptr++;
		}
		In_Ptr = &In_Ptr[(Width + 7) >> 3];
	}
	Out_Ptr = Out;
	Row_Size *= Height;
	Ptr = &Out_Ptr[Row_Size];
	for (Count = H_Repl - 1; Count > 0; Count--) {
		Move_Memory_M (Out_Ptr, Ptr, Row_Size);
		Ptr = &Ptr[Row_Size];
	}
}

/*
 *	Greatest Common Divisor - Euclid's algorithm:
 */

int Gcd (am, an)
int am, an;
{
	auto   int m, n, r;

	n = am; r = an;
	do {
		m = n;
		n = r;
	} while ((r = m % n) != 0);
	return (n);
}
