(* DVIReader *)

CONST
   ruletablesize  = 300;         (* maximum number of rules in a ruletable    *)
   ruletablesizem = ruletablesize - 1;
   chartablesize  = 3000;        (* maximum number of chars in a chartable    *)
   chartablesizem = chartablesize - 1;
   maxfontspec    = maxstring;   (* max length of a font file specification   *)
   maxspeciallen  = maxstring;   (* maximum length of \special string         *)
   maxTeXchar     = 255;         (* ignore character codes > 255              *)

TYPE
   (* Information about the rules and characters appearing on a page is stored
      in dynamic one-way lists to avoid imposing any limit on their numbers.
      To reduce pointer overheads, the nodes in these lists contain large
      tables (the values of ruletablesize and chartablesize have been chosen
      so that the vast majority of DVI pages will only require one-node lists).
      When interpreting a DVI page, DVIReader adds a new rule or character node
      to the TAIL of the relevant list.  This is done so that when DVItoVDU
      accesses such lists (starting at the head), rules and characters will be
      processed in somewhat the same sequence as seen in the DVI file; i.e.,
      top-to-bottom and left-to-right across the page.
      Since a character list is part of the information stored for a font,
      the precise sequence in which DVI characters are seen is not remembered.
      Font information is also linked together in a one-way list, but the
      ordering is more or less random (see, however, the SortFonts routine).
   *)
   ruleinfoptr = ^ruleinfo;
   ruleinfo    =                          (* a node in a list of ruletables   *)
      RECORD
         rulecount : INTEGER;             (* number of rules in ruletable     *)
         ruletable : ARRAY [0..ruletablesizem] OF
            RECORD
               hp, vp : INTEGER;          (* pixel coords of rule's ref point *)
               wd, ht : INTEGER;          (* dimensions of rule in pixels     *)
            END;
         nextrule  : ruleinfoptr;         (* next node in rule list           *)
      END;

   charinfoptr = ^charinfo;
   charinfo    =                          (* a node in list of chartables     *)
      RECORD
         charcount : INTEGER;             (* number of chars in chartable     *)
         chartable : ARRAY [0..chartablesizem] OF
            RECORD
               hp, vp : INTEGER;          (* pixel coords of char's ref point *)
               code   : 0..maxTeXchar;    (* char's code and pixeltable index *)
            END;
         nextchar  : charinfoptr;         (* next node in char list           *)
      END;

   (* DVIReader uses wd, ht, xo and yo to calculate minhp, minvp, maxhp and
      maxvp.  dwidth and pwidth are used to advance horizontally.
      The mapadr and bitmap fields are not used by DVIReader; they are used
      by FontReader and the main program to load character bitmaps.
   *)
   pixeltableptr = ^pixeltable;
   pixeltable    = ARRAY [0..maxTeXchar] OF
      RECORD
         wd, ht : INTEGER;       (* glyph width and height in pixels; they
                                    define the size of the smallest box
                                    containing all the black pixels           *)
         xo, yo : INTEGER;       (* x and y offsets from top left corner
                                    of glyph to character's reference point   *)
         dwidth : INTEGER;       (* advance width in DVI units computed from
                                    fix width stored in font file             *)
         pwidth : INTEGER;       (* advance width in pixels computed from
                                    fix width stored in font file             *)
         mapadr : INTEGER;       (* byte offset of bitmap info in font file   *)
         bitmap : int_or_mptr;   (* SYSDEP: starting address of bitmap        *)
      END;

   fontinfoptr = ^fontinfo;
   fontinfo    =                        (* a node in list of fonts            *)
      RECORD
         psfont      : BOOLEAN;         (* is this a PostScript font?         *)
         fontused    : BOOLEAN;         (* is font used on current page?      *)
         fontnum     : INTEGER;         (* DVI font number: -2^31 .. 2^30 - 1 *)
         scaledsize  : INTEGER;         (* scaled font size in DVI units      *)
         designsize  : INTEGER;         (* design size in DVI units           *)
         fontarea    : string;          (* explicit font directory            *)
         fontarealen : INTEGER;         (* length of fontarea                 *)
         fontname    : string;          (* font name; e.g., "cmr10"           *)
         fontnamelen : INTEGER;         (* length of font name                *)
         fontspec    : string;          (* complete font file pathname        *)
         fontspeclen : INTEGER;         (* length of fontspec                 *)
         fontexists  : BOOLEAN;         (* can fontspec be opened?            *)
         totalchars  : INTEGER;         (* number of chars from font on page  *)
         charlist    : charinfoptr;     (* head of char information list      *)
         chartail    : charinfoptr;     (* tail of char information list      *)
         pixelptr    : pixeltableptr;   (* allocated once: 1st time font used *)
         nextfont    : fontinfoptr;     (* next node in font list             *)
      END;

   (* For the parameter in MoveToTeXPage: *)
   TeXcounters = ARRAY [0..9] OF INTEGER;
   TeXpageinfo =
      RECORD
         value     : TeXcounters;               (* \count0..\count9 values    *)
         present   : ARRAY [0..9] OF BOOLEAN;   (* is counter relevant?       *)
         lastvalue : 0..9;                      (* last relevant counter      *)
      END;

   (* DVIReader also builds a list of \special info: *)
   specialinfoptr = ^specialinfo;
   specialinfo    = RECORD
                       special     : string;
                       hp, vp      : INTEGER;
                       nextspecial : specialinfoptr;
                    END;


VAR
   (* Most of these should be treated as read-only parameters:                *)
   DVImag       : INTEGER;         (* magnification stored in DVI file        *)
   totalpages   : INTEGER;         (* number of pages in DVI file             *)
   totalfonts   : INTEGER;         (* number of fonts in DVI file             *)
   currDVIpage  : INTEGER;         (* updated by MoveTo... calls              *)
   currTeXpage  : TeXcounters;     (* ditto                                   *)
   totalrules   : INTEGER;         (* number of rules on current page         *)
   rulelist     : ruleinfoptr;     (* head of rule information list           *)
   ruletail     : ruleinfoptr;     (* tail of rule information list           *)
   speciallist  : specialinfoptr;  (* head of \special information list       *)
   fontlist     : fontinfoptr;     (* head of font information list           *)
   currfont     : fontinfoptr;     (* InterpretPage's current font info       *)
   pageempty    : BOOLEAN;         (* is page empty of rules and chars?       *)
   minhp        : INTEGER;         (* minimum horizontal pixel coordinate     *)
   minvp        : INTEGER;         (* minimum vertical pixel coordinate       *)
   maxhp        : INTEGER;         (* maximum horizontal pixel coordinate     *)
   maxvp        : INTEGER;         (* maximum vertical pixel coordinate       *)

PROCEDURE InitDVIReader;                                               EXTERNAL;
PROCEDURE OpenDVIFile (name : string);                                 EXTERNAL;
PROCEDURE SetConversionFactor (resolution, magnification : INTEGER);   EXTERNAL;
PROCEDURE MoveToDVIPage  (n : INTEGER);                                EXTERNAL;
PROCEDURE MoveToNextPage (ascending : BOOLEAN);                        EXTERNAL;
FUNCTION  MoveToTeXPage  (VAR newTeXpage : TeXpageinfo) : BOOLEAN;     EXTERNAL;
FUNCTION  PixelRound (DVIunits : INTEGER) : INTEGER;                   EXTERNAL;
PROCEDURE InterpretPage;                                               EXTERNAL;
PROCEDURE SortFonts (VAR unusedlist : fontinfoptr);                    EXTERNAL;
PROCEDURE CloseDVIFile;                                                EXTERNAL;
