IMPLEMENTATION MODULE regisvdu;

(* Author:         Andrew Trevorrow
   Implementation: Modula-2 under VAX/UNIX 4.2 BSD
   Date Started:   June, 1986

   Description:
   Implements the InitREGIS procedure that initializes the generic VDU routines
   and parameters used in DVItoVDU.
   DVItoVDU assumes text lines start at 1 and increase down the screen.
   The GIGI (VK100), VT125 and VT240 VDUs use ANSI escape sequences to update
   the screen while DVItoVDU is in text mode, so line positions are the same.
   When updating the window in graphics mode (using ShowChar and ShowRectangle),
   DVItoVDU assumes the top left screen pixel is (0,0); h coordinates increase
   to the right and v coordinates increase down.
   The REGIS coordinate scheme for graphics is exactly the same.
*)

FROM vduinterface IMPORT
   DVIstatusl, windowstatusl, messagel, commandl, bottoml,
   windowh, windowv, windowwd, windowht,
   TeXtoASCII,
   StartText, MoveToTextLine, ClearTextLine, ClearScreen,
   StartGraphics, LoadFont, ShowChar, ShowRectangle,
   ResetVDU;

FROM ansivdu IMPORT
   ANSIMoveToTextLine, ANSIClearTextLine;

FROM screenio IMPORT
   Write, WriteCard, WriteString, WriteBuffer;

CONST
   ESC = 33C;
   DEL = 177C;

VAR
   charht : CARDINAL;   (* set in LoadFont and used in ShowChar *)
   lastv  : CARDINAL;   (* ShowChar remembers last vertical coordinate *)

(******************************************************************************)

PROCEDURE InitREGIS;

(* The dialogue region is the top 4 lines.
   The window region is the remaining area of the screen.
*)

BEGIN
DVIstatusl    := 1;
windowstatusl := 2;
messagel      := 3;
commandl      := 4;
bottoml       := 24;
(* DVItoVDU's coordinate scheme is the same as the REGIS scheme. *)
windowh  := 0;
windowv  := 80;      (* = height of 4 dialogue lines (better for LoadFont if
                          windowv is a multiple of 10) *)
windowwd := 768;
windowht := 480 - windowv;

StartText      := REGISStartText;
MoveToTextLine := ANSIMoveToTextLine;
ClearTextLine  := ANSIClearTextLine;
ClearScreen    := REGISClearScreen;
StartGraphics  := REGISStartGraphics;
LoadFont       := REGISLoadFont;
ShowChar       := REGISShowChar;
ShowRectangle  := REGISShowRectangle;
ResetVDU       := REGISResetVDU;

StartGraphics;                 (* for following REGIS commands *)

(* Set Text and Writing attributes to known initial states. *)
(* save current Text attributes; will be restored by ResetVDU *)
WriteString('T(B)');
(* default character set, no italic slant, direction right, default text size *)
WriteString('T(A0,I0,D0,S1)');
(* solid fill, no alternate, normal, shading disabled, overlay *)
WriteString('W(P1,A0,N0,S0,V)');

(* Define some macrographs for frequently used strings in ShowRectangle. *)
WriteString('@.');             (* clear macrograph storage *)
WriteString('@:E]W(S1)P[@;');  (* @E = enable shading for filled rectangle *)
WriteString('@:D]W(S0)@;');    (* @D = disable shading *)
WriteString('@:R]V[]V[+@;');   (* @R = mid part of drawing a row vector *)
WriteString('@:C]V[]V[,+@;');  (* @C = mid part of drawing a column vector *)

StartText;                     (* safer to leave in text mode *)
END InitREGIS;

(******************************************************************************)

PROCEDURE REGISStartText;

(* Note that DVItoVDU will only call MoveToTextLine, ClearTextLine,
   ClearScreen and ResetVDU while in text mode.
   We assume VDU will obey ANSI escape sequences while in text mode.
*)

BEGIN
                 (* SYSDEP: compiler did not like '\' !!! *)
Write(ESC); Write(134C);   (* leave graphics mode *)
END REGISStartText;

(******************************************************************************)

PROCEDURE REGISClearScreen;

BEGIN
Write(ESC);
WriteString('[2J');   (* erase entire screen *)
(* note that VT125 has a separate graphics plane which we need to erase *)
StartGraphics;        (* switch to REGIS mode *)
WriteString('S(E)');  (* erase graphics plane *)
StartText;            (* exit in text mode *)
END REGISClearScreen;

(******************************************************************************)

PROCEDURE REGISStartGraphics;

(* Note that DVItoVDU will only call LoadFont, ShowChar and ShowRectangle
   while in graphics mode.
*)

BEGIN
Write(ESC); WriteString('Pp');   (* enter graphics mode *)
lastv := 999999;                 (* undefined value for next ShowChar call *)
END REGISStartGraphics;

(******************************************************************************)

PROCEDURE REGISLoadFont (fontname : ARRAY OF CHAR;
                         fontsize : CARDINAL;
                         mag, hscale, vscale : REAL);

(* Use the given information to select an appropriate character size
   (based on horizontal AND vertical scaling) for future calls of ShowChar.
*)

VAR wd, ht : CARDINAL;   (* we will send T ( Swd Hht ) *)

BEGIN
Write('T'); Write('(');

(* scale fontsize horizontally and choose an appropriate text width *)
wd := TRUNC( FLOAT(fontsize) * mag * hscale + 0.5 ) DIV 9;
IF wd > 16 THEN wd := 16 END;         (* wd now in 0,1,2,...,16 *)
(* larger widths tend to be too big so adjust accordingly (trial and error) *)
IF wd > 1  THEN wd := wd DIV 2 END;
Write('S'); WriteCard(wd);

(* scale fontsize vertically and choose an appropriate text height *)
ht := TRUNC( FLOAT(fontsize) * mag * vscale + 0.5 ) DIV 10;
IF    ht < 1  THEN ht := 1            (* ht must not be 0 *)
ELSIF ht > 16 THEN ht := 16 END;      (* ht now in 1,2,...,16 *)
charht := ht * 10;                    (* charht now in 10,20,30,...,160 *)
(* restrict charht to <= windowv so screenv-charht in ShowChar will be >= 0 *)
IF charht > CARDINAL(windowv) THEN
   charht := windowv;
   ht     := windowv DIV 10;
END;
(* now reduce charht by one fifth to allow for descenders in ShowChar *)
charht := ((charht * 4) DIV 5) - 1;   (* exact if charht is multiple of 10 *)
Write('H'); WriteCard(ht);
(* Note that VT125 and GIGI VDUs sometimes vary the vertical thickness of text
   (only for odd ht values???).  VT240 does not; instead, charht is sometimes
   1 pixel too much and baseline won't agree with Box/Full characters!
*)

Write(')');
END REGISLoadFont;

(******************************************************************************)

PROCEDURE REGISShowChar (screenh, screenv : CARDINAL;
                         ch : CHAR);

(* Show the given Terse character (mapped to ASCII) using the given position.
   We remember the vertical position in lastv so we can reduce the output
   bytes needed to position the next Terse character on the same line.
   StartGraphics resets lastv to an undefined state (= 999999).
*)

VAR newch : CHAR;   (* = TeXtoASCII[ch] *)

BEGIN
Write('P'); Write('[');
WriteCard(screenh);
(* charht allows for descenders and is used to shift ref pt of REGIS ch
   (top left pixel) so that REGIS and TeX baselines will match.
   LoadFont guarantees that screenv - charht >= 0.
*)
DEC(screenv,charht);
IF lastv <> screenv THEN           (* we need to send new vertical coordinate *)
   Write(',');
   WriteCard(screenv);
   lastv := screenv;               (* remember for next ShowChar call *)
END;
Write(']');
Write('T');
newch := TeXtoASCII[ch];           (* convert TeX ch to ASCII *)
IF newch <> "'" THEN
   Write("'");   (* open quoted string *)
   IF newch <> '?' THEN
      (* newch is similar to TeX ch *)
      Write(newch);
   ELSE
      (* attempt to display something other than ? *)
      CASE ch OF
      13C..17C :   (* ff, fi, fl, ffi, ffl *)
          Write('f');
          (* REGIS doesn't care if no room at right edge *)
          CASE ch OF
          13C : Write('f') |
          14C : Write('i') |
          15C : Write('l') |
          16C,
          17C : Write('f');
                IF ch = 16C THEN
                   Write('i');
                ELSE
                   Write('l');
                END;
          END;
          |
      31C : Write('B');   (* German sharp S *)
          |
      32C, 33C, 35C, 36C :   (* diphthongs: ae, oe, AE, OE *)
          CASE ch OF
          32C : Write('a') |
          33C : Write('o') |
          35C : Write('A') |
          36C : Write('O')
          END;
          CASE ch OF
          32C, 33C : Write('e') |
          35C, 36C : Write('E')
          END;
          |
      40C : Write("/");   (* Polish suppressed l and L *)
      ELSE
          Write('?');
      END;
   END;
   Write("'");   (* close quoted string *)
ELSE
   Write('"'); Write("'"); Write('"');     (* send "'" *)
END;
END REGISShowChar;

(******************************************************************************)

PROCEDURE REGISShowRectangle (screenh, screenv,          (* top left pixel *)
                              width, height : CARDINAL;  (* size of rectangle *)
                              ch : CHAR);                (* black pixel *)

(* Display the given rectangle (without using the given black pixel character).
   DVItoVDU ensures the top left position is visible and the given
   dimensions do not go beyond the window edges.
*)

BEGIN
IF height = 1 THEN                          (* show row vector *)
   Write('P'); Write('[');                  (* move cursor to start of row *)
   WriteCard(screenh); Write(',');
   WriteCard(screenv);
   (* call @R macrograph to draw starting pixel and begin row *)
   Write('@'); Write('R');
   WriteCard(width-1); Write(']');
ELSIF width = 1 THEN                        (* show column vector *)
   Write('P'); Write('[');                  (* move cursor to start of column *)
   WriteCard(screenh); Write(',');
   WriteCard(screenv);
   (* call @C macrograph to draw starting pixel and begin column *)
   Write('@'); Write('C');
   WriteCard(height-1); Write(']');
ELSE
   (* assume height and width > 1 and use shading to fill rectangle *)
   Write('P'); Write('['); Write(',');      (* position to last row *)
   WriteCard(screenv+height-1);
   (* call @E macrograph to define shading reference line and start
      position command that moves to start of first row *)
   Write('@'); Write('E');
   WriteCard(screenh); Write(',');
   WriteCard(screenv);
   (* call @R macrograph to draw starting pixel and begin rectangle *)
   Write('@'); Write('R');
   WriteCard(width-1);
   (* call @D macrograph to disable shading *)
   Write('@'); Write('D');
END;
END REGISShowRectangle;

(******************************************************************************)

PROCEDURE REGISResetVDU;

(* We don't do a hardware reset, but leave VDU gracefully. *)

BEGIN
StartGraphics;             (* for following REGIS commands *)
WriteString('@.');         (* clear macrograph storage *)
WriteString('T(E)');       (* restore Text attributes saved in InitREGIS *)
StartText;                 (* safer to leave in text mode *)
END REGISResetVDU;

(******************************************************************************)

BEGIN
END regisvdu.
