MODULE DVItoVDU;

(* Author:         Andrew Trevorrow, University of Adelaide
   Implementation: University of Hamburg Modula-2 under VAX/VMS version 4
   Date Started:   July, 1984
   Released:       September, 1984 (version 1.0)

   Description:
   DVItoVDU allows pages from a DVI file produced by TeX82 to be viewed on a
   variety of VDU screens.  The program is interactive; see the
   "DVItoVDU USER GUIDE" for details on the user interface.
   People wishing to install or modify the program should read the
   "DVItoVDU SYSTEM GUIDE".

   Notes:
 - Debugging code is bracketed by (* DEBUG *) ... (* GUBED *).
   This code will be commented out in the final working version.
 - System-dependent code is indicated by the string "SYSDEP".
 - Uncertain or unfinished code is indicated by the string "???".
 - Procedures are defined in a top-down manner; each procedure is
   usually defined as soon as possible after its first use.
 - The above notes are also true for all the local modules used by DVItoVDU.

   Revised:
   October, 1984
 - No longer use ClearWindow since some VDUs can't do this
   quickly; DVItoVDU now repaints the dialogue region as well
   as the window region when the latter is updated.
 - Multiple commands are allowed and processed left to right
   as if they had occurred individually, except that the
   window region is only updated, if necessary, at the end.
 - Window can now move entirely outside page rectangle, but
   only just outside so the user can easily move back inside.
 - If the Window command has no parameters then we move the
   window to minhp,minvp (top left corner of page rectangle).
 - Released version 1.1 in October, 1984

   December, 1984
 - LoadFont allows VDU to download given font, or to simulate
   characters using given font size (scaledsize in pixels).
   ShowChar no longer passes the font size.
 - StartGraphics called when VDU is about to draw in window
   region; this makes some implementations of ShowRectangle more efficient.
   StartText called when VDU is about to update dialogue region.
   January, 1985
 - No longer use ShowPixel.
 - Implemented Hsize and Vsize commands that allow user to
   scale display by changing the size of the window region.
 - Implemented Xsize and Ysize commands that allow user to
   change the paper size DVItoVDU assumes will be used.
 - After selecting a page, we set window size and location to
   useful values (depending on the relative sizes of paper
   and unscaled window region, and the position of the page).
 - Display visible outlines of paper.
 - Delay checking of page edges as long as possible.
 - After interpreting a page, font table is reordered so that
   fonts with fewest characters will be displayed first.
 - Allow user to change display mode while window region is
   being updated (by typing Terse, Box or Full commands).
 - Fixed problem with negative dimensions.
 - Hit RETURN instead of CTRL-Z to abort page display.
   February, 1985
 - Handle integers and dimensions more like TeX.
 - Warn user that we ignore character codes > maxTeXchar.
 - DisplayOneChar is assigned FullChar2 when h/vscalefactor
   < 1.0 (and avoids drawing overlapping rows), otherwise
   it is assigned FullChar1 (now more efficient).
 - Released version 1.2 in February, 1985

   May, 1985
 - maxrules increased from 500 to 1000.
 - Show command also shows fonts NOT used on current page.
 - All terminal i/o done using ScreenIO routines.
 - Get character info from PXL files instead of RST files.
   June, 1985
 - DCL command line processing now done in DCLInterface.
   resolution, mag, paperwd, paperht, fontdir, dummyfont,
   helpname and vdu now all user-settable via VMS qualifiers.
   See DVITOVDU.CLD for qualifier names and default values.
 - Show command info is now paginated and shows qualifiers.
 - /XSIZE and /YSIZE qualifiers make Xsize and Ysize redundant.
 - Use Halt from ScreenIO to restore terminal channel and
   avoid CLI message (Hamburg's HALT generates such a message).
   E.g., Halt(2) halts program and sets $SEVERITY to 2 (=error).
   Halt(1) halts program and sets $SEVERITY to 1 (=success).
 - Coordinate pairs now shown as (h,v) instead of (v,h) to
   be closer to (x,y) Cartesian coordinates.  Parameter order
   in some routines has changed to reflect this, as has window
   status line and Window command.
 - Check for white or non-existent glyph in Box & FullChars.
   If pixel table's mapadr for current character = 0 then glyph
   is all white (if fix <> 0) or absent.
   July, 1985
 - Changes made to be consistent with new TeX and IMPRINT
   commands from Kellerman & Smith:
   1. fontdir now points to a list of string values.
   2. /XSIZE and /YSIZE are specified as dimensions with units
      in IN, CM, MM, PC, PT or PX.  Their values are converted
      to the nearest pixel and exported via paperwd and paperht.
   3. The above two-letter sequences are also the new DVItoVDU
      commands for changing the current units of dimensions.
 - Integers in a TeX page specification are now separated by
   periods instead of commas.  This is also more consistent
   with TeX's output and IMPRINT's /pages qualifier.
 - Avoid slow start-up by building complete fontspec for a
   font only when first required (in DoFont or ShowStatistics).
   ProcessFontDefs no longer uses mag to build fontspec;
   just stores away fontarea and fontname for later use by BuildFontSpec.
 - Released version 1.3 in July, 1985

   August, 1985
 - Moved DVI page structures and translation routines into
   DVIReader so they can be used by other device drivers.
   Fonts/chars/rules are now stored as dynamic lists.
 - Modified ProcessPage so that user can type "NNN..." and be
   warned about ALL pages off the paper, not just the last.
 - Procedures now appear in a top-down fashion.
   September, 1985
 - Now check for display abort or mode change in DisplayChars
   instead of each DisplayOneChar routine.
   We only call BusyRead approximately every 8th visible rule
   or character to avoid QIO overheads.
 - FullChar1/2 now use glyphcache and GetPXLGlyph from
   PXLReader to drastically reduce PXL file accesses.
   OpenPXLFile requires an additional parameter to indicate if
   glyphcache has to be initialized (only TRUE in OpenFontFile).
 - Slight bug in ShowHelp pagination fixed.
 - Released version 1.4 in September, 1985

   October, 1985
 - Above PXL file cacheing is no longer used since
   PXLReader now maps an entire PXL file into virtual memory.
 - Released version 1.5 in October, 1985

   March, 1986
 - Changes to the Modula-2 compiler in accordance with Wirth's
   1983 amendments has required the following changes:
   1. Definition modules no longer have an EXPORT list.
   2. MIN/MAX changed to Min/Max since the former are new
      standard functions.
   3. commpos and commlen are now CARDINAL to get around new
      restrictions on compatibility of VAR parameters.
 - ResetVDU is now called before all Halts, but before any error message
   since it might erase the screen! (as VT220 version does).
   September, 1986
 - fontdir is a single string once more (see DCLInterface and DVITOVDU.CLD).
 - No longer translate fontarea (if it ends in ":") in BuildFontSpec.
 - MySpecialRoutine now shows position and first 32 bytes of each \special.
 - Every "(* DEBUG" now has a matching "GUBED *)"; easier to (de)activate code.
 - MyPixelTableRoutine makes sure mapadr = 0 if ht or wd = 0.  This should not
   occur in a correctly built PXL file, but some programs (such as GFread and
   GFtoPXL on K&S's 2.0.1 tape) might not do the right thing.
 - Released version 1.6 in September, 1986

   November, 1986
 - PXLReader has become FontReader and can handle other font formats.
   All font-dependent code has been moved into FontReader.
 - FullChar1/2 now use bitmap pointer to access words in character bitmap.
   If bitmap = NIL then we call GetBitmap to dynamically allocate sufficient
   memory and return its address.  This is only done once per character.
   The bitmap memory is never deallocated.
 - OpenFontFile now called DrawingFont (and does not open font file).
   OpenPXLFile now called OpenFontFile.
 - Introduced global fontseen flag to make sure "Drawing ..." message is only
   seen once per visible font in a Full display.
 - fontexists flag in fontinfo is used to avoid opening/closing font files.
   This flag is initialized to FALSE by DVIReader and becomes TRUE if
   BuildFontSpec (now imported from FontReader) can find the font file.
 - PleaseReport is now a global routine and used before serious Halt(2) calls.
 - Length of commstring increased from 80 to 132.
 - Fixed minor pagination bug in ShowStatistics.
 - Increased maxviswords from 30 to 100; safer for high resolution values
   or for Metafont users who create huge characters as a way of doing graphics.
 - Released version 1.7 in January, 1987

   November, 1987 (while at The Open University)
 - Removed MySpecialRoutine; any \special commands are now shown in
   ShowStatistics.  Such a scheme is much friendlier for DVI files with
   lots of \specials!
 - Now import Cap function and new tfmdir qualifier from DCLInterface.
 - Introduced FullCharPS which gets assigned to DisplayOneChar if the
   display mode = Full and the font is a PostScript one.
 - Now call BusyRead approximately every 4th visible rule
   or character (was every 8th).
 - Added DVItoVDU banner line showing version number in initial display.
 - Added WritePtSize routine to show requested pt size for PostScript fonts.
 - Added ZoomInOut commands.  ZI halves current window dimensions and
   ZO doubles them.
 - Released version 2.0 in December, 1987

   June--August, 1988 (while at Aston University)
 - Added /psprefix qualifier to allow sites to specify the TFM file name prefix
   used to indicate a PostScript font.
 - Added /hoffset and /voffset qualifiers to allow shifting of page.
 - Increased maxviswords from 100 to 200; any glyphs wider than 200*32 pixels
   will now be truncated.
 - Released version 3.0 in August, 1988

 - Released version 3.1 in October, 1989
*)


(* SYSDEP: Since Modula-2 avoids the problem of system dependence by simply
   not providing any input/output routines etc., most of the following
   importations are highly VAX/VMS dependent.
*)

FROM SYSTEM IMPORT
   ADDRESS;          (* byte address *)

FROM FileSystem IMPORT
   File, Done,
   Open, Close,
   Eof, EOL,
   ReadChar;


(*******************************************************************************
   SYSDEP: The above modules are defined and implemented in the Modula-2 system
   library (MOD$SOURCE).  The following modules can be found in [TEX.DVITOVDU].
   See the .DEF files for details on how the imported identifiers should be
   used; implementation details can be found in the corresponding .MOD files.
*******************************************************************************)


(* SYSDEP: DVItoVDU uses the ScreenIO routines to do all terminal i/o.
   Note that ScreenIO sets the terminal channel to /ttsync/nowrap;
   we use Halt to restore the channel and then exit with either a success (1)
   or error (2) status.
*)

FROM ScreenIO IMPORT
   (* PROCEDURE *)
   Read, ReadString, BusyRead,
   Write, WriteString, WriteInt, WriteCard, WriteLn, WriteBuffer,
   Halt;


(* The DCLInterface module carries out the task of reading the DCL command
   line and extracting the DVI file parameter and qualifiers.
*)

FROM DCLInterface IMPORT
   (* VAR *)
   resolution, mag, paperwd, paperht, hoffset, voffset,
   psprefix, tfmdir, fontdir, dummyfont, helpname, vdu, DVIname,
   (* PROCEDURE *)
   Cap;


(* DVItoVDU uses the routines and data structures defined in DVIReader to move
   about randomly in the DVI file and to interpret pages.
*)

FROM DVIReader IMPORT
   (* CONST *)
   ruletablesize, chartablesize, maxfontspec, maxTeXchar,
   (* TYPE *)
   ruleinfo, ruleinfoptr,
   specialstring, specialinfo, specialinfoptr,
   fontstring, fontinfo, fontinfoptr,
   charinfo, charinfoptr, pixeltable, pixeltableptr,
   TeXcounters, TeXpageinfo,
   DVIerrorcodes,
   (* VAR *)
   DVImag, totalpages, totalfonts,
   currDVIpage, currTeXpage,
   rulelist, totalrules, speciallist, fontlist, currfont,
   minhp, minvp, maxhp, maxvp, pageempty,
   DVIErrorRoutine, PixelTableRoutine,
   (* PROCEDURE *)
   OpenDVIFile, SetConversionFactor,
   MoveToNextPage, MoveToDVIPage, MoveToTeXPage,
   PixelRound, InterpretPage, SortFonts,
   CloseDVIFile;


(* DVItoVDU gets character metrics and bitmaps from font files.
   FontReader can handle a variety of different font formats.
   No more than one font file will be open at any given time.
*)

FROM FontReader IMPORT
   (* VAR *)
   FillPixelTable, GetBitmap,
   (* PROCEDURE *)
   InitFontReader, BuildFontSpec,
   OpenFontFile, CloseFontFile;


(* DVItoVDU can work efficiently on a variety of VDUs without having to
   know all the nitty gritty details required to drive them.
   Modula-2's procedure variables and separate compilation facilities provide
   a very nice mechanism for achieving the desired terminal independence.
   The following generic VDU parameters and routines are initialized in the
   VDUInterface implementation module.
*)

FROM VDUInterface IMPORT
   (* VAR *)
   DVIstatusl, windowstatusl, messagel, commandl, bottoml,
   windowh, windowv, windowwd, windowht,
   StartText, MoveToTextLine, ClearTextLine, ClearScreen,
   StartGraphics, LoadFont, ShowChar, ShowRectangle,
   ResetVDU;


(*******************************************************************************
   DECLARATIONS FOR PROCESSING USER COMMANDS

   Most commands consist of one or two characters and can be entered
   in upper or lowercase.  Multiple commands are processed in the order
   given but we only update the window, if necessary, at the end.
   If a bad command is encountered, any further commands are ignored.
   Some commands can have parameters; they are all dimensions in terms of the
   current units.  Spaces before and after commands and parameters are ignored.
*)

CONST
   (* Possible commands are:                                                  *)
   (* i                  a positive integer; display ith DVI page             *)
   TeXpage   = '[';   (* start of a TeX page specification: [i0. ... .i9]     *)
   Next      = 'N';   (* display next DVI page, depending on direction        *)
   Forwards  = '>';   (* process DVI pages in ascending order                 *)
   Backwards = '<';   (* process DVI pages in descending order                *)
   Window    = 'W';   (* move window's top left corner to given position      *)
   Up        = 'U';   (* move window up a given amount                        *)
   Down      = 'D';   (* move window down a given amount                      *)
   Left      = 'L';   (* move window left a given amount                      *)
   Right     = 'R';   (* move window right a given amount                     *)
   Hsize     = 'H';   (* set scaledwd: window's horizontal size               *)
   Vsize     = 'V';   (* set scaledht: window's vertical size                 *)
   ZoomInOut = 'Z';   (* halve/double window dimensions                       *)
   Terse     = 'T';   (* display quick and nasty chars at reference points    *)
   Box       = 'B';   (* display box outlines of glyphs                       *)
   Full      = 'F';   (* display all pixels in glyphs                         *)
   In        = 'I';   (* get/show dimensions in inches                        *)
   Cm        = 'C';   (* get/show dimensions in centimetres                   *)
   Mm        = 'M';   (* get/show dimensions in millimetres                   *)
   PcPtPx    = 'P';   (* get/show dimensions in picas/points/pixels           *)
   Help      = '?';   (* display help on available commands                   *)
   Show      = 'S';   (* display useful statistics                            *)
   Quit      = 'Q';   (* have a guess                                         *)
   NULL = 0C;
   CR   = 15C;
   commprompt = 'Command:';

VAR
   commstring
      : ARRAY [0..131] OF CHAR;     (* holds user responses                   *)
   commpos   : CARDINAL;            (* current position in commstring         *)
   commlen   : CARDINAL;            (* length of commstring                   *)
   command   : CHAR;                (* starting character of command          *)
   ascending : BOOLEAN;             (* initially TRUE; changed by the
                                       Forwards/Backwards commands to control
                                       how to get the next DVI page           *)
   maxpix    : INTEGER;             (* maximum absolute pixel value;
                                       depends on resolution                  *)
   (* These flags are used to handle multiple commands:                       *)
   screenjustcleared,               (* has screen just been cleared?          *)
   paintDVIStatus,                  (* does DVI status line need updating?    *)
   paintWindowStatus,               (* does window status line need updating? *)
   paintwindow,                     (* does window region need updating?      *)
   pageoffpaper,                    (* is page off paper?                     *)
   badcommand : BOOLEAN;            (* was there a bad command?               *)


(*******************************************************************************
   DECLARATIONS FOR DISPLAYING A PAGE

   The reference points of characters and rules on a page are stored as
   pairs of horizontal and vertical paper pixel coordinates.
   The paper coordinate scheme is described in detail in DVIReader.
   The screen coordinate scheme is described in detail in VDUInterface.
   To update the window region, DVItoVDU maps visible paper pixels
   to screen pixels using windowh and windowv to help with translation,
   and windowwd and windowht to help with scaling.
   What the user sees depends on the current displaymode, the current size
   of the window region (scaledwd by scaledht are in paper pixels and determine
   the horizontal and vertical scaling factors), and the current paper position
   of the window region's top left corner; i.e., (windowleft,windowtop).

   A NOTE ON THE SCALING METHOD USED BY DVItoVDU:
   We desire the following conditions when scaling paper pixels to
   screen pixels:
   1. Rules/glyphs having the same top/bottom/left/right paper coordinates also
      have the same screen coordinates (e.g., to ensure baselines line up).
      This condition is incompatible with a rule/glyph staying the same
      width and height as the window position changes!  Too bad.
   2. After being scaled, visible pixel positions must not exceed the
      window region's edges.  In our case, only the bottom and right edges are
      a problem because scaling starts at the top left corner of the window.
      For efficiency, we use two different scaling functions depending on
      whether the h/vscalefactors are < 1.0 or not.
   3. Scaled heights and widths must be > 0 even when h/vscalefactors
      approach 0.  If h/vscalefactors are > 1.0 then the width/height of
      paper pixels increase accordingly.
*)

CONST
   abortkey   = 15C;                (* user aborts display by hitting RETURN  *)

VAR
   displaymode    : (tersemode      (* show quick and nasty chars at ref pts  *)
                    ,boxmode        (* show box outlines of glyphs            *)
                    ,fullmode       (* show all pixels in glyphs              *)
                    );
   currentunits   : (inunits        (* get/show dimensions in inches          *)
                    ,cmunits        (* get/show dimensions in centimetres     *)
                    ,mmunits        (* get/show dimensions in millimetres     *)
                    ,pcunits        (* get/show dimensions in picas           *)
                    ,ptunits        (* get/show dimensions in points          *)
                    ,pxunits        (* get/show dimensions in pixels          *)
                    );
   papertop,
   paperleft,
   paperbottom,
   paperright     : INTEGER;        (* these define the edges of the paper    *)
   windowtop,
   windowleft,
   windowbottom,
   windowright    : INTEGER;        (* these define the current window edges  *)
   allpagevisible : BOOLEAN;        (* is all of page visible in window?      *)
   outsidepage    : BOOLEAN;        (* is entire window outside page?         *)
   scaledht       : INTEGER;        (* current window height in paper pixels  *)
   scaledwd       : INTEGER;        (* current window width in paper pixels   *)
   vscalefactor   : REAL;           (* windowht / scaledht                    *)
   hscalefactor   : REAL;           (* windowwd / scaledwd                    *)

   (* Expand/ShrinkHpos and Expand/ShrinkVpos are assigned to these
      procedure variables depending on the values of h/vscalefactor.
   *)
   ScaleVpos,
   ScaleHpos      : PROCEDURE (INTEGER) : INTEGER;

   (* TerseChar, BoxChar or FullChar1/2 are assigned to DisplayOneChar
      depending on the current displaymode (which the user can change by
      hitting the Terse/Box/Full commands while DisplayChars is executing).
      If a Full display and the current font is a PostScript font then
      FullCharPS is assigned to DisplayOneChar.
   *)
   DisplayOneChar : PROC;
   thisruleinfo   : ruleinfoptr;    (* current rule info in rulelist          *)
   unusedfont     : fontinfoptr;    (* first unused font in sorted fontlist   *)
   thisfontinfo   : fontinfoptr;    (* current font info in sorted fontlist   *)
   thischarinfo   : charinfoptr;    (* current char info in charlist          *)
   thischar       : CARDINAL;       (* current index into current chartable   *)
   fontseen       : BOOLEAN;        (* is "Drawing ..." message visible?      *)
   fontopen       : BOOLEAN;        (* is thisfontinfo^.fontspec open?        *)
   useraborted    : BOOLEAN;        (* did user abort page display?           *)
   charvisible    : BOOLEAN;        (* was character actually displayed?      *)


(******************************************************************************)

PROCEDURE TopLevel;

(* Note that the implementation blocks of all imported modules have already
   been executed.  The more important of these are listed below in order
   of execution, along with their effects:
   ScreenIO      - sets the terminal channel to /ttsync/nowrap
                   (Halt will be used to restore channel and exit)
   DCLInterface  - reads the DCL command line and initializes the DVI file
                   parameter and qualifier values
   VDUInterface  - initializes the generic VDU parameters and routines
                   according to the vdu value returned by DCLInterface
*)

BEGIN
Initialize;
DVIErrorRoutine := MyDVIErrorRoutine;       (* called by DVIReader upon error *)
OpenDVIFile(DVIname);                       (* and read DVImag etc. *)
IF mag = 0 THEN mag := DVImag END;          (* no /MAG value, so use DVImag *)
SetConversionFactor(resolution,mag);        (* for DVIReader *)
PixelTableRoutine := MyPixelTableRoutine;   (* called by InterpretPage *)
InitFontReader;                             (* assign font-dependent routines *)
StartText;
ClearScreen;
UpdateDVIStatusLine;
UpdateWindowStatusLine;
ClearMessageLine;
WriteString('This is DVItoVDU, version 3.1');
REPEAT
   NextCommandLine;                         (* parse and execute command(s) *)
UNTIL command = Quit;
Finish;
END TopLevel;

(******************************************************************************)

PROCEDURE Initialize;

BEGIN
(* TeX will not generate dimensions > than about 38 feet, so we
   choose an absolute limit on our dimensions to be 40 feet = 480in.
*)
IF MAX(INTEGER) DIV INTEGER(resolution) < 480 THEN
   maxpix := MAX(INTEGER);
ELSE
   maxpix := 480 * resolution;
END;

(* top left corner of paper is fixed at (-1",-1") *)
papertop    := -INTEGER(resolution);
paperleft   := -INTEGER(resolution);
paperbottom := papertop  + INTEGER(paperht) - 1;
paperright  := paperleft + INTEGER(paperwd) - 1;

(* User sees the following status values before requesting the first page.
   Note that DVIReader has already initialized currDVIpage and currTeXpage.
*)
ascending := TRUE;             (* process DVI pages in ascending order *)
displaymode := tersemode;
windowtop := 0;                (* window location *)
windowleft := 0;
scaledht := windowht;          (* window size is initially unscaled *)
scaledwd := windowwd;
minhp := 0; minvp := 0;        (* page location *)
maxhp := 0; maxvp := 0;
currentunits := inunits;       (* units are initially inches *)

(* initialize the scaling routines *)
vscalefactor := 1.0;
hscalefactor := 1.0;
ScaleVpos    := ShrinkVpos;    (* use when vscalefactor <= 1.0 *)
ScaleHpos    := ShrinkHpos;    (* use when hscalefactor <= 1.0 *)
END Initialize;

(******************************************************************************)

PROCEDURE MyDVIErrorRoutine (DVIerror : DVIerrorcodes);

(* DVIErrorRoutine for DVIReader which has just detected one of the errors
   described in DVIReader's definition module.
*)

BEGIN
CASE DVIerror OF
(* these errors are detected in OpenDVIFile; they are considered fatal *)
   DVIunopened      :
      ResetVDU;
      WriteString("Couldn't open ");
      WriteString(DVIname); Write('!'); WriteLn;
      Halt(2);      |
   DVIempty         :
      ResetVDU;
      WriteString(DVIname);
      WriteString(' is empty!'); WriteLn;
      Halt(2);      |
   DVIbadid         :
      ResetVDU;
      WriteString(DVIname);
      WriteString(' is not a valid DVI file!'); WriteLn;
      Halt(2);      |
   DVIstackoverflow :
      ResetVDU;
      WriteString('Stack capacity exceeded!'); WriteLn;
      PleaseReport;
      Halt(2);      |
(* this error is detected in InterpretPage; we warn user but continue *)
   DVIbadchar       :
      WITH currfont^ DO
         ClearMessageLine;
         WriteString('Ignoring unknown character from ');
         WriteString(fontspec); Write('!');
         WaitForReturn;
      END;          |
(* this error should never happen *)
   DVIcatastrophe   :
      ResetVDU;
      WriteLn;
      WriteString('Something awful has happened!'); WriteLn;
      PleaseReport;
      Halt(2);
ELSE
      ResetVDU;
      WriteLn;
      WriteString('Bug in MyDVIErrorRoutine!'); WriteLn;
      PleaseReport;
      Halt(2);
END;
END MyDVIErrorRoutine;

(******************************************************************************)

PROCEDURE PleaseReport;

BEGIN
WriteString('Please tell your local TeXnician.'); WriteLn;
END PleaseReport;

(******************************************************************************)

PROCEDURE ClearMessageLine;

(* Clear message line and move cursor to start of line.
   We don't show any message here; that will usually be done
   immediately after calling this routine.
*)

BEGIN
ClearTextLine(messagel);
MoveToTextLine(messagel);
END ClearMessageLine;

(******************************************************************************)

PROCEDURE WaitForReturn;

(* DVItoVDU has just displayed an important message.
   To ensure message is seen we wait for user to hit RETURN.
*)

VAR ch : CHAR;

BEGIN
WriteString('   RETURN:');
WriteBuffer;
REPEAT Read(ch) UNTIL ch = CR;
END WaitForReturn;

(******************************************************************************)

PROCEDURE MyPixelTableRoutine;

(* PixelTableRoutine for DVIReader which has just allocated a new pixeltable
   for currfont^.  DVIReader calls this routine from InterpretPage only
   ONCE per font (the first time the font is used).
   If this is the first time we've seen the font then we build fontspec first.
   (Note that ShowStatistics may call BuildFontSpec first.)
   If we can't open the font file we return dummyfont values, but using the
   current font's scaledsize.
   We get the pixeltable info using FillPixelTable from FontReader.
*)

BEGIN
WITH currfont^ DO
   IF fontspeclen = 0 THEN       (* need to build fontspec *)
      BuildFontSpec(currfont);
   END;
   ClearMessageLine;
   IF OpenFontFile(fontspec) THEN
      WriteString('Loading font data from ');
      WriteString(fontspec);
      IF psfont THEN WritePtSize(scaledsize) END;
      WriteLn;
   ELSIF OpenFontFile(dummyfont) THEN
      (* we will fill pixeltable with dummyfont values *)
      WriteString("Couldn't open "); WriteString(fontspec);
      WriteString("!   Loading dummy font.");
      WaitForReturn;
      ClearMessageLine;
      WriteBuffer;               (* clears message line immediately *)
   ELSE
      ResetVDU;
      WriteString("Couldn't open dummy font ");
      WriteString(dummyfont); Write('!'); WriteLn;
      PleaseReport;
      Halt(2);
   END;
   FillPixelTable;               (* depends on format of font file *)
   CloseFontFile;
END;
END MyPixelTableRoutine;

(******************************************************************************)

PROCEDURE WritePtSize (scaledsize : INTEGER);

(* Show given font size (in DVI units) in terms of (possibly magnified) pts. *)

VAR realdim : REAL;   fracpart : CARDINAL;

BEGIN
WriteString(' at ');
realdim := (FLOAT(scaledsize) / FLOAT(10000H)) * (FLOAT(mag) / 1000.0);
(* show realdim to an accuracy of 1 decimal place *)
IF ABS(realdim) < 0.05 THEN
   WriteString('0');
ELSE
   IF realdim < 0.0 THEN
      Write('-');
      realdim := ABS(realdim);
   END;
   realdim := realdim + 0.05;     (* round up to 1 decimal place *)
   WriteCard(TRUNC(realdim));     (* whole part *)
   fracpart := TRUNC((realdim - FLOAT(TRUNC(realdim))) * 10.0);   (* 0..9 *)
   IF fracpart > 0 THEN
      Write('.');
      WriteCard(fracpart);
   END;
END;
WriteString('pt');
END WritePtSize;

(******************************************************************************)

PROCEDURE UpdateDVIStatusLine;

(* Show totalpages, currDVIpage, currTeXpage, direction and displaymode. *)

VAR i, lastnonzero : CARDINAL;

BEGIN
ClearTextLine(DVIstatusl);
MoveToTextLine(DVIstatusl);
WriteString('Total pages='); WriteCard(totalpages);
WriteString('   DVI page='); WriteCard(currDVIpage);
WriteString('   TeX page='); Write('[');
lastnonzero := 9;
WHILE (lastnonzero > 0) AND (currTeXpage[lastnonzero] = 0) DO
   DEC(lastnonzero);        (* find last counter with non-zero value *)
END;
(* always show \count0 but don't show trailing 0 counters *)
FOR i := 0 TO lastnonzero DO
   WriteInt(currTeXpage[i]);
   IF i <> lastnonzero THEN
      Write('.');
   END;
END;
Write(']');
WriteString('   Next=');
IF ascending THEN
   Write('>');
ELSE
   Write('<');
END;
WriteString('   ');
CASE displaymode OF
   tersemode : WriteString('Terse') |
   boxmode   : WriteString('Box')   |
   fullmode  : WriteString('Full')
END;
WriteLn;
END UpdateDVIStatusLine;

(******************************************************************************)

PROCEDURE UpdateWindowStatusLine;

(* Show current window location and size, page location and size, and units. *)

BEGIN
ClearTextLine(windowstatusl);
MoveToTextLine(windowstatusl);
WriteString('Window at (');    WriteDimension(windowleft);
Write(',');                    WriteDimension(windowtop);
WriteString(') ');             WriteDimension(scaledwd);
WriteString(' by ');           WriteDimension(scaledht);
WriteString('   Page at (');   WriteDimension(minhp);
Write(',');                    WriteDimension(minvp);
WriteString(') ');             WriteDimension(maxhp-minhp+1);
WriteString(' by ');           WriteDimension(maxvp-minvp+1);
WriteString('   ');
CASE currentunits OF
   inunits : WriteString('IN') |
   cmunits : WriteString('CM') |
   mmunits : WriteString('MM') |
   pcunits : WriteString('PC') |
   ptunits : WriteString('PT') |
   pxunits : WriteString('PX')
END;
WriteLn;
END UpdateWindowStatusLine;

(******************************************************************************)

PROCEDURE WriteDimension (pixels : INTEGER);

(* Show the given pixel dimension in terms of currentunits. *)

VAR realdim : REAL;   fracpart : CARDINAL;

BEGIN
CASE currentunits OF
   inunits : realdim := FLOAT(pixels) / FLOAT(resolution) |
   cmunits : realdim := FLOAT(pixels) / FLOAT(resolution) * 2.54 |
   mmunits : realdim := FLOAT(pixels) / FLOAT(resolution) * 25.4 |
   pcunits : realdim := FLOAT(pixels) / FLOAT(resolution) * 72.27 / 12.0 |
   ptunits : realdim := FLOAT(pixels) / FLOAT(resolution) * 72.27 |
   pxunits : WriteInt(pixels); RETURN
END;
(* show realdim to an accuracy of 1 decimal place *)
IF ABS(realdim) < 0.05 THEN
   WriteString('0.0');
ELSE
   IF realdim < 0.0 THEN
      Write('-');
      realdim := ABS(realdim);
   END;
   realdim := realdim + 0.05;     (* round up to 1 decimal place *)
   WriteCard(TRUNC(realdim));     (* whole part *)
   Write('.');
   fracpart := TRUNC((realdim - FLOAT(TRUNC(realdim))) * 10.0);   (* 0..9 *)
   WriteCard(fracpart);
END;
END WriteDimension;

(******************************************************************************)

PROCEDURE NextCommandLine;

(* Prompt user for next command line, parse response and call the
   appropriate command handler for each command in the line.
*)

VAR n : INTEGER;             (* returned by GetInteger call *)

BEGIN
ClearTextLine(commandl);
MoveToTextLine(commandl);
WriteString(commprompt);
WriteBuffer;
ReadString(commstring);      (* read new command line *)
ClearMessageLine;            (* erase message line at this stage *)
commlen := LEN(commstring);
commpos := 0;
WHILE (commlen > 0) AND (commstring[commlen-1] = ' ') DO
   DEC(commlen);          (* ignore any trailing spaces *)
END;
(* initialize flags for multiple command processing *)
badcommand        := FALSE;
paintWindowStatus := FALSE;
paintDVIStatus    := FALSE;
paintwindow       := FALSE;
screenjustcleared := FALSE;
pageoffpaper      := FALSE;
WHILE (commpos < commlen) AND (NOT badcommand) DO
   (* next command is defined by the next non-space character in commstring *)
   WHILE commstring[commpos] = ' ' DO
      INC(commpos);          (* ignore any spaces *)
   END;
   command := Cap(commstring[commpos]);
   CASE command OF
      Window    : INC(commpos);
                  WindowMove;
                  IF (currDVIpage <> 0) AND (NOT badcommand)
                                      THEN paintWindowStatus := TRUE END;
                |
      Up,
      Down      : INC(commpos);
                  WindowUpDown;
                  IF currDVIpage <> 0 THEN paintWindowStatus := TRUE END;
                |
      Left,
      Right     : INC(commpos);
                  WindowLeftRight;
                  IF currDVIpage <> 0 THEN paintWindowStatus := TRUE END;
                |
      Hsize     : INC(commpos);
                  SetWindowWidth;
                  IF currDVIpage <> 0 THEN
                     NewLocation(windowleft,windowtop);
                  END;
                  paintWindowStatus := TRUE;
                |
      Vsize     : INC(commpos);
                  SetWindowHeight;
                  IF currDVIpage <> 0 THEN
                     NewLocation(windowleft,windowtop);
                  END;
                  paintWindowStatus := TRUE;
                |
      ZoomInOut : INC(commpos);
                  ZoomWindow;
                  IF (currDVIpage <> 0) AND (NOT badcommand) THEN
                     NewLocation(windowleft,windowtop);
                  END;
                  IF NOT badcommand THEN paintWindowStatus := TRUE END;
                |
      Next      : INC(commpos);
                  IF NextPageFound() THEN
                     ProcessPage;
                  END;
                |
      '0'..'9'  : IF GetInteger(commstring,commlen,commpos,n)
                     (* must be true, and commpos now after last digit *)
                     AND DVIPageFound(n) THEN
                     ProcessPage;
                  END;
                |
      TeXpage   : IF TeXPageFound() THEN
                     (* commpos incremented in ParseTeXpage *)
                     ProcessPage;
                  END;
                |
      Forwards  : INC(commpos);
                  ascending := TRUE;
                  paintDVIStatus := TRUE;
                |
      Backwards : INC(commpos);
                  ascending := FALSE;
                  paintDVIStatus := TRUE;
                |
      Terse     : INC(commpos);
                  displaymode := tersemode;
                  paintDVIStatus := TRUE;
                  IF currDVIpage <> 0 THEN paintwindow := TRUE END;
                |
      Box       : INC(commpos);
                  displaymode := boxmode;
                  paintDVIStatus := TRUE;
                  IF currDVIpage <> 0 THEN paintwindow := TRUE END;
                |
      Full      : INC(commpos);
                  displaymode := fullmode;
                  paintDVIStatus := TRUE;
                  IF currDVIpage <> 0 THEN paintwindow := TRUE END;
                |
      In, Cm, Mm,
      PcPtPx    : INC(commpos);
                  ChangeUnits;
                  IF NOT badcommand THEN paintWindowStatus := TRUE END;
                |
      Help      : INC(commpos);
                  ShowHelp;
                |
      Show      : INC(commpos);
                  ShowStatistics;
                  ClearScreen;
                  screenjustcleared := TRUE;
                  paintDVIStatus := TRUE;
                  paintWindowStatus := TRUE;
                  IF currDVIpage <> 0 THEN paintwindow := TRUE END;
                |
      Quit      : RETURN;
   ELSE
      INC(commpos);
      ClearMessageLine;
      WriteString('Unknown command!   Type ');
      Write(Help); WriteString(' for help.');
      BadCommandMessage;
   END;
END;
IF paintwindow THEN
   DisplayPage;     (* only update window after processing all commands *)
ELSE
   IF paintDVIStatus THEN UpdateDVIStatusLine END;
   IF paintWindowStatus THEN UpdateWindowStatusLine END;
END;
END NextCommandLine;

(******************************************************************************)

PROCEDURE WindowMove;

(* Syntax of Window command is  W hpos,vpos  where hpos and vpos are
   dimensions with leading and/or trailing spaces.  If hpos,vpos absent then
   we move to minhp,minvp (top left corner of page rectangle).
*)

VAR hpos, vpos : INTEGER;   (* move window to this new position *)

BEGIN
(* commpos is positioned after W *)
IF GetDimension(commstring,commlen,commpos,hpos) THEN
   WHILE (commpos < commlen) AND (commstring[commpos] = ' ') DO
      INC(commpos);   (* skip any spaces before comma *)
   END;
   IF (commpos = commlen) OR              (* , vpos is missing *)
      (commstring[commpos] <> ',') THEN   (* , is missing *)
      ClearMessageLine;
      WriteString('Comma expected!');
      IF commpos < commlen THEN INC(commpos) END;
      BadCommandMessage;
   ELSE
      INC(commpos);   (* skip over comma *)
      IF GetDimension(commstring,commlen,commpos,vpos) THEN
         NewLocation(hpos,vpos);
      ELSE
         ClearMessageLine;
         WriteString('Vertical coordinate expected!');
         IF commpos < commlen THEN INC(commpos) END;
         BadCommandMessage;
      END;
   END;
ELSE
   NewLocation(minhp,minvp);   (* hpos,vpos absent *)
END;
END WindowMove;

(******************************************************************************)

PROCEDURE GetDimension (str     : ARRAY OF CHAR;  (* in *)
                        strlen  : CARDINAL;       (* in *)
                        VAR pos : CARDINAL;       (* in/out *)
                        VAR n   : INTEGER         (* out *)
                       ) : BOOLEAN;

(* Extract a dimension from given str starting at given pos.
   n returns the corresponding number of pixels in the dimension
   (which is an integer or real value in terms of currentunits);
   pos is also used to return the position after the dimension.
   If no dimension is found then set n to 0 and return FALSE (pos will only
   change if leading spaces were skipped).
   If ABS(n) > maxpix then set n to sign * maxpix.
   Valid syntax of a dimension is  integer[.{digit}]  or  .{digit}  where
   an integer is defined by GetInteger.
   Real dimensions are truncated to 4 decimal places.
   Note that a sign or decimal point by itself is valid and sets n to 0.
*)

VAR sign, intdim : INTEGER;
    fracpart, divisor : CARDINAL;
    absrealdim : REAL;
    intpresent, dimtoobig : BOOLEAN;

BEGIN
(* GetInteger does not remember a sign by itself, so we need to check
   for -ve dimensions like -.5 first.
*)
WHILE (pos < strlen) AND (str[pos] = ' ') DO   (* skip any spaces *)
   INC(pos);
END;
sign := 1;
IF (pos < strlen) AND (str[pos] = '-') THEN
   sign := -1;
END;
intpresent := GetInteger(str,strlen,pos,intdim);
IF (NOT intpresent) AND ((pos = strlen) OR (str[pos] <> '.')) THEN
   n := 0;
   RETURN FALSE;
END;
(* dimension is valid; if no integer part then intdim will be 0; sign = +|-1 *)
IF (pos = strlen) OR (str[pos] <> '.') THEN
   (* no fractional part *)
   absrealdim := FLOAT(ABS(intdim));
ELSE
   (* extract fractional part *)
   INC(pos);       (* skip over decimal point *)
   divisor := 1;
   fracpart := 0;
   WHILE (pos < strlen) AND (str[pos] >= '0') AND (str[pos] <= '9') DO
      (* only consider up to 4 decimal places *)
      IF divisor < 10000 THEN
         divisor := divisor * 10;
         fracpart := fracpart * 10 + (ORD(str[pos]) - ORD('0'));
      END;
      INC(pos);
   END;
   absrealdim := FLOAT(ABS(intdim)) + (FLOAT(fracpart) / FLOAT (divisor));
END;
(* calculate n based on absrealdim, sign and currentunits *)
dimtoobig := FALSE;
CASE currentunits OF
   inunits :
      IF absrealdim > FLOAT(maxpix) / FLOAT(resolution) THEN
         dimtoobig := TRUE;
      ELSE
         n := sign * TRUNC(absrealdim * FLOAT(resolution) + 0.5);
      END; |
   cmunits :
      IF absrealdim > (FLOAT(maxpix) / FLOAT(resolution)) * 2.54 THEN
         dimtoobig := TRUE;
      ELSE
         n := sign * TRUNC((absrealdim / 2.54) * FLOAT(resolution) + 0.5);
      END; |
   mmunits :
      IF absrealdim > (FLOAT(maxpix) / FLOAT(resolution)) * 25.4 THEN
         dimtoobig := TRUE;
      ELSE
         n := sign * TRUNC((absrealdim / 25.4) * FLOAT(resolution) + 0.5);
      END; |
   pcunits :
      IF absrealdim > (FLOAT(maxpix) / FLOAT(resolution)) * (72.27 / 12.0) THEN
         dimtoobig := TRUE;
      ELSE
         n := sign * TRUNC((absrealdim / 72.27) * 12.0 * FLOAT(resolution) +
                                                                           0.5);
      END; |
   ptunits :
      IF absrealdim > (FLOAT(maxpix) / FLOAT(resolution)) * 72.27 THEN
         dimtoobig := TRUE;
      ELSE
         n := sign * TRUNC((absrealdim / 72.27) * FLOAT(resolution) + 0.5);
      END; |
   pxunits :
      IF absrealdim > FLOAT(maxpix) THEN
         dimtoobig := TRUE;
      ELSE
         n := sign * TRUNC(absrealdim + 0.5);
      END;
END;
IF dimtoobig THEN n := sign * maxpix END;
RETURN TRUE;
END GetDimension;

(******************************************************************************)

PROCEDURE GetInteger (str      : ARRAY OF CHAR;  (* in *)
                      strlen   : CARDINAL;       (* in *)
                      VAR pos  : CARDINAL;       (* in/out *)
                      VAR n    : INTEGER         (* out *)
                     ) : BOOLEAN;

(* Extract an integer from given str starting at given pos.
   pos is also used to return the position after the integer.
   If no integer is found then set n to 0 and return FALSE (pos will only
   change if leading spaces were skipped).
   If ABS(n) > limit then set n to sign * limit.
   Valid syntax is  +{digit}  or  -{digit}  or  digit{digit}.
   Note that a + or - by itself is valid and sets n to 0.
*)

CONST limit = 2147483647;          (* SYSDEP: TeX's limit = 2^31 - 1.
                                      Should also be >= maxpix.
                                      Note that this also defines the range of
                                      page numbers the user can ask for! *)
      threshold = limit DIV 10;    (* nearing overflow *)

VAR   absval, last : CARDINAL;
      sign : INTEGER;
      inttoobig : BOOLEAN;

BEGIN
WHILE (pos < strlen) AND (str[pos] = ' ') DO   (* skip any spaces *)
   INC(pos);
END;
absval := 0; sign := 1; last := pos;
inttoobig := FALSE;
IF pos < strlen THEN
   IF str[pos] = '-' THEN
      sign := -1; INC(last);
   ELSIF str[pos] = '+' THEN
      INC(last);
   END;
   WHILE (last < strlen) AND
         (str[last] >= '0') AND (str[last] <= '9') DO
      IF (absval > threshold) OR ((absval = threshold) AND (str[last] > '7'))
         THEN
         inttoobig := TRUE;
      ELSE
         absval := absval * 10 + (ORD(str[last]) - ORD('0'));
      END;
      INC(last);
   END;
END;
IF pos = last THEN
   n := 0;
   RETURN FALSE;
ELSE
   pos := last;
   IF inttoobig THEN absval := limit END;
   n := sign * INTEGER(absval);
   RETURN TRUE;
END;
END GetInteger;

(******************************************************************************)

PROCEDURE BadCommandMessage;

(* A bad command has just been detected and some sort of message displayed.
   Note that commpos is pointing to just after the problem character.
   If there are further commands then we show user what will be ignored.
*)

VAR i : CARDINAL;

BEGIN
badcommand := TRUE;
ClearTextLine(commandl);
MoveToTextLine(commandl);
WriteString(commprompt);
FOR i := 0 TO commpos-1 DO Write(commstring[i]) END;
Write('!');                   (* put ! after the problem character *)
IF commpos < commlen THEN
   WriteString('   Ignoring:');
   FOR i := commpos TO commlen-1 DO Write(commstring[i]) END;
END;
WaitForReturn;
ClearMessageLine;
ClearTextLine(commandl);
END BadCommandMessage;

(******************************************************************************)

PROCEDURE NewLocation (newhp, newvp : INTEGER);

(* Change window location to given position and update window edges.
   If pageempty is TRUE then window moves to (paperleft,papertop).
   If the entire window moves outside the page rectangle then outsidepage
   becomes TRUE and we restrict movement to just beyond the edge(s) so that
   user can easily move window (via Up,Down,Left,Right) to positions
   in which one or more window and page edges coincide.
   Note that allpagevisible is also updated.
*)

BEGIN
IF currDVIpage = 0 THEN   (* message only seen after W,U,D,L,R commands *)
   ClearMessageLine;
   WriteString("You haven't selected a page yet!");
   BadCommandMessage;
   RETURN;
END;
IF pageempty THEN
   newvp := papertop;
   newhp := paperleft;
ELSE
   (* check if new position puts window entirely outside edges;
      if so then minimize the movement needed to keep this true *)
   outsidepage := FALSE;
   IF newvp > maxvp THEN
      outsidepage := TRUE;
      newvp := maxvp + 1;
   ELSIF newvp < (minvp - scaledht + 1) THEN
      outsidepage := TRUE;
      newvp := minvp - scaledht;
   END;
   IF newhp > maxhp THEN
      outsidepage := TRUE;
      newhp := maxhp + 1;
   ELSIF newhp < (minhp - scaledwd + 1) THEN
      outsidepage := TRUE;
      newhp := minhp - scaledwd;
   END;
END;
windowtop := newvp;
windowleft := newhp;
windowbottom := windowtop + scaledht - 1;
windowright := windowleft + scaledwd - 1;
(* allpagevisible will only be sensible if not pageempty *)
allpagevisible := (minvp >= windowtop) AND (maxvp <= windowbottom) AND
                  (minhp >= windowleft) AND (maxhp <= windowright);
(* even if pageempty or window hasn't moved we must still call DisplayPage *)
paintwindow := TRUE;
END NewLocation;

(******************************************************************************)

PROCEDURE WindowUpDown;

VAR amount : INTEGER;   (* move window up/down this many pixels *)

BEGIN
(* commpos is positioned after U or D *)
IF GetDimension(commstring,commlen,commpos,amount) THEN
   (* do nothing *)
ELSE
   amount := scaledht;   (* if amount absent, set to window height *)
END;
IF command = Up THEN
   amount := -amount;
END;
NewLocation(windowleft,windowtop+amount);
END WindowUpDown;

(******************************************************************************)

PROCEDURE WindowLeftRight;

VAR amount : INTEGER;   (* move window left/right this many pixels *)

BEGIN
(* commpos is positioned after L or R *)
IF GetDimension(commstring,commlen,commpos,amount) THEN
   (* do nothing *)
ELSE
   amount := scaledwd;   (* if amount absent, set to window width *)
END;
IF command = Left THEN
   amount := -amount;
END;
NewLocation(windowleft+amount,windowtop);
END WindowLeftRight;

(******************************************************************************)

PROCEDURE SetWindowWidth;

(* Set horizontal size of window region to given dimension; if <= 0 then set
   horizontal size to 1 pixel.
   If no parameter then use the unscaled width represented by windowwd.
*)

VAR wd : INTEGER;

BEGIN
(* commpos is positioned after H *)
IF GetDimension(commstring,commlen,commpos,wd) THEN
   (* note that maximum value of wd is restricted to maxpix *)
   IF wd <= 0 THEN wd := 1 END;
   NewWindowWidth(wd);
ELSE
   NewWindowWidth(windowwd);   (* parameter absent *)
END;
END SetWindowWidth;

(******************************************************************************)

PROCEDURE NewWindowWidth (wd : INTEGER);

(* Set window width to given value (> 0 and <= max dimension). *)

BEGIN
scaledwd := wd;
hscalefactor := FLOAT(windowwd) / FLOAT(scaledwd);
(* following method avoids testing hscalefactor each time in ScaleHpos *)
IF hscalefactor > 1.0 THEN
   ScaleHpos := ExpandHpos;
ELSE
   ScaleHpos := ShrinkHpos;
END;
END NewWindowWidth;

(******************************************************************************)

PROCEDURE ExpandHpos (h : INTEGER) : INTEGER;

(* Return a scaled value for the given horizontal window coordinate. *)

BEGIN
RETURN TRUNC ( FLOAT(h) * hscalefactor + 0.5 );     (* hscalefactor > 1.0 *)
END ExpandHpos;

(******************************************************************************)

PROCEDURE ShrinkHpos (h : INTEGER) : INTEGER;

(* Return a scaled value for the given horizontal window coordinate. *)

BEGIN
RETURN TRUNC ( (FLOAT(h) + 0.5) * hscalefactor );   (* hscalefactor <= 1.0 *)
END ShrinkHpos;

(******************************************************************************)

PROCEDURE SetWindowHeight;

(* Set vertical size of window region to given dimension; if <= 0 then set
   vertical size to 1 pixel.
   If no parameter then use the unscaled height represented by windowht.
*)

VAR ht : INTEGER;

BEGIN
(* commpos is positioned after V *)
IF GetDimension(commstring,commlen,commpos,ht) THEN
   (* note that maximum value of ht is restricted to maxpix *)
   IF ht <= 0 THEN ht := 1 END;
   NewWindowHeight(ht);
ELSE
   NewWindowHeight(windowht);   (* parameter absent *)
END;
END SetWindowHeight;

(******************************************************************************)

PROCEDURE NewWindowHeight (ht : INTEGER);

(* Set window height to given value (> 0 and <= max dimension). *)

BEGIN
scaledht := ht;
vscalefactor := FLOAT(windowht) / FLOAT(scaledht);
(* following method avoids testing vscalefactor each time in ScaleVpos *)
IF vscalefactor > 1.0 THEN
   ScaleVpos := ExpandVpos;
ELSE
   ScaleVpos := ShrinkVpos;
END;
END NewWindowHeight;

(******************************************************************************)

PROCEDURE ExpandVpos (v : INTEGER) : INTEGER;

(* Return a scaled value for the given vertical window coordinate. *)

BEGIN
RETURN TRUNC ( FLOAT(v) * vscalefactor + 0.5 );     (* vscalefactor > 1.0 *)
END ExpandVpos;

(******************************************************************************)

PROCEDURE ShrinkVpos (v : INTEGER) : INTEGER;

(* Return a scaled value for the given vertical window coordinate. *)

BEGIN
RETURN TRUNC ( (FLOAT(v) + 0.5) * vscalefactor );   (* vscalefactor <= 1.0 *)
END ShrinkVpos;

(******************************************************************************)

PROCEDURE ZoomWindow;

(* Parse the rest of a ZoomInOut command and do it.
   commpos is pointing to next position in commandstr (and should be I or O).
*)

VAR nextch : CHAR;

BEGIN
IF commpos < commlen THEN
   nextch := Cap(commstring[commpos]);
   INC(commpos);
ELSE
   nextch := ' ';
END;
IF nextch = 'I' THEN
   (* scaledwd and scaledht are > 0 *)
   NewWindowWidth((scaledwd + 1) DIV 2);
   NewWindowHeight((scaledht + 1) DIV 2);
ELSIF nextch = 'O' THEN
   (* avoid overflow *)
   IF maxpix DIV 2 > scaledwd THEN
      NewWindowWidth(scaledwd * 2);
   ELSE
      NewWindowWidth(maxpix);
   END;
   IF maxpix DIV 2 > scaledht THEN
      NewWindowHeight(scaledht * 2);
   ELSE
      NewWindowHeight(maxpix);
   END;
ELSE
   ClearMessageLine;
   WriteString('ZI or ZO expected!');
   BadCommandMessage;
END;
END ZoomWindow;

(******************************************************************************)

PROCEDURE NextPageFound () : BOOLEAN;

(* User has selected next page in DVI file; what they get will depend on
   the current DVI page and whether we are ascending or not.
   Return TRUE iff we can move to next page.
*)

BEGIN
IF (currDVIpage = 1) AND (NOT ascending) THEN
   ClearMessageLine;
   WriteString('You are looking at first DVI page!');
   BadCommandMessage;
   RETURN FALSE;
ELSIF (currDVIpage = totalpages) AND ascending THEN
   ClearMessageLine;
   WriteString('You are looking at last DVI page!');
   BadCommandMessage;
   RETURN FALSE;
ELSE
   MoveToNextPage(ascending);   (* position to next DVI page *)
   RETURN TRUE;
END;
END NextPageFound;

(******************************************************************************)

PROCEDURE DVIPageFound (n : CARDINAL) : BOOLEAN;

(* User has selected a particular DVI page number.
   Move to page n and return TRUE iff n is in 1..totalpages.
*)

BEGIN
IF (n < 1) OR (n > totalpages) THEN
   ClearMessageLine;
   IF totalpages > 1 THEN
      WriteString('You can only request DVI pages 1 to ');
      WriteCard(totalpages);   Write('!');
   ELSE
      WriteString('You can only request DVI page 1!');
   END;
   BadCommandMessage;
   RETURN FALSE;
ELSE
   MoveToDVIPage(n);   (* position to given DVI page *)
   RETURN TRUE;
END;
END DVIPageFound;

(******************************************************************************)

PROCEDURE TeXPageFound () : BOOLEAN;

(* Return TRUE iff TeX page specification is valid and exists.
   If so then position to lowest matching page.
*)

VAR newTeXpage : TeXpageinfo;

BEGIN
IF ParseTeXpage(newTeXpage) THEN
   IF MoveToTeXPage(newTeXpage) THEN
      RETURN TRUE;           (* we found lowest matching page *)
   ELSE
      ClearMessageLine;
      WriteString('No TeX page matches your request!');
      BadCommandMessage;
      RETURN FALSE;
   END;
ELSE
   RETURN FALSE;             (* invalid TeX page specification *)
END;
END TeXPageFound;

(******************************************************************************)

PROCEDURE ParseTeXpage (VAR newTeXpage : TeXpageinfo)   (* out *)
                       : BOOLEAN;

(* Return TRUE iff TeX page specification in commstring is valid.  If so then
   newTeXpage will contain the appropriate information for MoveToTeXPage.
   The syntax of a TeX page specification is [n{.n}] where n is any integer as
   defined by GetInteger.  Up to 10 integers may be given and are separated by
   periods, even if absent.  Trailing periods may be omitted.  Spaces before
   and after integers and periods are skipped.  The 10 positions correspond to
   the \count0, \count1, ... ,\count9 values that TeX stores with every page.
   commpos is initially pointing at [.
*)

BEGIN
WITH newTeXpage DO
   lastvalue := 0;
   LOOP
      INC(commpos);
      present[lastvalue] := GetInteger(commstring, commlen, commpos,
                                       value[lastvalue]);
      (* commpos now at commlen, space, period, non-digit or ']' *)
      WHILE (commpos < commlen) AND (commstring[commpos] = ' ') DO
         INC(commpos);   (* skip any spaces *)
      END;
      IF commpos = commlen THEN           (* check this first! *)
         ClearMessageLine;
         WriteString('] expected!');
         BadCommandMessage;               (* commpos at commlen *)
         RETURN FALSE;
      END;
      IF commstring[commpos] = ']' THEN   (* end of TeX page specification *)
         INC(commpos);                    (* possibly further commands *)
         EXIT;
      END;
      IF lastvalue < 9 THEN
         INC(lastvalue);
      ELSE
         ClearMessageLine;
         WriteString("] expected after 10 integers!");
         INC(commpos);
         BadCommandMessage;
         RETURN FALSE;
      END;
      IF commstring[commpos] <> '.' THEN
         ClearMessageLine;
         WriteString('Period, integer or ] expected!');
         INC(commpos);
         BadCommandMessage;
         RETURN FALSE;
      END;
   END;
   WHILE (lastvalue > 0) AND (NOT present[lastvalue]) DO
      DEC(lastvalue);
   END;
END;
RETURN TRUE;
END ParseTeXpage;

(******************************************************************************)

PROCEDURE ProcessPage;

(* We are ready to interpret the current DVI page and fill in the various data
   structures imported from DVIReader.
   This routine will also:
   set the window size and location to useful values (depending on the relative
   sizes of the paper and unscaled window region, as well as the page location),
   update pageoffpaper (after checking to see if it was TRUE for the previous
   page processed as part of a multiple command string),
   set screenjustcleared, paintwindow and paintWindowStatus to TRUE,
   set paintDVIStatus to FALSE.
*)

VAR halfht, halfwd : INTEGER;

BEGIN
(* We check pageoffpaper here so user can type "NNNNNNNNNNNNN..." and note ALL
   the pages that are off the paper, not just the last one processed.
*)
IF pageoffpaper THEN
   ClearMessageLine;
   WriteString('Page off paper!');   (* the previous page *)
   WaitForReturn;
END;
ClearScreen;
screenjustcleared := TRUE;
UpdateDVIStatusLine;       (* a MoveTo... routine has updated currDVI/TeXpage *)
paintDVIStatus := FALSE;
InterpretPage;             (* fill in DVIReader's page data structures *)
SortFonts(unusedfont);     (* sort fonts in order of least chars and return
                              pointer to first unused font *)
ClearMessageLine;          (* clear any message *)
IF pageempty THEN
   minhp := 0; maxhp := 0; minvp := 0; maxvp := 0;   (* for window status *)
END;

(* Try viewing as much of paper as possible and without too much distortion: *)
IF ((paperwd < paperht) AND (windowwd >= windowht)) OR
   ((paperwd = paperht) AND (windowwd >  windowht)) THEN
   halfht := paperht DIV 2;
   IF ODD(paperht) THEN INC(halfht) END;     (* ensure bottom outline visible *)
   NewWindowHeight(halfht);                  (* try top half of paper *)
   NewWindowWidth(paperwd);
   NewLocation(paperleft,papertop);          (* top left corner of paper *)
   IF (NOT pageempty) AND outsidepage THEN
      NewLocation(paperleft,papertop+halfht);   (* try moving down *)
   END;
ELSIF ((paperwd > paperht) AND (windowwd <= windowht)) OR
      ((paperwd = paperht) AND (windowwd <  windowht)) THEN
   halfwd := paperwd DIV 2;
   IF ODD(paperwd) THEN INC(halfwd) END;     (* ensure right outline visible *)
   NewWindowHeight(paperht);
   NewWindowWidth(halfwd);                   (* try left half of paper *)
   NewLocation(paperleft,papertop);          (* top left corner of paper *)
   IF (NOT pageempty) AND outsidepage THEN
      NewLocation(paperleft+halfwd,papertop);   (* try moving right *)
   END;
ELSE
   (* paper shape matches unscaled window shape *)
   NewWindowHeight(paperht);                 (* try all of paper *)
   NewWindowWidth(paperwd);
   NewLocation(paperleft,papertop);          (* top left corner of paper *)
END;
(* If part/all of page is off paper then we set window size and location so
   user can just see ALL of paper AND ALL of page.
*)
IF (NOT pageempty) AND
   ((minhp < paperleft)  OR (minvp < papertop) OR
    (maxhp > paperright) OR (maxvp > paperbottom)) THEN
   NewWindowHeight(Max(maxvp,paperbottom) - Min(minvp,papertop) + 1);
   NewWindowWidth (Max(maxhp,paperright) - Min(minhp,paperleft) + 1);
   NewLocation    (Min(minhp,paperleft),Min(minvp,papertop));
   pageoffpaper := TRUE;
ELSE
   pageoffpaper := FALSE;   (* page is empty or fits on paper *)
END;

paintWindowStatus := TRUE;
paintwindow := TRUE;
END ProcessPage;

(******************************************************************************)

PROCEDURE Min (a,b : INTEGER) : INTEGER;

(* Return the minimum value of a and b. *)

BEGIN
IF a < b THEN RETURN a ELSE RETURN b END;
END Min;

(******************************************************************************)

PROCEDURE Max (a,b : INTEGER) : INTEGER;

(* Return the maximum value of a and b. *)

BEGIN
IF a > b THEN RETURN a ELSE RETURN b END;
END Max;

(******************************************************************************)

PROCEDURE ChangeUnits;

(* Parse the rest of an In, Cm, Mm or PcPtPx command.
   commpos is pointing to next position in commandstr.
*)

VAR nextch : CHAR;

BEGIN
IF commpos < commlen THEN
   nextch := Cap(commstring[commpos]);
   INC(commpos);
ELSE
   nextch := ' ';
END;
IF    (command = In)     AND (nextch = 'N') THEN currentunits := inunits;
ELSIF (command = Cm)     AND (nextch = 'M') THEN currentunits := cmunits;
ELSIF (command = Mm)     AND (nextch = 'M') THEN currentunits := mmunits;
ELSIF (command = PcPtPx) AND (nextch = 'C') THEN currentunits := pcunits;
ELSIF (command = PcPtPx) AND (nextch = 'T') THEN currentunits := ptunits;
ELSIF (command = PcPtPx) AND (nextch = 'X') THEN currentunits := pxunits;
ELSE
   ClearMessageLine;
   WriteString('Unknown units!   ');
   CASE command OF
      In     : WriteString('IN') |
      Cm     : WriteString('CM') |
      Mm     : WriteString('MM') |
      PcPtPx : WriteString('PC, PT or PX')
   END;
   WriteString(' expected.');
   BadCommandMessage;
END;
END ChangeUnits;

(******************************************************************************)

PROCEDURE ShowHelp;

(* Help information is displayed in lines 1 to bottoml-2.
   We assume that bottoml is at least 3 and that VDU screen is at least
   maxline characters wide.
*)

CONST (* SYSDEP: lines in helpname should have <= maxline characters *)
      maxline = 80;

VAR   helpfile : File;
      outline : ARRAY [0..maxline-1] OF CHAR;
      i : CARDINAL;
      lines : INTEGER;
      ch, answer : CHAR;

BEGIN
Open(helpfile,helpname,FALSE);   (* SYSDEP: read only *)
IF NOT Done() THEN
   ClearMessageLine;
   WriteString("Couldn't open help file ");
   WriteString(helpname); Write('!');
   WaitForReturn;
   ClearMessageLine;
ELSE
   ClearScreen;
   MoveToTextLine(1);
   lines := 0;
   LOOP
      ReadChar(helpfile,ch);     (* 1st char in line or EOF *)
      IF Eof(helpfile) THEN
         ClearTextLine(bottoml);
         MoveToTextLine(bottoml);
         WriteString('Hit RETURN key to resume page display:');
         WriteBuffer;
         REPEAT Read(answer) UNTIL answer = CR;
         EXIT;
      ELSIF lines >= (bottoml-2) THEN    (* blank line before prompt *)
         ClearTextLine(bottoml);
         MoveToTextLine(bottoml);
         WriteString('Hit RETURN key to resume page display,');
         WriteString(' or any other key for more help:');
         WriteBuffer;
         Read(answer);
         IF answer = CR THEN EXIT END;
         ClearScreen;
         MoveToTextLine(1);
         lines := 0;             (* reset line count *)
      END;
      outline := '';
      i := 0;
      WHILE ch <> EOL DO
         IF i < maxline THEN outline[i] := ch END;
         ReadChar(helpfile,ch);
         INC(i);
      END;
      WriteString(outline); WriteLn;
      INC(lines);
   END;
   Close(helpfile);
   ClearScreen;
   screenjustcleared := TRUE;
   paintDVIStatus := TRUE;
   paintWindowStatus := TRUE;
   IF currDVIpage <> 0 THEN paintwindow := TRUE END;
END;
END ShowHelp;

(******************************************************************************)

PROCEDURE ShowStatistics;

(* Show qualifier values and rule/font/character/special statistics.
   Note that UserHitsReturn controls pagination and takes the place of WriteLn.
*)

VAR
   linecount, fontcount : INTEGER;
   tempspecial : specialinfoptr;
   ch : CHAR;

BEGIN
ClearScreen;
MoveToTextLine(1);
linecount := 1;
WriteString('DVI file          = '); WriteString(DVIname);
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('VDU               = '); WriteString(vdu);
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('Resolution        = '); WriteCard(resolution);
WriteString(' pixels per inch');
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('Magnification     = '); WriteCard(mag);
IF mag <> DVImag THEN
   WriteString(' (DVI mag of '); WriteCard(DVImag);
   WriteString(' was overridden)');
ELSE
   WriteString(' (DVI mag)');
END;
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('TFM directory     = '); WriteString(tfmdir);
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('PS font prefix    = '); WriteString(psprefix);
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('Font directory    = '); WriteString(fontdir);
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('Dummy font        = '); WriteString(dummyfont);
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('Help file         = '); WriteString(helpname);
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('Horizontal offset = '); WriteDimension(hoffset); WriteUnits;
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('Vertical offset   = '); WriteDimension(voffset); WriteUnits;
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('Paper wd by ht    = ');
WriteDimension(paperwd); WriteUnits; WriteString(' by ');
WriteDimension(paperht); WriteUnits;
IF UserHitsReturn(linecount) THEN RETURN END;
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('Total rules on current page = ');
WriteCard(totalrules);
IF UserHitsReturn(linecount) THEN RETURN END;
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('Total fonts on ALL pages = ');
WriteCard(totalfonts);
IF UserHitsReturn(linecount) THEN RETURN END;
WriteString('(if used on current page then total chars given)');
IF UserHitsReturn(linecount) THEN RETURN END;
fontcount := 0;
thisfontinfo  := fontlist;
WHILE thisfontinfo <> NIL DO
   WITH thisfontinfo^ DO
      IF fontspeclen = 0 THEN               (* need to build fontspec *)
         BuildFontSpec(thisfontinfo);       (* fontexists may become TRUE *)
      END;
      WriteString(fontspec);
      IF psfont THEN WritePtSize(scaledsize) END;
      IF NOT fontexists THEN
         WriteString(' does not exist!');   (* use dummyfont instead *)
      END;
      IF fontused THEN
         INC(fontcount);
         WriteString('   (total chars = ');
         WriteCard(totalchars);   Write(')');
      END;
      IF UserHitsReturn(linecount) THEN RETURN END;
      thisfontinfo := nextfont;
   END;
END;
IF currDVIpage = 0 THEN
   WriteString("You haven't selected a page yet.");
ELSE
   WriteString('Total fonts on current page = ');
   WriteInt(fontcount);
END;
IF speciallist <> NIL THEN
   IF UserHitsReturn(linecount) THEN RETURN END;
   IF UserHitsReturn(linecount) THEN RETURN END;
   WriteString('\special commands:');
   IF UserHitsReturn(linecount) THEN RETURN END;
   tempspecial := speciallist;
   REPEAT
      WITH tempspecial^ DO
         WriteString('At (');
         WriteDimension(hp); Write(','); WriteDimension(vp);
         WriteString('): ');
         WriteString(contents);   (* assume displayable chars??? *)
         tempspecial := nextspecial;
         IF (tempspecial <> NIL) AND UserHitsReturn(linecount) THEN RETURN END;
      END;
   UNTIL tempspecial = NIL;
END;
WriteLn;
MoveToTextLine(bottoml);
WriteString('Hit RETURN key to resume page display:');
WriteBuffer;
REPEAT Read(ch) UNTIL ch = CR;
END ShowStatistics;

(******************************************************************************)

PROCEDURE WriteUnits;

BEGIN
CASE currentunits OF
   inunits : WriteString('in') |
   cmunits : WriteString('cm') |
   mmunits : WriteString('mm') |
   pcunits : WriteString('pc') |
   ptunits : WriteString('pt') |
   pxunits : WriteString('px')
END;
END WriteUnits;

(******************************************************************************)

PROCEDURE UserHitsReturn (VAR linecount : INTEGER) : BOOLEAN;

(* Do a WriteLn and return TRUE iff linecount = bottoml-2 AND user hits CR.
   If linecount < bottoml-2 then return FALSE; if not, and user hits
   something other than CR, then prepare a new screen before returning FALSE.
*)

VAR ch : CHAR;

BEGIN
WriteLn;
IF linecount = bottoml-2 THEN    (* prompt for next screen *)
   MoveToTextLine(bottoml);
   WriteString('Hit RETURN key to resume page display,');
   WriteString(' or any other key for more:');
   WriteBuffer;
   Read(ch);
   IF ch = CR THEN RETURN TRUE END;
   ClearScreen;
   MoveToTextLine(1);
   linecount := 1;
ELSE
   INC(linecount);
END;
RETURN FALSE;
END UserHitsReturn;

(******************************************************************************)

PROCEDURE DisplayPage;

(* Display page in window region based on window location and size,
   and displaymode.  This routine is only called if paintwindow is TRUE
   after all commands have been processed.
*)

VAR vispage : REAL;        (* fraction of page rectangle currently visible *)
    left, right,           (* visible edges of page rectangle *)
    top, bottom : INTEGER;

BEGIN
IF screenjustcleared THEN   (* avoid doing it again *)
   IF paintDVIStatus THEN UpdateDVIStatusLine END;
   IF paintWindowStatus THEN UpdateWindowStatusLine END;
ELSE
   ClearScreen;   (* would prefer ClearWindow but some VDUs have trouble *)
   UpdateDVIStatusLine;
   UpdateWindowStatusLine;
END;
StartGraphics;
DisplayPaperEdges;
StartText;
IF pageempty THEN
   ClearMessageLine;
   WriteString('Page is empty.');
ELSIF outsidepage THEN
   IF pageoffpaper THEN CheckPageEdges END;
   ClearMessageLine;
   WriteString('Window is ');
   IF windowtop > maxvp THEN
      WriteString('below ');
      IF (windowleft > maxhp) OR (windowleft < minhp - scaledwd + 1) THEN
         WriteString('and ');
      END;
   ELSIF windowtop < minvp - scaledht + 1 THEN
      WriteString('above ');
      IF (windowleft > maxhp) OR (windowleft < minhp - scaledwd + 1) THEN
         WriteString('and ');
      END;
   END;
   IF windowleft > maxhp THEN
      WriteString('to the right of ');
   ELSIF windowleft < minhp - scaledwd + 1 THEN
      WriteString('to the left of ');
   END;
   WriteString('page.');
ELSE
   (* Page is not empty and part or all of it is visible. *)
   StartGraphics;
   useraborted := FALSE;
   DisplayRules;
   IF NOT useraborted THEN
      DisplayChars;
   END;
   StartText;
   IF pageoffpaper THEN CheckPageEdges END;
   IF allpagevisible THEN
      ClearMessageLine;
      WriteString('Entire page is visible.');
   END;
END;
END DisplayPage;

(******************************************************************************)

PROCEDURE DisplayPaperEdges;

(* Display visible outlines of the imaginary sheet of paper.
   Thickness of outlines = 1 screen pixel no matter what the h and v scaling.
*)

CONST
   edgepixel = '.';             (* black pixel for outlines on non-graphic VDUs;
                                   note that VDUInterface sets
                                   TeXtoASCII['.'] := '.'       *)

VAR
   top, bot, left, right,       (* visible edges of paper in paper pixels *)
   scaledtop, scaledleft,       (* scaled visible edges in screen pixels *)
   scaledbot, scaledright,
   scaledheight, scaledwidth    (* scaled width and height *)
   : INTEGER;

BEGIN
(* first check if any part of paper is visible *)
IF papertop    > windowbottom THEN RETURN END;
IF paperbottom < windowtop    THEN RETURN END;
IF paperleft   > windowright  THEN RETURN END;
IF paperright  < windowleft   THEN RETURN END;
(* part or all of paper is visible, so return visible region *)
top   := Max(papertop,windowtop);
bot   := Min(paperbottom,windowbottom);
left  := Max(paperleft,windowleft);
right := Min(paperright,windowright);
scaledtop  := ScaleVpos(top - windowtop) + windowv;
scaledleft := ScaleHpos(left - windowleft) + windowh;
IF vscalefactor > 1.0 THEN
   scaledbot    := ScaleVpos(bot + 1 - windowtop) - 1 + windowv;
ELSE
   scaledbot    := ScaleVpos(bot - windowtop) + windowv;
END;
IF hscalefactor > 1.0 THEN
   scaledright  := ScaleHpos(right + 1 - windowleft) - 1 + windowh;
ELSE
   scaledright  := ScaleHpos(right - windowleft) + windowh;
END;
scaledheight := scaledbot - scaledtop + 1;
scaledwidth  := scaledright - scaledleft + 1;
(* Only show visible edges if they are also paper outlines! *)
IF left = paperleft THEN          (* left outline visible *)
   ShowRectangle(scaledleft, scaledtop, 1, scaledheight, edgepixel);
END;
IF bot = paperbottom THEN         (* bottom outline visible *)
   ShowRectangle(scaledleft, scaledbot, scaledwidth, 1, edgepixel);
END;
IF top = papertop THEN            (* top outline visible *)
   ShowRectangle(scaledleft, scaledtop, scaledwidth, 1, edgepixel);
END;
IF right = paperright THEN        (* right outline visible *)
   ShowRectangle(scaledright, scaledtop, 1, scaledheight, edgepixel);
END;
END DisplayPaperEdges;

(******************************************************************************)

PROCEDURE DisplayRules;

(* Display all pixels in rules, regardless of current displaymode.
   Rules will be displayed in the same order as in the DVI page (essentially
   top-down and left-right) because of the way DVIReader builds a rulelist.
*)

CONST
   rulepixel = '*';             (* black pixel for rules on non-graphic VDUs;
                                   note that VDUInterface sets
                                   TeXtoASCII['*'] := '*'       *)

VAR
   top, bottom, left, right,    (* visible edges of rule *)
   scaledtop, scaledleft,       (* scaled visible edges *)
   scaledbot, scaledright,
   scaledwidth, scaledheight    (* scaled width and height *)
     : INTEGER;
   thisrule : CARDINAL;
   keyhit : CHAR;               (* returned by BusyRead if TRUE *)

BEGIN
thisruleinfo := rulelist;
WHILE thisruleinfo <> NIL DO
   WITH thisruleinfo^ DO
      thisrule := 0;
      WHILE thisrule < rulecount DO
         WITH ruletable[thisrule] DO
            (* check if any part of rule is visible *)
            (* vp,hp is bottom left corner of rule on page *)
            IF RectangleVisible
                  (vp-ht+1,vp,hp,hp+wd-1,   (* rule edges *)
                   top,bottom,left,right)   (* visible rectangle *)
               THEN
               (* show all pixels in this rectangle *)
               scaledtop  := ScaleVpos(top - windowtop) + windowv;
               scaledleft := ScaleHpos(left - windowleft) + windowh;
               IF vscalefactor > 1.0 THEN
                  scaledbot   := ScaleVpos(bottom+1-windowtop) - 1 + windowv;
               ELSE
                  scaledbot   := ScaleVpos(bottom-windowtop) + windowv;
               END;
               IF hscalefactor > 1.0 THEN
                  scaledright := ScaleHpos(right+1-windowleft) - 1 + windowh;
               ELSE
                  scaledright := ScaleHpos(right-windowleft) + windowh;
               END;
               scaledheight := scaledbot - scaledtop + 1;
               scaledwidth  := scaledright - scaledleft + 1;
               ShowRectangle
                  (scaledleft,        (* h coord of top left cnr *)
                   scaledtop,         (* v coord of top left cnr *)
                   scaledwidth,
                   scaledheight,
                   rulepixel);
               (* only check keyboard after every nth visible rule (n ~ 4) *)
               IF (thisrule MOD 4 = 0) AND BusyRead(keyhit) THEN
                  keyhit := Cap(keyhit);
                  IF (keyhit = Terse) AND (displaymode <> tersemode) THEN
                     displaymode := tersemode;
                     StartText;
                     UpdateDVIStatusLine;
                     StartGraphics;
                  ELSIF (keyhit = Box) AND (displaymode <> boxmode) THEN
                     displaymode := boxmode;
                     StartText;
                     UpdateDVIStatusLine;
                     StartGraphics;
                  ELSIF (keyhit = Full) AND (displaymode <> fullmode) THEN
                     displaymode := fullmode;
                     StartText;
                     UpdateDVIStatusLine;
                     StartGraphics;
                  ELSIF keyhit = abortkey THEN
                     useraborted := TRUE;   (* checked in DisplayPage *)
                     RETURN;
                  END;
               END;
            END;
         END;
         INC(thisrule);
      END;
      thisruleinfo := nextrule;
   END;
END;
END DisplayRules;

(******************************************************************************)

PROCEDURE RectangleVisible (intop, inbot, inleft, inright : INTEGER;
                            VAR outtop, outbot, outleft, outright : INTEGER
                           ) : BOOLEAN;

(* Return TRUE iff part or all of given rectangle would be visible
   in the current window.  Iff so, then we also return the visible
   region; the input and possible output rectangles are defined by their
   top, bottom, left and right edges in paper pixel coordinates.
*)

BEGIN
IF allpagevisible THEN   (* all of rectangle must be visible *)
   outtop := intop; outbot := inbot; outleft := inleft; outright := inright;
   RETURN TRUE;
END;
IF intop   > windowbottom THEN RETURN FALSE END;
IF inbot   < windowtop THEN RETURN FALSE END;
IF inleft  > windowright THEN RETURN FALSE END;
IF inright < windowleft THEN RETURN FALSE END;
(* part or all of rectangle is visible, so return visible region *)
outtop   := Max(intop,windowtop);
outbot   := Min(inbot,windowbottom);
outleft  := Max(inleft,windowleft);
outright := Min(inright,windowright);
RETURN TRUE;
END RectangleVisible;

(******************************************************************************)

PROCEDURE DisplayChars;

(* Display all characters on a font by font basis.  How characters will be
   represented depends on the current displaymode (which the user can change
   while the window is being updated by typing the Terse/Box/Full commands).
   Fonts will be displayed in order of ascending totalchars (due to SortFonts).
   Characters in a font will be displayed in a top-down, left-right manner
   because of the way DVIReader builds a charlist.
*)

VAR keyhit   : CHAR;       (* check for abort or mode change *)

BEGIN
thisfontinfo := fontlist;
WHILE thisfontinfo <> unusedfont DO
   (* SortFont makes sure we only consider used fonts *)
   WITH thisfontinfo^ DO
      fontseen := FALSE;
      fontopen := FALSE;
      CASE displaymode OF
         tersemode : DisplayOneChar := TerseChar |
         boxmode   : DisplayOneChar := BoxChar   |
         fullmode  : IF psfont THEN
                        DisplayOneChar := FullCharPS;
                     ELSIF (vscalefactor < 1.0) OR (hscalefactor < 1.0) THEN
                        DisplayOneChar := FullChar2;
                     ELSE
                        DisplayOneChar := FullChar1;
                     END
      END;

      (* To help some VDUs select appropriately sized characters, we need to
         pass the scaledsize of the font (converted to unscaled paper pixels),
         the overall mag, and the current h/vscalefactors.
      *)
      LoadFont(fontspec,
               PixelRound(scaledsize),
               FLOAT(mag)/1000.0,
               hscalefactor,
               vscalefactor);

      thischarinfo := charlist;
      WHILE thischarinfo <> NIL DO    (* display chars in chartable *)
         WITH thischarinfo^ DO
            thischar := 0;
            WHILE thischar < charcount DO
               DisplayOneChar;

               (* SYSDEP:
                  We only check for abort or mode change after ~ 4 visible
                  characters to avoid too many BusyRead calls.
                  It would have been nicer to use an AST routine for keyboard
                  interrupts but it only seems possible for control characters.
               *)
               IF charvisible AND (thischar MOD 4 = 0) AND BusyRead(keyhit) THEN
                  keyhit := Cap(keyhit);
                  IF (keyhit = Terse) AND (displaymode <> tersemode) THEN
                     DisplayOneChar := TerseChar;
                     displaymode := tersemode;
                     StartText;
                     UpdateDVIStatusLine;
                     StartGraphics;
                  ELSIF (keyhit = Box) AND (displaymode <> boxmode) THEN
                     DisplayOneChar := BoxChar;
                     displaymode := boxmode;
                     StartText;
                     UpdateDVIStatusLine;
                     StartGraphics;
                  ELSIF (keyhit = Full) AND (displaymode <> fullmode) THEN
                     IF psfont THEN
                        DisplayOneChar := FullCharPS;
                     ELSIF (vscalefactor < 1.0) OR (hscalefactor < 1.0) THEN
                        DisplayOneChar := FullChar2;
                     ELSE
                        DisplayOneChar := FullChar1;
                     END;
                     displaymode := fullmode;
                     StartText;
                     UpdateDVIStatusLine;
                     StartGraphics;
                  ELSIF keyhit = abortkey THEN
                     IF fontseen THEN
                        IF fontopen THEN CloseFontFile END;
                        StartText;
                        ClearMessageLine;   (* clear drawing message *)
                     END;
                     (* no need to set useraborted; DisplayRules done first *)
                     RETURN;
                  END;
               END;

               INC(thischar);
            END;
            thischarinfo := nextchar;
         END;
      END;
      IF fontseen THEN
         IF fontopen THEN CloseFontFile END;
         StartText;
         ClearMessageLine;   (* clear drawing message *)
         StartGraphics;      (* might be more fonts *)
      END;
   thisfontinfo := nextfont;
   END;
END;
END DisplayChars;

(******************************************************************************)

PROCEDURE TerseChar;

(* Display a quick and nasty representation of character only if ref pt visible.
   Just how good the representation is depends on the capabilities of the VDU.
   We don't bother checking if glyph is actually all white or non-existent.
*)

BEGIN
WITH thisfontinfo^ DO
WITH thischarinfo^.chartable[thischar] DO
   IF PixelVisible(hp,vp) THEN   (* ref pt of char is visible *)
      ShowChar(ScaleHpos(hp - windowleft) + windowh,
               ScaleVpos(vp - windowtop) + windowv,
               CHR(code));
      charvisible := TRUE;
   ELSE
      charvisible := FALSE;   (* checked in DisplayChars *)
   END;
END;
END;
END TerseChar;

(******************************************************************************)

PROCEDURE PixelVisible (hpos, vpos : INTEGER) : BOOLEAN;

(* Return TRUE iff given paper pixel would be visible in current window. *)

BEGIN
IF allpagevisible THEN RETURN TRUE END;
IF vpos < windowtop THEN RETURN FALSE END;
IF vpos > windowbottom THEN RETURN FALSE END;
IF hpos < windowleft THEN RETURN FALSE END;
IF hpos > windowright THEN RETURN FALSE END;
RETURN TRUE;
END PixelVisible;

(******************************************************************************)

PROCEDURE BoxChar;

(* Display visible box outlines of glyph.
   Thickness of outlines = 1 screen pixel no matter what the h and v scaling.
*)

VAR
   vpmyo, hpmxo,                (* vp-yo, hp-xo: glyph's top and left edges *)
   top, bottom, left, right,    (* visible edges of glyph *)
   scaledtop, scaledleft,       (* scaled visible edges *)
   scaledbot, scaledright,
   scaledheight, scaledwidth    (* scaled width and height *)
      : INTEGER;
   ch : CHAR;

BEGIN
WITH thisfontinfo^ DO
WITH thischarinfo^.chartable[thischar] DO
WITH pixelptr^[code] DO
   IF mapadr = 0 THEN RETURN END;              (* glyph all white or absent *)
   (* check if any part of glyph is visible *)
   vpmyo := vp-yo;
   hpmxo := hp-xo;
   IF RectangleVisible
         (vpmyo, vpmyo+ht-1, hpmxo, hpmxo+wd-1,(* glyph edges *)
          top,bottom,left,right)               (* visible part *)
      THEN
      scaledtop  := ScaleVpos(top - windowtop) + windowv;
      scaledleft := ScaleHpos(left - windowleft) + windowh;
      IF vscalefactor > 1.0 THEN
         scaledbot    := ScaleVpos(bottom + 1 - windowtop) - 1 + windowv;
      ELSE
         scaledbot    := ScaleVpos(bottom - windowtop) + windowv;
      END;
      IF hscalefactor > 1.0 THEN
         scaledright  := ScaleHpos(right + 1 - windowleft) - 1 + windowh;
      ELSE
         scaledright  := ScaleHpos(right - windowleft) + windowh;
      END;
      scaledheight := scaledbot - scaledtop + 1;
      scaledwidth  := scaledright - scaledleft + 1;
      ch := CHR(code);
      (* Only show edges that are also glyph outlines!
         Following method reduces the number of ShowRectangle calls needed for
         very small boxes.
      *)
      IF ((scaledheight < 3) AND (top = vpmyo) AND (bottom = vpmyo+ht-1)) OR
         ((scaledwidth < 3) AND (left = hpmxo) AND (right = hpmxo+wd-1)) THEN
         ShowRectangle(scaledleft, scaledtop, scaledwidth, scaledheight, ch);
      ELSE
         IF left = hpmxo THEN           (* left outline visible *)
            ShowRectangle(scaledleft, scaledtop, 1, scaledheight, ch);
         END;
         IF bottom = vpmyo+ht-1 THEN    (* bottom outline visible *)
            ShowRectangle(scaledleft, scaledbot, scaledwidth, 1, ch);
         END;
         IF top = vpmyo THEN            (* top outline visible *)
            ShowRectangle(scaledleft, scaledtop, scaledwidth, 1, ch);
         END;
         IF right = hpmxo+wd-1 THEN     (* right outline visible *)
            ShowRectangle(scaledright, scaledtop, 1, scaledheight, ch);
         END;
      END;
      charvisible := TRUE;
   ELSE
      charvisible := FALSE;   (* checked in DisplayChars *)
   END;
END;
END;
END;
END BoxChar;

(******************************************************************************)

PROCEDURE FullCharPS;

(* Display filled rectangle approximating the extent of PostScript glyph. *)

VAR
   vpmyo, hpmxo,                (* vp-yo, hp-xo: glyph's top and left edges *)
   top, bottom, left, right,    (* visible edges of glyph *)
   scaledtop, scaledleft,       (* scaled visible edges *)
   scaledbot, scaledright,
   scaledheight, scaledwidth    (* scaled width and height *)
      : INTEGER;

BEGIN
WITH thisfontinfo^ DO
WITH thischarinfo^.chartable[thischar] DO
WITH pixelptr^[code] DO
   IF mapadr = 0 THEN RETURN END;              (* glyph all white or absent *)
   (* check if any part of glyph is visible *)
   vpmyo := vp-yo;
   hpmxo := hp-xo;
   IF RectangleVisible
         (vpmyo, vpmyo+ht-1, hpmxo, hpmxo+wd-1,(* glyph edges *)
          top,bottom,left,right)               (* visible part *)
      THEN
      IF NOT fontseen THEN
         DrawingFont;
         fontseen := TRUE;                     (* only display message once *)
      END;
      scaledtop  := ScaleVpos(top - windowtop) + windowv;
      scaledleft := ScaleHpos(left - windowleft) + windowh;
      IF vscalefactor > 1.0 THEN
         scaledbot    := ScaleVpos(bottom + 1 - windowtop) - 1 + windowv;
      ELSE
         scaledbot    := ScaleVpos(bottom - windowtop) + windowv;
      END;
      IF hscalefactor > 1.0 THEN
         scaledright  := ScaleHpos(right + 1 - windowleft) - 1 + windowh;
      ELSE
         scaledright  := ScaleHpos(right - windowleft) + windowh;
      END;
      scaledheight := scaledbot - scaledtop + 1;
      scaledwidth  := scaledright - scaledleft + 1;
      ShowRectangle(scaledleft, scaledtop,
                    scaledwidth, scaledheight, CHR(code));
      charvisible := TRUE;
   ELSE
      charvisible := FALSE;   (* checked in DisplayChars *)
   END;
END;
END;
END;
END FullCharPS;

(******************************************************************************)

PROCEDURE FullChar1;

(* Display all pixels in a glyph using bitmap from font file.
   This procedure is assigned to DisplayOneChar when h AND vscalefactors are
   >= 1.0, so we don't have to worry about scaledheights/widths being 0.
*)

VAR
   vpmyo, hpmxo,               (* vp-yo, hp-xo: glyph's top and left edges *)
   top, bottom, left, right,   (* visible edges of glyph *)
   scaledv, scalednextv,       (* scaled vertical positions for rows *)
   scaledh,                    (* scaled h coord of start of run within row *)
   scaledwidth, scaledheight,  (* scaled width and height of row *)
   thisrow, thisbit            (* in paper coordinates *)
      : INTEGER;
   ptr : ADDRESS;              (* pointer into bitmap *)
   wordsperrow,                (* rows of bitmap are word aligned *)
   firstbit, lastbit,          (* 0..wordsperrow*32 - 1 *)
   firstword,                  (* 0..wordsperrow-1 *)
   bitpos : CARDINAL;          (* 0..31 *)
   glyphword : BITSET;         (* current word in bitmap *)
   inrun : BOOLEAN;            (* are we in a run of black pixels in row? *)

BEGIN
WITH thisfontinfo^ DO
WITH thischarinfo^.chartable[thischar] DO
WITH pixelptr^[code] DO
   IF mapadr = 0 THEN RETURN END;              (* glyph all white or absent *)
   (* check if any part of glyph is visible *)
   vpmyo := vp-yo;
   hpmxo := hp-xo;
   IF RectangleVisible
         (vpmyo,vpmyo+ht-1,hpmxo,hpmxo+wd-1,   (* glyph edges *)
          top,bottom,left,right)               (* visible part *)
      THEN
      IF NOT fontseen THEN
         DrawingFont;
         fontseen := TRUE;                     (* only display message once *)
      END;
      IF bitmap = NIL THEN
         IF NOT fontopen THEN
            IF fontexists THEN
               IF NOT OpenFontFile(fontspec) THEN NotFound(fontspec) END;
            ELSE
               IF NOT OpenFontFile(dummyfont) THEN NotFound(dummyfont) END;
            END;
            fontopen := TRUE;                  (* only open font once *)
         END;
         GetBitmap(ht, wd,                     (* dimensions of bitmap *)
                   mapadr,                     (* bitmap info in font file *)
                   bitmap);                    (* starting address of bitmap *)
      END;
      wordsperrow := (wd + 31) DIV 32;         (* words in one row of bitmap *)
      firstbit := CARDINAL(left-hpmxo);        (* first visible bit in row *)
      lastbit  := CARDINAL(right-hpmxo);       (* last visible bit *)
      firstword := firstbit DIV 32;            (* first visible word *)
      (* calculate scaled v coord of first visible row *)
      scaledv := ScaleVpos(top - windowtop) + windowv;

      (* only consider visible rows; thisrow := top to bottom *)
      thisrow := top;
      LOOP
         (* calculate scaled v coord of next row *)
         scalednextv  := ScaleVpos(thisrow + 1 - windowtop) + windowv;
         scaledheight := scalednextv - scaledv;   (* can't be 0 *)
         (* move to first byte of first visible word in this row *)
         ptr := bitmap +
                4 * (CARDINAL(thisrow - vpmyo) * wordsperrow + firstword);
         glyphword := BITSET(ptr^);
         bitpos := 31 - (firstbit MOD 32);   (* 31..0 *)
         inrun := FALSE;

         (* display black pixel runs in row, doing any h/v expansion *)
         (* only consider visible bits; thisbit := left to right *)
         thisbit := left;
         LOOP
            IF bitpos IN glyphword THEN      (* start/continue run *)
               IF NOT inrun THEN
                  inrun := TRUE;
                  (* remember start of run *)
                  scaledh := ScaleHpos(thisbit - windowleft) + windowh;
               END;
            ELSIF inrun THEN                 (* 0 bit has ended run *)
               inrun := FALSE;
               scaledwidth := ScaleHpos(thisbit - windowleft) + windowh
                              - scaledh;
               ShowRectangle
                  (scaledh,scaledv,scaledwidth,scaledheight,CHR(code));
            END;
            IF thisbit = right THEN EXIT (* bit loop *) END;
            IF bitpos = 0 THEN
               INC(ptr,4);       (* move to next word of bitmap *)
               glyphword := BITSET(ptr^);
               bitpos := 31;     (* look at first bit in this word *)
            ELSE
               DEC(bitpos);      (* look at next bit in word *)
            END;
            INC(thisbit);
         END; (* bit loop *)

         IF inrun THEN        (* show run at end of row; INC thisbit *)
            scaledwidth := ScaleHpos(thisbit + 1 - windowleft) + windowh
                           - scaledh;
            ShowRectangle
               (scaledh,scaledv,scaledwidth,scaledheight,CHR(code));
         END;
         IF thisrow = bottom THEN EXIT (* row loop *) END;
         scaledv := scalednextv;
         INC(thisrow);
      END; (* row loop *)

      charvisible := TRUE;
   ELSE
      charvisible := FALSE;   (* checked in DisplayChars *)
   END;
END;
END;
END;
END FullChar1;

(******************************************************************************)

PROCEDURE FullChar2;

(* Display all pixels in a glyph using bitmap from font file.
   This procedure is assigned to DisplayOneChar when h/vscalefactor < 1.0.
   The algorithm avoids overlapping rows when vscalefactor < 1.0.
   When hscalefactor < 1.0, it is not worth the extra code to avoid overlapping
   runs of 1 bits because the majority of character glyphs have only one or two
   runs per row.
*)

CONST
   (* SYSDEP: 200 * 32 = 6400 bits = maximum pixel width of glyph.
      If any fonts have glyphs wider than this then increase maxviswords.
   *)
   maxviswords = 200;

TYPE
   (* SYSDEP: BITSET is 32 bit word with elements 31,30,29,...,0 *)
   glyphrow = ARRAY [0..maxviswords-1] OF BITSET;

VAR
   vpmyo, hpmxo,               (* vp-yo, hp-xo: glyph's top and left edges *)
   top, bottom, left, right,   (* visible edges of glyph *)
   scaledv, scalednextv,       (* scaled vertical positions for rows *)
   scaledh,                    (* scaled horizontal positions within row *)
   scaledwidth, scaledheight,  (* scaled width and height of row *)
   thisrow, thisbit            (* in paper coordinates *)
       : INTEGER;
   row : glyphrow;             (* holds VISIBLE bits in one row of glyph;
                                  possibly > one row if vscalefactor < 1.0 *)
   ptr : ADDRESS;              (* pointer into bitmap *)
   wordsperrow,                (* rows of bitmap are word aligned *)
   firstbit, lastbit,          (* somewhere in 0 .. wordsperrow*32-1 *)
   firstword, lastword,        (* somewhere in 0 .. wordsperrow-1 *)
   endword,                    (* = visible words in row, - 1 *)
   wordpos,                    (* 0 .. endword *)
   bitpos,                     (* 31 .. 0 *)
   i : CARDINAL;
   inrun : BOOLEAN;            (* are we in a run of black pixels in row? *)

BEGIN
WITH thisfontinfo^ DO
WITH thischarinfo^.chartable[thischar] DO
WITH pixelptr^[code] DO
   IF mapadr = 0 THEN RETURN END;              (* glyph all white or absent *)
   (* check if any part of glyph is visible *)
   vpmyo := vp-yo;
   hpmxo := hp-xo;
   IF RectangleVisible
         (vpmyo,vpmyo+ht-1,hpmxo,hpmxo+wd-1,   (* glyph edges *)
          top,bottom,left,right)               (* visible part *)
      THEN
      IF NOT fontseen THEN
         DrawingFont;
         fontseen := TRUE;                     (* only display message once *)
      END;
      IF bitmap = NIL THEN
         IF NOT fontopen THEN
            IF fontexists THEN
               IF NOT OpenFontFile(fontspec) THEN NotFound(fontspec) END;
            ELSE
               IF NOT OpenFontFile(dummyfont) THEN NotFound(dummyfont) END;
            END;
            fontopen := TRUE;                  (* only open font once *)
         END;
         GetBitmap(ht, wd,                     (* dimensions of bitmap *)
                   mapadr,                     (* bitmap info in font file *)
                   bitmap);                    (* starting address of bitmap *)
      END;
      wordsperrow := (wd + 31) DIV 32;         (* words in one row of bitmap *)
      firstbit    := CARDINAL(left-hpmxo);     (* first visible bit *)
      lastbit     := CARDINAL(right-hpmxo);    (* last visible bit *)
      firstword   := firstbit DIV 32;          (* first visible word *)
      lastword    := lastbit DIV 32;           (* last visible word *)
      endword     := lastword - firstword;

      (* we impose a limit on width of glyph (unlikely to be exceeded) *)
      IF endword > maxviswords-1 THEN
         endword := maxviswords-1;             (* truncate glyph *)
      END;

      (* set the visible words in row to 0 *)
      FOR i := 0 TO endword DO row[i] := {} END;
      (* calculate scaled v coord of first visible row *)
      scaledv := ScaleVpos(top - windowtop) + windowv;

      (* only consider visible rows; thisrow := top to bottom *)
      thisrow := top;
      LOOP
         (* move to first byte of first visible word in this row *)
         ptr := bitmap +
                4 * (CARDINAL(thisrow - vpmyo) * wordsperrow + firstword);
         (* get row of visible words from bitmap and OR with row array *)
         wordpos := 0;
         LOOP
            row[wordpos] := BITSET(ptr^) + row[wordpos];   (* set union *)
            IF wordpos = endword THEN EXIT END;
            INC(wordpos);
            INC(ptr,4);                                    (* next word *)
         END;
         (* calculate scaled v coord of next row *)
         scalednextv  := ScaleVpos(thisrow + 1 - windowtop) + windowv;
         scaledheight := scalednextv - scaledv;
         IF (scaledheight > 0) OR (thisrow = bottom) THEN
            (* display black pixels in row, doing any h/v expansion *)
            IF scaledheight < 1 THEN scaledheight := 1 END;   (* avoid 0 *)
            inrun := FALSE;
            bitpos := 31 - (firstbit MOD 32);   (* 31..0 *)
            wordpos := 0;

            (* only consider visible bits; thisbit := left to right *)
            thisbit := left;
            LOOP
               IF bitpos IN row[wordpos] THEN   (* start/continue run *)
                  IF NOT inrun THEN             (* remember start of run *)
                     inrun := TRUE;
                     scaledh := ScaleHpos(thisbit - windowleft) + windowh;
                  END;
               ELSIF inrun THEN     (* 0 bit has ended run *)
                  inrun := FALSE;
                  scaledwidth := ScaleHpos(thisbit - windowleft) + windowh
                                 - scaledh;
                  IF scaledwidth < 1 THEN scaledwidth := 1 END;   (* avoid 0 *)
                  ShowRectangle
                     (scaledh,scaledv,scaledwidth,scaledheight,CHR(code));
               END;
               IF thisbit = right THEN EXIT (* bit loop *) END;
               IF bitpos = 0 THEN   (* look at first bit in next word of row *)
                  INC(wordpos);
                  bitpos := 31;
               ELSE                 (* look at next bit in word *)
                  DEC(bitpos);
               END;
               INC(thisbit);
            END; (* bit loop *)
            IF inrun THEN           (* show run at end of row; INC thisbit *)
               scaledwidth := ScaleHpos(thisbit + 1 - windowleft) + windowh
                              - scaledh;
               IF scaledwidth < 1 THEN scaledwidth := 1 END;   (* avoid 0 *)
               ShowRectangle
                  (scaledh,scaledv,scaledwidth,scaledheight,CHR(code));
            END;

            IF thisrow = bottom THEN EXIT (* row loop *) END;
            (* else reset the visible words in row to 0 *)
            FOR i := 0 TO endword DO row[i] := {} END;
         END;
         scaledv := scalednextv;
         INC(thisrow);
      END; (* row loop *)

      charvisible := TRUE;
   ELSE
      charvisible := FALSE;   (* checked in DisplayChars *)
   END;
END;
END;
END;
END FullChar2;

(******************************************************************************)

PROCEDURE DrawingFont;

(* We are about to draw the first visible character (in a Full display) of
   thisfontinfo^.fontspec.
*)

BEGIN
StartText;
ClearMessageLine;
WriteString("Drawing characters from ");
WITH thisfontinfo^ DO
   IF fontexists THEN
      WriteString(fontspec);
      IF psfont THEN WritePtSize(scaledsize) END;
   ELSE
      WriteString("dummy font!");
   END;
END;
WriteLn;
StartGraphics;
END DrawingFont;

(******************************************************************************)

PROCEDURE NotFound (VAR fspec : ARRAY OF CHAR);

(* The given font could not be opened even though it was successfully opened
   before in MyPixelTableRoutine.
*)

BEGIN
StartText;
ClearMessageLine;
ResetVDU;
WriteString("Couldn't open font "); WriteString(fspec); Write('!'); WriteLn;
PleaseReport;
Halt(2);
END NotFound;

(******************************************************************************)

PROCEDURE CheckPageEdges;

(* One or more page edges do not fall within the paper edges.
   This routine is called after the page & paper have been displayed so
   user can see how bad the problem is.
*)

BEGIN
IF minhp < paperleft THEN
   ClearMessageLine;
   WriteString('Page beyond left edge by ');
   WriteDimension(paperleft - minhp);
   PaperMessage;
END;
IF maxhp > paperright THEN
   ClearMessageLine;
   WriteString('Page beyond right edge by ');
   WriteDimension(maxhp - paperright);
   PaperMessage;
END;
IF minvp < papertop THEN
   ClearMessageLine;
   WriteString('Page above top edge by ');
   WriteDimension(papertop - minvp);
   PaperMessage;
END;
IF maxvp > paperbottom THEN
   ClearMessageLine;
   WriteString('Page below bottom edge by ');
   WriteDimension(maxvp - paperbottom);
   PaperMessage;
END;
END CheckPageEdges;

(******************************************************************************)

PROCEDURE PaperMessage;

(* Called by CheckPageEdges to remind user of the paper size. *)

BEGIN
CASE currentunits OF
   inunits : WriteString('in') |
   cmunits : WriteString('cm') |
   mmunits : WriteString('mm') |
   pcunits : WriteString('pc') |
   ptunits : WriteString('pt') |
   pxunits : WriteString('px')
END;
WriteString('!   (Paper is ');
WriteDimension(paperwd);   WriteString(' by ');
WriteDimension(paperht);   Write(')');
WaitForReturn;
ClearMessageLine;
END PaperMessage;

(******************************************************************************)

PROCEDURE Finish;

BEGIN
CloseDVIFile;
ClearScreen;
MoveToTextLine(1);
WriteLn;
ResetVDU;
Halt(1);    (* restore terminal channel and exit with success *)
END Finish;

(******************************************************************************)

BEGIN
TopLevel;
END DVItoVDU.
