/*
   GNU xhippo: a GTK-based playlist manager.
   Copyright 1998, 1999, 2000 Adam Sampson <azz@gnu.org>

   Please report bugs to bug-xhippo@gnu.org.

   This file is part of GNU xhippo.

   GNU xhippo is free software; you can redistribute and/or modify it
   under the terms of that license as published by the Free Software
   Foundation; either version 2 of the License, or (at your option)
   any later version.
   
   GNU xhippo is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with GNU xhippo; see the file COPYING. If not, write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
   MA 02111-1307 USA, or see http://www.gnu.org/.

*/

#include "xhippo.h"

/* Prototypes for static functions. */
static void add_file_action(gchar *extension, gchar *action);
static void add_user_command(gchar *description, gchar *command);

/* Globals. */
guint mode_play_on_start = 0, 
  mode_scroll_catchup = 0, 
  mode_left_scroll = 0,
  mode_must_play_all = 0, 
  mode_one_time = 0, 
  mode_show_pid = 0,
  mode_read_id3_tags = 0,
  mode_save_window_pos = 0,
  mode_start_mini = 0,
  mode_play_ordered = 0,
  mode_strip_extension = 0,
  mode_hide_player_output = 0,
  mode_sort_on_load = 0,
  mode_demangle_names = 0,
  mode_playlist_title = 0,
  mode_title_basename = 0,
  mode_no_check_files = 0,
  mode_write_playing = 0,
  mode_skip_path = 0,
  attribute_mini = 0;
gchar *attribute_playlist_dir = NULL;
gint attribute_xpos = 100, 
  attribute_ypos = 100, 
  attribute_width = 300, 
  attribute_height = 300;
fileaction *fileactions = NULL;
usercommand *usercommands = NULL;

/* Configuration commands. */
static configcommand configcommands[] = {
  { 0, CT_INT, "autostart", &mode_play_on_start },
  { 0, CT_INT, "scroll", &mode_scroll_catchup },
  { 0, CT_INT, "leftscroll", &mode_left_scroll },
  { 0, CT_INT, "mustplayall", &mode_must_play_all },
  { 0, CT_INT, "onetime", &mode_one_time },
  { 0, CT_INT, "showpid", &mode_show_pid },
  { 0, CT_INT, "readid3", &mode_read_id3_tags },
  { 0, CT_INT, "savewinstate", &mode_save_window_pos },
  { 0, CT_INT, "startmini", &mode_start_mini },
  { 0, CT_INT, "ordered", &mode_play_ordered },
  { 0, CT_INT, "stripextension", &mode_strip_extension },
  { 0, CT_INT, "hideplayeroutput", &mode_hide_player_output },
  { 0, CT_INT, "sortonload", &mode_sort_on_load },
  { 0, CT_INT, "demanglenames", &mode_demangle_names },
  { 0, CT_INT, "playlisttitle", &mode_playlist_title },
  { 0, CT_INT, "titlebasename", &mode_title_basename },
  { 0, CT_INT, "nocheckfiles", &mode_no_check_files },
  { 0, CT_INT, "writeplaying", &mode_write_playing },
  { 0, CT_INT, "skippath", &mode_skip_path },
  { 0, CT_STRING, "playlistdir", &attribute_playlist_dir },
  { 0, CT_FUNCTION2, "type", add_file_action },
  { 0, CT_FUNCTION2, "usercommand", add_user_command },
  { 1, CT_FUNCTION1, "load", read_playlist },
  { 1, CT_FUNCTION1, "exec", system },
  { 0, CT_END, NULL, NULL }
};

/* Add a file action to the list. */
static void add_file_action(gchar *extension, gchar *action) {
  fileaction *fa = (fileaction *)malloc(sizeof(fileaction));

  fa->extension = strdup(extension);
  fa->action = strdup(action);
  
  fa->next = fileactions;
  fileactions = fa;
}

/* Add a user command to the list. This must be done in order in order
   to avoid confusing the user, so we have to append to the end of the
   list. */
static void add_user_command(gchar *description, gchar *command) {
  usercommand *uc = (usercommand *)malloc(sizeof(usercommand)), *p;
  
  uc->description = strdup(description);
  uc->command = strdup(command);
  uc->next = NULL;

  if (!usercommands) {
    usercommands = uc;
  } else {
    for (p = usercommands; p->next; p = p->next) /* empty */;
    p->next = uc;
  }
}

/* Execute a configuration command. This may have uses other than
   reading the config file in the future (perhaps for remote
   control). */
void execute_command(gchar *command, gint pass) {
  gchar *cmd = strdup(command), *opta, *optb = NULL;
  configcommand *cc;

  if (!*cmd) return;
  if (*cmd == '#') return;

  opta = strchr(cmd, ':');
  if (opta) {
    *opta++ = '\0';
    optb = strchr(opta, ':');
    if (optb) *optb++ = '\0';
  }

  for (cc = configcommands; cc->type != CT_END; cc++) {
    if (strcmp(cmd, cc->command) != 0) continue;
    if (cc->pass != pass) continue;
    
    switch (cc->type) {
    case CT_INT: {
      if (!opta) {
	fprintf(stderr, "Command '%s' needs an argument.", cmd);
	goto out;
      }
      *(guint *)cc->target = atoi(opta);
      break;
    }
    case CT_STRING: {
      if (!opta) {
	fprintf(stderr, "Command '%s' needs an argument.", cmd);
	goto out;
      }
      *(gchar **)cc->target = strdup(opta);
      break;
    }
    case CT_FUNCTION1: {
      void (*func)(gchar *a) = cc->target;

      if (!opta) {
	fprintf(stderr, "Command '%s' needs an argument.", cmd);
	goto out;
      }
      func(opta);
      break;
    }
    case CT_FUNCTION2: {
      void (*func)(gchar *a, gchar *b) = cc->target;

      if (!opta || !optb) {
	fprintf(stderr, "Command '%s' needs two arguments.", cmd);
	goto out;
      }
      func(opta, optb);
      break;
    }
    case CT_END:
      /* This will not happen. */
      break;
    }
    
    /* And, since we've matched the command, there's no point
       in looking any further. */
    break;
  }
  
 out:
  free(cmd);
}

/* Read a config file, named in name. Pass 0 is before the GUI is
   shown, pass 1 is after. */
void read_config(gchar *name, gint pass) {
  FILE *f;
  gchar line[STRBUFSIZE];

  f = fopen(name, "r");
  if (!f) return;

  while (fgets(line, STRBUFSIZE, f)) {
    chomp(line);
    execute_command(line, pass);
  }
  
  fclose(f);
}

/* Read config files. */
void read_config_files(gint pass) {
  gchar rcname[STRBUFSIZE];

  read_config(SYSTEMXHIPPOCONFIG, pass);
  snprintf(rcname, STRBUFSIZE, "%s/.xhippo/config", getenv("HOME"));
  read_config(rcname, pass);
  read_config("xhippo.config", pass);
}

/* Get the directory in which the user wants to store playlists. */
void get_playlist_dir(gchar *buf, int buflen) {
  /* If attribute_playlist_dir is set, use that. */
  if (attribute_playlist_dir) {
    snprintf(buf, buflen, "%s", attribute_playlist_dir);
    return;
  }
  
  /* If $HOME/.xhippo/playlists is a directory, use that. */
  snprintf(buf, buflen, "%s/.xhippo/playlists/", getenv("HOME"));
  if (filetype(buf) == 2) return;

  /* Else use nothing. */
  strcpy(buf, "");
}

/* Save the window position. */
void save_window_state() {
  char filename[STRBUFSIZE];
  FILE *f;

  snprintf(filename, STRBUFSIZE, "%s/.xhippo/winstate", getenv("HOME"));
  if ((f = fopen(filename, "w"))) {
    fprintf(f, "%d\n%d\n%d\n%d\n%d\n", attribute_xpos, attribute_ypos, 
	    attribute_mini, attribute_width, attribute_height);
    fclose(f);
  } else {
    g_print(STRNOWINSTATE);
  }
}

/* Read the window position. */
void read_window_state() {
  char filename[STRBUFSIZE];
  char line[STRBUFSIZE];
  FILE *f;

  snprintf(filename, STRBUFSIZE, "%s/.xhippo/winstate", getenv("HOME"));
  if ((f = fopen(filename, "r"))) {
    fgets(line, STRBUFSIZE, f);
    attribute_xpos = atoi(line);
    fgets(line, STRBUFSIZE, f);
    attribute_ypos = atoi(line);
    fgets(line, STRBUFSIZE, f);
    attribute_mini = atoi(line);
    fgets(line, STRBUFSIZE, f);
    attribute_width = atoi(line);
    fgets(line, STRBUFSIZE, f);
    attribute_height = atoi(line);
    fclose(f);
  } /* else fail silently, it happens all the time */
}

