/*
 * Copyright (C) 2000 - 2001 Loic Dachary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <mysql.h>
#include <string.h>
#include <stdlib.h>

#include <getopttools.h>
#include <salloc.h>
#include <regex.h>                                                                                                                    
#include <server.h>
#include <sqlutil.h>

static int verbose = 0;

static char* url2server(server_t* params, const char* url);
static server_t* params_alloc();

typedef struct server_entry {
  /* Domain name */
  char* netloc;
  /* Regular expression with at least one capturing sequence. */
  char* regexp;
  /* regexp field in compiled form. */
  regex_t* compiled_regexp;
} server_entry_t;

static struct option long_options[] =
{
  /*- Verbose server code */
  {"verbose_server", 0, &verbose, 1},
  {0, 0, 0, SERVER_OPTIONS}
};

static struct option_help long_options_help[] =
{
  {"verbose_server","medium verbose server" },
  {"0", ""}
};

struct option* server_options(struct option [])
{
  return long_options;
}

struct option_help* server_help_options(struct option_help [])
{
  return long_options_help;
}

server_t* server_alloc(int argc, char** argv, struct option options[])
{
  server_t* params = params_alloc();

  opterr = 0;
  optind = 0;
  while(1) {
    /* `getopt_long' stores the option index here. */
    int option_index = 0;
    int c;

    c = getopt_long_only(argc, argv, "-", options, &option_index);

    /* Detect the end of the options. */
    if (c == -1)
      break;
     
    switch (c)
      {
      case 0:
	/* If this option set a flag, do nothing else now. */
	
	if (options[option_index].flag != 0)
	  break;
	if(!strcmp(options[option_index].name, "some_option")) {
	  break;
	}
	break;
      default:
	break;
      }
  }

  return params;
}

int server_hook(server_t* params, webbase_hook_params_t* hook)
{
  params->hook = hook;
  return 1;
}

int server_load_re(server_t* params)
{
  MYSQL_RES *res;
  MYSQL_ROW row;
  char* query = "select * from home_re";

  smysql_query(&params->base->mysql, query);
  res = smysql_store_result(&params->base->mysql);

  if(mysql_num_rows(res) > 0) {
    int entries_size = sizeof(server_entry_t) * mysql_num_rows(res);
    int index = 0;
    params->netloc2re_filled = 1;
    params->entries = (server_entry_t*)smalloc(entries_size);
    memset(params->entries, '\0', entries_size);
    while((row = mysql_fetch_row(res))) {
      server_entry_t *entry = &params->entries[index];
      char* netloc = row[0];
      char* re = row[1];
      if(!netloc || !re) {
	fprintf(stderr, "server_load_re: null netloc or re found\n");
	continue;
      }
      if(verbose) fprintf(stderr, "server_load_re: found %s %s\n", netloc, re);
      entry->compiled_regexp = (regex_t*)smalloc(sizeof(regex_t));
      memset(entry->compiled_regexp, '\0', sizeof(regex_t));
      {
	int ret;
	if((ret = regcomp(entry->compiled_regexp, re, REG_EXTENDED|REG_ICASE)) != 0) {
	  char buffer[256];
	  regerror(ret, entry->compiled_regexp, buffer, 256);
	  fprintf(stderr, "server_load_re: %s %s: regcomp failed: %s\n", netloc, re, buffer);
	  regfree(entry->compiled_regexp);
	  free(entry->compiled_regexp);
	  continue;
	}
      }
      entry->netloc = strdup(netloc);
      entry->regexp = strdup(re);

      if(!strcasecmp(netloc, "DEFAULT")) {
	params->default_re = entry;
      } else {
	hash_alloc_insert(params->netloc2re, strdup(netloc), entry);
      }
    }
  }

  mysql_free_result(res);
  return 1;
}

static char* url2server(server_t* params, const char* url)
{
  webbase_hook_params_t* hook = params->hook;
  
  if(hook && hook->url2server) {
    return hook->url2server(hook, url);
  } else {
    static uri_t* url_object = 0;
    int url_length = strlen(url);
    server_entry_t* entry = 0;
    char* server;

    if(!url_object) {
      url_object = uri_alloc((char*)url, url_length);
    } else {
      uri_realloc(url_object, (char*)url, url_length);
    }

    server = uri_netloc(url_object);

    if(params->netloc2re_filled) {
      hnode_t* node = hash_lookup(params->netloc2re, server);
      if(node) {
	entry = (server_entry_t*)hnode_get(node);
      }
    }
    if(!entry && params->default_re) {
      entry = params->default_re;
    }

    if(entry) {
      regex_t* regexp = entry->compiled_regexp;
      regmatch_t match[2];
      int result = regexec(regexp, url, 2, match, 0);

      switch(result) {
      case 0:
	if(match[1].rm_so >= 0) {
	  /*
	   * Return the first match.
	   */
	  int tmp_length = match[1].rm_eo - match[1].rm_so;
	  char* tmp = smalloc(tmp_length + 1);
	  memcpy(tmp, url + match[1].rm_so, tmp_length);
	  tmp[tmp_length] = '\0';
	  if(verbose) fprintf(stderr, "url2server: %s -> %s\n", url, tmp);

	  return tmp;
	} else {
	  fprintf(stderr, "url2server: regexec(%s,%s) did not capture any string, reverse to default\n", entry->regexp, url);
	}
	break;
      case REG_NOMATCH:
	break;
      default:
	{
	  char buffer[256];
	  regerror(result, entry->compiled_regexp, buffer, 256);
	  fprintf(stderr, "url2server: regexec(%s,%s) failed: %s\n", entry->regexp, url, buffer);
	}
	break;
      }
    }

    return strdup(server);
  }
}

int server_id(server_t* params, const char* url)
{
  webbase_t* base = params->base;
  static char* query = 0;
  static int query_size = 0;
  char* server = url2server(params, url);
  char rowid_string[WEBBASE_INTEGER_VALUE_SIZE + 1];
  int ret = 0;

  static_alloc(&query, &query_size, strlen(server) + 256);
  sprintf(query, "select rowid from servers where server = '%s'", server);
  if(sql_select_value(&base->mysql, query, rowid_string, WEBBASE_INTEGER_VALUE_SIZE)) {
    ret = atoi(rowid_string);
  } else {
    fprintf(stderr, "server_id: failed to retrieve record in servers for server %s\n", server);
  }

  free(server);
  return ret;
}

int server_add(server_t* params, const char* url)
{
  webbase_t* base = params->base;
  static char* query = 0;
  static int query_size = 0;
  char* server = url2server(params, url);
  char rowid_string[WEBBASE_INTEGER_VALUE_SIZE + 1];

  static_alloc(&query, &query_size, strlen(server) + 256);
  sprintf(query, "select rowid from servers where server = '%s'", server);
  if(sql_select_value(&base->mysql, query, rowid_string, WEBBASE_INTEGER_VALUE_SIZE)) {
    sprintf(query, "update servers set refcount = refcount + 1 where rowid = %s\n", rowid_string);
    smysql_query(&base->mysql, query);
    return atoi(rowid_string);
  } else {
    sprintf(query, "insert into servers (server, refcount) values ('%s', 1)", server);
    smysql_query(&base->mysql, query);
    return mysql_insert_id(&base->mysql);
  }
}

int server_remove(server_t* params, const char* url)
{
  webbase_t* base = params->base;
  static char* query = 0;
  static int query_size = 0;
  char* server = url2server(params, url);
  MYSQL_RES *res;
  MYSQL_ROW row;
  int ret = 1;
  
  static_alloc(&query, &query_size, strlen(server) + 256);
  sprintf(query, "select rowid,refcount from servers where server = '%s'", server);
  smysql_query(&base->mysql, query);
  res = smysql_store_result(&base->mysql);

  if(mysql_num_rows(res) > 0) {
    if((row = mysql_fetch_row(res))) {
      if(!row[0] || !row[1]) {
	fprintf(stderr, "server_remove: unexpectd null value for rowid or refcount at server %s\n", server);
	ret = 0;
      } else {
	int refcount = atoi(row[1]);
	if(refcount <= 1) {
	  sprintf(query, "delete from servers where rowid = %s", row[0]);
	} else {
	  sprintf(query, "update servers set refcount = refcount - 1 where rowid = %s", row[0]);
	}
	smysql_query(&base->mysql, query);
      }
    }
  } else {
    fprintf(stderr, "server_remove: failed to drop entry for server %s because no row found in servers table\n", server);
    ret = 0;
  }

  mysql_free_result(res);
  free(server);

  return ret;
}

void server_free(server_t* params)
{
  _K(hash_free)(params->netloc2re);
  if(params->entries_length > 0) {
    int i;
    for(i = 0; i < params->entries_length; i++) {
      server_entry_t* entry = &params->entries[i];
      /* Skip empty entries. */
      if(entry->netloc) {
	free(entry->netloc);
	free(entry->regexp);
	regfree(entry->compiled_regexp);
	free(entry->compiled_regexp);
      }
    }
    free(params->entries);
  }
}

static server_t* params_alloc()
{
  server_t* params = (server_t*)smalloc(sizeof(server_t));
  memset((char*)params, '\0', sizeof(server_t));
  params->netloc2re = hash_create(33, 0, 0);
  hash_set_allocator(params->netloc2re, 0, 0, 0);

  return params;
}
