/*
 *   Copyright (C) 1999 Loic Dachary
 *
 *   This program is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <WebbaseDl.h>

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif /* HAVE_MALLOC_H */
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#include <stdio.h>
#include <string.h>

#include <ltdl.h>

WebbaseDl* webbasedl_instance = 0;

WebbaseDl::WebbaseDl()
{
  name2function = hash_create(33, 0, 0);
}

WebbaseDl::~WebbaseDl()
{
  hash_free(name2function);
}

WebbaseDl* WebbaseDl::Instance()
{
  if(!webbasedl_instance)
    webbasedl_instance = new WebbaseDl();
  return webbasedl_instance;
}

struct option* WebbaseDl::Options()
{
  static struct option long_options[] = {
    {"dl", 1, 0, 0},
    {0, 0, 0, WEBBASE_DL_OPTIONS}
  };

  return long_options;
}

int WebbaseDl::HandleOption(struct option* option)
{
  int found = 1;
  
  if(!strcmp(option->name, "dl")) {
    Find(optarg);
  } else {
    found = 0;
  }

  return found;
}

webbasedl_init_function_t WebbaseDl::Find(const char* name)
{
  webbasedl_init_function_t function = Loaded(name);

  if(!function) {
    function = Load(name);
    if(function) {
      hash_alloc_insert(name2function, strdup(name), (void*)function);
    }
  }

  return function;
}

webbasedl_init_function_t WebbaseDl::Loaded(const char* name)
{
  hnode_t* node = hash_lookup(name2function, (void*)name);
  webbasedl_init_function_t function = 0;

  if(node)
    function = (webbasedl_init_function_t)hnode_get(node);

  return function;
}

webbasedl_init_function_t WebbaseDl::Load(const char* name)
{
  webbasedl_init_function_t init_function = 0;
  char* filename = 0;
  char* function = 0;

  if(!name) {
    fprintf(stderr, "WebbaseDl::Load: null name\n");
    goto err;
  }
  
  if(lt_dlinit() > 0) {
    fprintf(stderr, "WebbaseDl::Load: lt_dlinit: %s\n", lt_dlerror());
    goto err;
  }

  {
    filename = (char*)malloc(strlen(name) + 1 + 128);
    sprintf(filename, "lib%s.la", name);
    lt_dlhandle handle = lt_dlopen(filename);
    if(!handle) {
      fprintf(stderr, "WebbaseDl::Load: lt_dlhandle: %s %s\n", filename, lt_dlerror());
      goto err;
    }

    function = (char*)malloc(strlen(name) + 1 + 128);
    sprintf(function, "%s_init", name);
    lt_ptr_t symbol = lt_dlsym(handle, function);
    if(!symbol) {
      fprintf(stderr, "WebbaseDl::Load: lt_dlsym: %s\n", lt_dlerror());
      goto err;
    }

    init_function = (webbasedl_init_function_t)symbol;
  }
 err:
  if(function) free(function);
  if(filename) free(filename);
  return init_function;
}
