/*
 * Copyright (C) 2000, 2001  Lorenzo Bettini <bettini@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <iostream>
#include <fstream>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "fileutil.h"
#include "my_string.h"

using namespace std;

static void file_error(const string &error, const string &file);
static void internal_error(const string &error);

char *
read_file(const string &fileName)
{
  char *buffer = 0;
  long int char_count;

  // we open it as binary otherwise we may experience problems under
  // Windows system: when we fread, the number of char read can be
  // less then char_count, and thus we'd get an error...
  ifstream file (fileName.c_str (), ios_base::binary);
  if (! file.is_open () )
    file_error ("Error operning", fileName);
  else
    {
      // let's go to the end of the file...
      file.seekg (0, ios::end);
      if (! file)
        file_error ("Error positioning", fileName);

      // ...to read the dimension
      char_count = file.tellg ();
      if (! file)
        file_error ("Error reading position", fileName);

      buffer = new char [char_count + 1];
      if (! buffer)
        internal_error ("Memory allocation failed");
      
      file.seekg (0, ios::beg);
      if (! file)
        file_error ("Error positioning to start", fileName);
        
      //copy file into memory
      file.read (buffer, char_count);
      buffer[char_count] = '\0';

      file.close ();
    }

  return buffer;
}

void
file_error(const string &error, const string &file)
{
  cerr << PACKAGE << ": " << error << ", file " << file << endl;
  exit (1);
}

void
internal_error(const string &error)
{
  cerr << PACKAGE << ": Internal error: " << error << endl;
  exit (1);
}

// output file name = <outputDir> + input file name + ext
string
createOutputFileName(const string &inputFileName, char *outputDir, const string &ext )
{
  string input_file_name;
  char path_separator = '/';

  if (! outputDir)
    input_file_name = inputFileName;
  else
    {
      string::size_type pos_of_sep;
      pos_of_sep = inputFileName.find_last_of('/');
      
      if (pos_of_sep == string::npos) // try with DOS separator
        {
  	  pos_of_sep = inputFileName.find_last_of ('\\');
          if (pos_of_sep != string::npos)
            path_separator = '\\';
        }


      if (pos_of_sep != string::npos)
        input_file_name = inputFileName.substr (pos_of_sep + 1);
      else
        input_file_name = inputFileName;
    }

  string outputFileName;

  if (outputDir)
    {
      outputFileName += outputDir;
      outputFileName += path_separator;
    }
  outputFileName += input_file_name;
  outputFileName += ext;

  return outputFileName;
}

unsigned int
get_line_count(const string &file_name)
{
  ifstream input (file_name.c_str ());

  if (! input)
    file_error ("Error opening", file_name);

  unsigned int count = get_line_count (input);

  input.close ();

  return count;
}

unsigned int
get_line_count(istream &input)
{
  unsigned int count = 0;
  string line;

  while (true)
    {
      getline (input, line);
      if (! input.eof ())
        ++count;
      else
        break;
    }

  return count;
}

string
get_file_extension(const string &s)
{
  string::size_type pos_of_sep;

  pos_of_sep = s.rfind(".");

  if (pos_of_sep == string::npos)
    return ""; // no extension

  return s.substr (pos_of_sep + 1);
}
