/* Source Installer, Copyright (c) 2005 Claudio Fontana

   srcregex.c - simple wrapper to the POSIX extended regex

   This program is free software; you can redistribute it and/or modify
       it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
       (at your option) any later version.

   This program is distributed in the hope that it will be useful,
       but WITHOUT ANY WARRANTY; without even the implied warranty of
       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
       GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
       along with this program (look for the file called COPYING);
       if not, write to the Free Software Foundation, Inc.,
           51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

       You can contact the author (Claudio Fontana) by sending a mail
       to claudio@gnu.org
*/

#include "src_stdinc.h"

struct srcinst_matches {
  int count;
  regmatch_t* matches;
  char* string;
};

struct srcinst_regex {
  regex_t preg;
};

SRCINST_MATCH srcinst_init_match(int nmatch) {
  struct srcinst_matches* m;
  m = srcinst_malloc(sizeof(struct srcinst_matches));
  m->string = 0;
  m->count = nmatch;
  m->matches = nmatch == 0 ? 0 : srcinst_malloc(nmatch * sizeof(regmatch_t));
  return m;
}

void srcinst_resize_match(SRCINST_MATCH m, int nmatch) {
  /* string remains the same */
  struct srcinst_matches* pmatch;
  pmatch = m;
  pmatch->count = nmatch;

  if (nmatch == 0) {
    if (pmatch->matches) { free(pmatch->matches); pmatch->matches = 0; }

  } else {
    pmatch->matches = srcinst_realloc(pmatch->matches, nmatch);
  }
}

void srcinst_free_match(SRCINST_MATCH m) {
  struct srcinst_matches* pmatch;
  pmatch = m;

  if (pmatch->matches)
    { free(pmatch->matches); }

  free(m);
}

char* srcinst_get_match(SRCINST_MATCH pmatch, int i) {
  char* p; regmatch_t* this; struct srcinst_matches* m;
  size_t len;
  m = pmatch; 

  if (i < 0 || i >= m->count)
    return 0;

  this = m->matches + i;

  if (this->rm_so == -1)
    return 0;

  p = srcinst_malloc((len = (this->rm_eo - this->rm_so)) + 1);
  memcpy(p, m->string + this->rm_so, len);
  p[len] = 0;
  return p;
}

int srcinst_get_match_len(SRCINST_MATCH pmatch, int i) {
  struct srcinst_matches* m; regmatch_t* this;
  m = pmatch;
  
  if (i < 0 || i >= m->count)
    return -1;

  this = m->matches + i;

  if (this->rm_so == -1)
    return -1;

  return this->rm_eo - this->rm_so;
}

int srcinst_get_match_offset(SRCINST_MATCH pmatch, int i) {
  struct srcinst_matches* m; regmatch_t* this;
  m = pmatch;
  
  if (i < 0 || i >= m->count)
    return -1;

  this = m->matches + i;

  if (this->rm_so == -1)
    return -1;

  return this->rm_so;
}

SRCINST_REGEX srcinst_init_regex(char* regex, int cflags) {
  char errbuf[SRCINST_BUFSIZE]; struct srcinst_regex* preg; int err;
  preg = srcinst_malloc(sizeof(struct srcinst_regex));
  
  err = regcomp(&(preg->preg), regex, cflags == -1 ? REG_EXTENDED : cflags);

  if (err != 0) {
    regerror(err, &(preg->preg), errbuf, sizeof(errbuf));
    srcinst_error(SRCINST_ERR_REGEX, errbuf, regex);
  }
  return preg;
}

int srcinst_exec_regex(SRCINST_REGEX reg, char* string, SRCINST_MATCH m) {
  struct srcinst_regex* preg; struct srcinst_matches* matches;
  preg = reg; matches = m;

  if (matches) 
    matches->string = string;

  return !regexec(&(preg->preg), string, matches ? matches->count : 0,
		  matches ? matches->matches : 0, 0);
}

void srcinst_free_regex(SRCINST_REGEX reg) {
  struct srcinst_regex* preg;
  preg = reg;
  regfree(&(preg->preg));
  free(reg);
}

