"======================================================================
|
|   SUnit testing framework - implementation
|
|   This file is in the public domain.
|
 ======================================================================"

Object subclass: #TestResource
	instanceVariableNames: 'name description '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SUnit'!

TestResource class
	instanceVariableNames: 'current '!

Object subclass: #TestSuite
	instanceVariableNames: 'tests resources name '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SUnit'!

TestSuite class
	instanceVariableNames: ''!

Object subclass: #TestCase
	instanceVariableNames: 'testSelector '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SUnit'!

TestCase class
	instanceVariableNames: ''!

Object subclass: #TestResult
	instanceVariableNames: 'failures errors passed '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SUnit'!

TestResult class
	instanceVariableNames: ''!

!TestResource methodsFor: 'Accessing'!

description
	description isNil ifTrue: [^''].
	^description!

description: aString
	description := aString!

name
	name isNil ifTrue: [^self printString].
	^name!

name: aString
	name := aString!

resources
	^self resources! !

!TestResource methodsFor: 'Init / Release'!

initialize
	self setUp! !

!TestResource methodsFor: 'Testing'!

isAvailable
	"override to provide information on the readiness of the resource"
	^true!

isUnavailable
	"override to provide information on the readiness of the resource"
	^self isAvailable not! !

!TestResource methodsFor: 'Printing'!

printOn: aStream
	aStream nextPutAll: self class printString! !

!TestResource methodsFor: 'Running'!

setUp
	"Does nothing. Subclasses should override this
	to initialize their resource"!

signalInitializationError
        ^self class signalInitializationError!

tearDown
	"Does nothing. Subclasses should override this
	to tear down their resource"! !

!TestResource class methodsFor: 'Accessing'!

current
	current isNil ifTrue: [current := self new].
	^current!

current: aTestResource
	current := aTestResource!

signalInitializationError
        ^TestResult signalErrorWith: 'Resource ' , self name , ' could not be initialized'! !

!TestResource class methodsFor: 'Testing'!

resources
        ^#()!

isAbstract
	"Override to true if a TestCase subclass is Abstract and should not have
	TestCase instances built from it"
	^self name = #TestResource!

isAvailable
	^self current notNil!

isUnavailable
	^self isAvailable not! !

!TestResource class methodsFor: 'Creation'!

new
	^super new initialize!

reset
	current notNil ifTrue: 
		[current tearDown.
		current := nil]! !

!TestResource methodsFor: 'Printing'!

printOn: aStream
        aStream nextPutAll: self class printString! !

!TestSuite methodsFor: 'Dependencies'!

addDependentToHierachy: anObject
	self sunitAddDependent: anObject.
	self tests do: [ :each | each addDependentToHierachy: anObject]!

removeDependentFromHierachy: anObject
	self sunitRemoveDependent: anObject.
	self tests do: [ :each | each removeDependentFromHierachy: anObject]! !

!TestSuite methodsFor: 'Accessing'!

addTest: aTest
	self tests add: aTest!

addTests: aCollection 
	aCollection do: [:eachTest | self addTest: eachTest]!

defaultResources
	^self tests 
		inject: Set new
		into: 
			[:coll :testCase | 
			coll
				addAll: testCase resources;
				yourself]!

name
	^name!

name: aString
	name := aString!

resources
	resources isNil ifTrue: [resources := self defaultResources].
	^resources!

resources: anObject
	resources := anObject!

tests
	tests isNil ifTrue: [tests := OrderedCollection new].
	^tests! !

!TestSuite methodsFor: 'Testing'!

areAllResourcesAvailable
	^self resources 
		inject: true
		into: [:total :each | each isAvailable & total]! !

!TestSuite methodsFor: 'Running'!

run
	| result |
	result := TestResult new.
	self areAllResourcesAvailable ifFalse: [^TestResult signalErrorWith: 'Resource could not be initialized'].
	[self run: result] sunitEnsure: [self resources do: [:each | each reset]].
	^result!

run: aResult 
	self tests do: [:each |  
        	each isLogging ifTrue: [
			each failureLog print: each; space
		].
		self sunitChanged: each.
		each run: aResult.
        	each isLogging ifTrue: [
			each failureLog nl
		]
	]! !

!TestSuite class methodsFor: 'Creation'!

named: aString

	^self new
		name: aString;
		yourself! !

!TestCase methodsFor: 'Dependencies'!

addDependentToHierachy: anObject 
	"an empty method. for Composite compability with TestSuite"!

removeDependentFromHierachy: anObject 
	"an empty method. for Composite compability with TestSuite"! !

!TestCase methodsFor: 'Testing'!

areAllResourcesAvailable
	^self resources 
		inject: true
		into: [:total :each | each isAvailable & total]! !

!TestCase methodsFor: 'Accessing'!

assert: aBoolean
	aBoolean
	    ifTrue: [
		self isLogging ifTrue: [
		    self failureLog show: $. ]
	    ]
	    ifFalse: [ self signalFailure: 'Assertion failed']!

deny: aBoolean
	self assert: aBoolean not!

deny: aBoolean description: aString
        self assert: aBoolean not description: aString!

deny: aBoolean description: aString resumable: resumableBoolean
        self
                assert: aBoolean not
                description: aString
                resumable: resumableBoolean!

shouldnt: aBlock raise: anExceptionalEvent description: aString
        ^self assert: (self executeShould: aBlock inScopeOf: anExceptionalEvent) not
	      description: aString!

should: aBlock description: aString
        self assert: aBlock value description: aString!

assert: aBoolean description: aString resumable: resumableBoolean
        | exception |
        aBoolean
                ifFalse:
                        [self logFailure: aString.
                        exception := resumableBoolean
                                       ifTrue: [TestResult resumableFailure]
                                       ifFalse: [TestResult failure].
                        exception sunitSignalWith: aString]!

assert: aBoolean description: aString
        aBoolean ifFalse: [
                self logFailure: aString.
                TestResult failure sunitSignalWith: aString]!

signalFailure: aString
        TestResult failure sunitSignalWith: aString!

isLogging
        "By default, we're logging failures to the Transcript"
        ^true!

logFailure: aString
        self isLogging ifTrue: [
                self failureLog
                        nl;
                        nextPutAll: aString;
                        flush]!

failureLog
        "dialect-specific"

        ^Transcript!

resources
        | allResources resourceQueue |
        allResources := Set new.
        resourceQueue := OrderedCollection new.
        resourceQueue addAll: self class resources.
        [resourceQueue isEmpty] whileFalse: [
                | next |
                next := resourceQueue removeFirst.
                allResources add: next.
                resourceQueue addAll: next resources].
        ^allResources!

selector
	^testSelector!

should: aBlock
	self assert: aBlock value!

should: aBlock raise: anExceptionalEvent 
	^self assert: (self executeShould: aBlock inScopeOf: anExceptionalEvent)!

shouldnt: aBlock
	self deny: aBlock value!

shouldnt: aBlock raise: anExceptionalEvent 
	^self assert: (self executeShould: aBlock inScopeOf: anExceptionalEvent) not!

shouldnt: aBlock description: aString
        self deny: aBlock value description: aString!

should: aBlock raise: anExceptionalEvent description: aString
        ^self assert: (self executeShould: aBlock inScopeOf: anExceptionalEvent)                description: aString!

signalFailure: aString
	TestResult failure sunitSignalWith: aString! !

!TestCase methodsFor: 'Running'!

debug
        self resources do: [:res |
                res isAvailable ifFalse: [^res signalInitializationError]].
        [(self class selector: testSelector) runCase]
                sunitEnsure: [self resources do: [:each | each reset]]!

debugAsFailure
        | semaphore |
        semaphore := Semaphore new.
        self resources do: [:res |
                res isAvailable ifFalse: [^res signalInitializationError]].
        [semaphore wait. self resources do: [:each | each reset]] fork.
        (self class selector: testSelector) runCaseAsFailure: semaphore.!

openDebuggerOnFailingTestMethod
	"SUnit has halted one step in front of the failing test method. Step over the 'self halt' and 
	 send into 'self perform: testSelector' to see the failure from the beginning"
	self
		halt;
		performTest!

run
	| result |
	result := TestResult new.
	self run: result.
	^result!

run: aResult
	aResult runCase: self!

runCase
	self setUp.
	[self performTest] sunitEnsure: [self tearDown]!

runCaseAsFailure: aSemaphore
	[self setUp.
	self openDebuggerOnFailingTestMethod] 
		     sunitEnsure: [aSemaphore signal. self tearDown]!

setUp!

tearDown! !

!TestCase methodsFor: 'Private'!

executeShould: aBlock inScopeOf: anExceptionalEvent 
	^[[aBlock value. false]
		sunitOn: anExceptionalEvent
		do: [:ex | ^true]]
			sunitOn: TestResult error
			do: [:ex | ^false]!

performTest
	self perform: testSelector sunitAsSymbol!

setTestSelector: aSymbol
	testSelector := aSymbol! !

!TestCase methodsFor: 'Printing'!

printOn: aStream
	aStream
		nextPutAll: self class printString;
		nextPutAll: '>>';
		nextPutAll: testSelector! !

!TestCase class methodsFor: 'Accessing'!

allTestSelectors
	^self sunitAllSelectors select: [:each | 'test*' sunitMatch: each]!

resources
	^#()!

testSelectors
	^self sunitSelectors select: [:each | 'test*' sunitMatch: each]! !

!TestCase class methodsFor: 'Building Suites'!

buildSuite

	| suite |
	^self isAbstract 
		ifTrue: 
			[suite := TestSuite new.
			suite name: self name asString.
			self allSubclasses do: [:each | each isAbstract ifFalse: [suite addTest: each buildSuiteFromSelectors]].
			suite]
		ifFalse: [self buildSuiteFromSelectors]!

buildSuiteFromAllSelectors
	^self buildSuiteFromMethods: self allTestSelectors!

buildSuiteFromLocalSelectors
	^self buildSuiteFromMethods: self testSelectors!

buildSuiteFromMethods: testMethods 
	^testMethods 
		inject: ((TestSuite new)
				name: self name asString;
				yourself)
		into: 
			[:suite :selector | 
			suite
				addTest: (self selector: selector);
				yourself]!

buildSuiteFromSelectors
	^self shouldInheritSelectors
		ifTrue: [self buildSuiteFromAllSelectors]
		ifFalse: [self buildSuiteFromLocalSelectors]! !

!TestCase class methodsFor: 'Instance Creation'!

debugAsFailure: aSymbol
	^(self selector: aSymbol) debugAsFailure!

debug: aSymbol
	^(self selector: aSymbol) debug!

run: aSymbol
	^(self selector: aSymbol) run!

selector: aSymbol
	^self new setTestSelector: aSymbol!

suite
	^self buildSuite! !

!TestCase class methodsFor: 'Testing'!

isAbstract
	"Override to true if a TestCase subclass is Abstract and should not have
	TestCase instances built from it"
	^self name = #TestCase.!

shouldInheritSelectors
	"answer true to inherit selectors from superclasses"

	^true! !

!TestResult methodsFor: 'Accessing'!

correctCount
	"depreciated - use #passedCount"
	^self passedCount!

defects
	^self errors copy
	     addAll: self failures;
	     yourself!

errorCount
	^self errors size!

errors
	errors isNil ifTrue: [errors := Set new].
	^errors!

failureCount
	^self failures size!

failures
	failures isNil ifTrue: [failures := Set new].
	^failures!

passed
	passed isNil ifTrue: [passed := Set new].
	^passed!

passedCount
	^self passed size!

runCount
	^self passedCount + self failureCount + self errorCount!

tests
	^(OrderedCollection new: self runCount)
		addAll: self passed;
		addAll: self errors;
		addAll: self defects;
		yourself! !

!TestResult methodsFor: 'Testing'!

hasErrors

	^self errors size > 0!

hasFailures

	^self failures size > 0!

hasPassed

	^self hasErrors not and: [self hasFailures not]!

isError: aTestCase

	^self errors includes: aTestCase!

isFailure: aTestCase
	^self failures includes: aTestCase!

isPassed: aTestCase

	^self passed includes: aTestCase! !

!TestResult methodsFor: 'Init / Release'!

initialize
! !

!TestResult methodsFor: 'Printing'!

printOn: aStream
	aStream
		nextPutAll: self runCount printString;
		nextPutAll: ' run, ';
		nextPutAll: self passedCount printString;
		nextPutAll: ' passed, ';
		nextPutAll: self failureCount printString;
		nextPutAll: ' failed, ';
		nextPutAll: self errorCount printString;
		nextPutAll:' error'.
	self errorCount ~= 1
		ifTrue: [aStream nextPut: $s].! !

!TestResult methodsFor: 'Running'!

runCase: aTestCase
	| testCasePassed |
	testCasePassed := [ [aTestCase runCase.  true] 
		sunitOn: self class failure do: [:signal | 
        		aTestCase isLogging ifTrue: [ aTestCase failureLog show: $f ].
			self failures add: aTestCase.
			signal sunitExitWith: false]]
				sunitOn: self class error do: [:signal |
        				aTestCase isLogging ifTrue: [
						thisContext backtraceOn: aTestCase failureLog.
						aTestCase failureLog show: $e.
					].
					self errors add: aTestCase.
					signal sunitExitWith: false].

	testCasePassed ifTrue: [self passed add: aTestCase]! !

!TestResult class methodsFor: 'Exceptions'!

error
	^self exError!

exError
	"Change for Dialect"
	^Error!

failure
	^TestFailure!

resumableFailure
	^ResumableTestFailure!

signalErrorWith: aString 
	self error sunitSignalWith: aString!

signalFailureWith: aString 
	self failure sunitSignalWith: aString! !

!TestResult class methodsFor: 'Init / Release'!

new
	^super new initialize! !

