//
// Part of the ht://Dig package   <http://www.htdig.org/>
// Copyright (c) 1999, 2000 The ht://Dig Group
// For copyright details, see the file COPYING in your distribution
// or the GNU General Public License version 2 or later
// <http://www.gnu.org/copyleft/gpl.html>
//
// $Id: WordSearch.cc,v 1.1 2000/10/24 14:23:29 loic Exp $
//
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include <WordSearch.h>

WordSearch::WordSearch(WordList* nwords)
{
  //
  // Internal
  //
  words = nwords;
  verbose = 0;

  //
  // Input/Output
  //
  limit_bottom = 0;

  //
  // Input
  //
  limit_count = 0;
  context_in.trunc();
  expr = 0;
  bounded = 0;

  //
  // Output
  //
  matches = 0;
  matches_size = 0;
  matches_length = 0;
  context_out.trunc();
}

WordMatch **WordSearch::Search()
{
  int ret = 0;

  if(verbose) fprintf(stderr, "WordSearch::Search: non optimized expression %s\n", (char*)expr->Get());
  if(expr->Optimize() != OK)
    return 0;
  if(verbose) fprintf(stderr, "WordSearch::Search: optimized expression %s\n", (char*)expr->Get());
  
  if(expr->Count(matches_total) != OK) return OK;
  
  //
  // Build space for results
  //
  matches_size = limit_count + 1;
  matches = new WordMatch* [matches_size];
  memset((char*)matches, '\0', sizeof(WordMatch*) * matches_size);
  matches_length = 0;

  //
  // Move to first possible position. 
  //
  if(expr->WalkInit() != OK)
    goto end;

  if(expr->ContextRestore(context_in) == NOTOK)
    goto end;
  ret = bounded ? SearchLoopBounded(expr) : SearchLoop(expr);
  //
  // Don't bother saving the context if at end of 
  // search (WORD_WALK_ATEND) or error (NOTOK)
  //
  if(ret == OK && expr->ContextSave(context_out) == NOTOK)
    goto end;

end:
  expr->WalkFinish();

  if(ret == NOTOK || matches_length <= 0) {
    unsigned int i;
    for(i = 0; i < matches_length; i++)
      delete matches[i];
    delete [] matches;
    matches = 0;
  }

  return matches;
}

int WordSearch::SearchLoop(WordTree *expr)
{
  int ret = OK;
  unsigned int i;
  //
  // Skip the first <limit_bottom> documents
  //
  {
    for(i = 0; i < limit_bottom; i++) {
      if((ret = expr->WalkNext()) != OK)
	return ret;
    }
  }
  //
  // Get documents up to <limit_count> or exhaustion
  //
  for(matches_length = 0; matches_length < limit_count; matches_length++) {
    if((ret = expr->WalkNext()) != OK) {
      break;
    } else {
      WordMatch* match = new WordMatch(words->GetContext());
      match->match = expr->GetDocument();
      if(expr->IsA() != WORD_TREE_LITERAL)
	match->info = ((WordTreeOperand*)expr)->GetInfo();
      matches[matches_length] = match;
      if(verbose) fprintf(stderr, "WordSearch::Search: match %s\n", (char*)matches[matches_length]->match.Get());
    }
  }

  return ret;
}

int WordSearch::SearchLoopBounded(WordTree *expr)
{
  int ret = OK;
  unsigned int i;
  
  for(i = 0; i < limit_count; i++) {
    WordMatch* match = new WordMatch(words->GetContext());
    matches[i] = match;
  }
  
  //
  // Get documents up to <limit_count> or exhaustion
  //
  for(i = 0; i < limit_bottom + limit_count; i++) {
    if((ret = expr->WalkNext()) != OK) {
      break;
    } else {
      WordMatch* match = matches[i % limit_count];
      match->match = expr->GetDocument();
      if(expr->IsA() != WORD_TREE_LITERAL)
	match->info = ((WordTreeOperand*)expr)->GetInfo();
    }
  }

  limit_bottom = i == 0 ? 0 : (((i - 1) / limit_count) * limit_count);
  matches_length = i == 0 ? 0 : (((i - 1) % limit_count) + 1);

  for(unsigned int invalid = matches_length; invalid < limit_count; invalid++) {
    delete matches[invalid];
    matches[invalid] = 0;
  }
  
  return ret;
}
