/* print.c */

/*----------------------------------------------------------------------
-- This file is a part of the MARST package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <ctype.h>
#include "algol.h"

/*--------------------------------------------------------------------*/

static void outputf(char *fmt, ...)
{     /* formats and prints data to standard output channel */
      va_list arg;
      char str[255+1], *ptr;
      va_start(arg, fmt);
      vsprintf(str, fmt, arg);
      assert(strlen(str) <= sizeof(str) - 1);
      va_end(arg);
      for (ptr = str; *ptr != '\0'; ptr++) outchar(1, *ptr);
      return;
}

/*--------------------------------------------------------------------*/

static void outarray(char *name, struct dv *dv, int type)
{     /* prints entire array */
      int n = dv->n, i[9], k;
      void *base = dv->base;
      assert(1 <= n && n <= 9);
      /* initialize subscript list */
      for (k = 0; k < n; k++) i[k] = dv->d[k].lo;
      /* print individual array elements */
      for (;;)
      {  /* print array name */
         outputf("%s", name);
         for (k = 0; k < n; k++)
            outputf("%c%d", k == 0 ? '[' : ',', i[k]);
         outputf("] = ");
         /* print element value */
         switch (type)
         {  case 'r':
               outputf(REAL_FMT "\n", *(double *)base);
               base = ((double *)base) + 1;
               break;
            case 'i':
               outputf("%d\n", *(int *)base);
               base = ((int *)base) + 1;
               break;
            case 'b':
               outputf("%s\n", *(bool *)base ? "true" : "false");
               base = ((bool *)base) + 1;
               break;
            default:
               assert(type != type);
         }
         /* compute subscripts of the next element (row-wise) */
         for (k = n-1; k >= 0; k--)
         {  i[k]++;
            if (i[k] <= dv->d[k].up) break;
            i[k] = dv->d[k].lo;
         }
         if (k < 0) break;
      }
      return;
}

/*--------------------------------------------------------------------*/

#define F_REAL    0x0001
#define F_INT     0x0002
#define F_BOOL    0x0004
#define F_LABEL   0x0008
#define F_ARRAY   0x0010
#define F_STRING  0x0080

void print(int n, ...)
{     /* pseudo procedure print */
      /* each object to be printed is represented in variable parameter
         list as triple (flags, name, ptr) where flags is object flags
         as generated by the translator, name is the object identifier
         (if applicable), and ptr is a pointer to value of object or to
         array dope vector */
      va_list arg;
      int k;
      va_start(arg, n);
      for (k = 1; k <= n; k++)
      {  int flags  = va_arg(arg, int);
         char *name = va_arg(arg, char *);
         switch (flags)
         {  case F_REAL:
               {  /* real quantity */
                  double val = va_arg(arg, double);
                  if (name != NULL) outputf("%s = ", name);
                  outputf(REAL_FMT "\n", val);
               }
               break;
            case F_INT:
               {  /* integer quantity */
                  int val = va_arg(arg, int);
                  if (name != NULL) outputf("%s = ", name);
                  outputf("%d\n", val);
               }
               break;
            case F_BOOL:
               {  /* Boolean quantity */
                  bool val = va_arg(arg, bool);
                  if (name != NULL) outputf("%s = ", name);
                  outputf("%s\n", val ? "true" : "false");
               }
               break;
            case F_LABEL:
               {  /* value of label */
                  struct label val = va_arg(arg, struct label);
                  if (name != NULL) outputf("%s = ", name);
                  outputf("(%p,%d)\n", val.jump, val.kase);
               }
               break;
            case F_ARRAY:
               {  /* array */
                  struct arg val = va_arg(arg, struct arg);
                  struct dv *dv = val.arg1;
                  int type = (int)val.arg2;
                  assert(name != NULL);
                  if (dv->base == NULL)
                     outputf("array `%s' has no elements\n", name);
                  else
                     outarray(name, dv, type);
               }
               break;
            case F_STRING:
               {  /* string */
                  struct arg val = va_arg(arg, struct arg);
                  char *str = val.arg1, *ptr;
                  if (name != NULL) outputf("%s = ", name);
                  outputf("\"");
                  for (ptr = str; *ptr != '\0'; ptr++)
                  {  if (iscntrl(*ptr))
                        outputf("\\x%02X", *ptr);
                     else if (*ptr == '"')
                        outputf("\\\"");
                     else
                        outputf("%c", *ptr);
                  }
                  outputf("\"\n");
               }
               break;
            default:
               /* it is no sense to print objects of the other kinds,
                  because they can't posses values */
               assert(flags != flags);
         }
      }
      va_end(arg);
      return;
}

/* eof */
