/* 
   XGColor.m

   NSColor for GNUstep GUI X/GPS Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: January 1996
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998
   
   This file is part of the GNUstep GUI X/GPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <stdlib.h>

#include <gnustep/xgps/XGContext.h>
#include <wraster/wraster.h>

//
// Backend structure for XGColor
//
typedef struct _XGColor_struct
{
  XColor color;
  BOOL is_alloced;
} XGColor_struct;

#define XRCOLOR (((XGColor_struct *)be_color_reserved)->color)
#define XRALLOC (((XGColor_struct *)be_color_reserved)->is_alloced)

static XGColor *XRLightGrayColor = nil;
static XGColor *XRBlackColor = nil;
static XGColor *XRWhiteColor = nil;
static XGColor *XRDarkGrayColor = nil;

NSColor * XRLightGray(void)                     { return XRLightGrayColor; }
NSColor * XRBlack(void)                         { return XRBlackColor; }
NSColor * XRWhite(void)                         { return XRWhiteColor; }
NSColor * XRDarkGray(void)                      { return XRDarkGrayColor; }

@implementation XGColor

- (XColor) xColor
{
  if (!XRALLOC)
    {
      if (be_color_reserved)  				// allocate a color 
	{
	  RColor rcolor; 
	  Status ret;
	  NSColor *c = self;
	  RContext *rcontext = [(XGContext *)[XGContext currentContext] 
	  				xrContext];

	  if (([[c colorSpaceName] isEqual: NSCalibratedRGBColorSpace])
	    || ([[c colorSpaceName] isEqual: NSDeviceRGBColorSpace]))
	    {
	      rcolor.red = 255 * [c redComponent];
	      rcolor.green = 255 * [c greenComponent];
	      rcolor.blue = 255 * [c blueComponent];
	    }
	  else if (([[c colorSpaceName] isEqual: NSCalibratedWhiteColorSpace])
	    || ([[c colorSpaceName] isEqual: NSDeviceWhiteColorSpace]))
	    {
	      rcolor.red = rcolor.green = rcolor.blue = 
		255 * [c whiteComponent];
	    }
	  else
	    {
	      rcolor.red = rcolor.green = rcolor.blue = 120;
	    }

	  ret = RGetClosestXColor(rcontext, &rcolor, &(XRCOLOR));
	  if (ret == False)
	    NSLog(@"Failed to alloc color (%d,%d,%d)\n",
		  (int)rcolor.red, (int)rcolor.green, (int)rcolor.blue);
	  else
	    {
	      RGB_component.red = ((float)XRCOLOR.red) / 65535;
	      RGB_component.green = ((float)XRCOLOR.green) / 65535;
	      RGB_component.blue = ((float)XRCOLOR.blue) / 65535;
	      XRALLOC = YES;
	    }
	}
      else
	[NSException raise: NSInternalInconsistencyException
		    format: @"XGColor without malloced space"];
    }

  return XRCOLOR;
}

- (void) setXColor: (XColor)xcolor
{
  XRCOLOR = xcolor;
}

//
// Class methods
//
+ (void) initialize
{
  if (self == [XGColor class])
    {
      [self setVersion: 1];
      [self poseAsClass: [NSColor class]];
      XRWhiteColor = [[NSColor whiteColor] retain];
      XRBlackColor = [[NSColor blackColor] retain];
      XRDarkGrayColor = [[NSColor darkGrayColor] retain];
      XRLightGrayColor = [[NSColor lightGrayColor] retain];
    }
}

+ (NSColor *) colorWithCalibratedWhite: (float)white 
				 alpha: (float)alpha
{
  if (alpha == 1.0)
    {
      if (white == NSLightGray)
	{
	  return XRLightGray();
	}
      else if (white == NSBlack)
	{
	  return XRBlack();
	}
      else if (white == NSWhite)					 
	{
	  return XRWhite();
	}
      else if (white == NSDarkGray)					 
	{
	  return XRDarkGray();
	}
    }

  return [super colorWithCalibratedWhite: white alpha: alpha];
}

+ (NSColor *) darkGrayColor
{
  NSColor *cachedColor = nil;

  if ((cachedColor = XRDarkGray()))
    return cachedColor;

  return [super colorWithCalibratedWhite: NSDarkGray alpha: 1.0];
}

+ (NSColor *) lightGrayColor
{
  NSColor *cachedColor = nil;

  if ((cachedColor = XRLightGray()))
    return cachedColor;

  return [super colorWithCalibratedWhite: NSLightGray alpha: 1.0];
}

+ (NSColor *) blackColor
{
  NSColor *cachedColor = nil;

  if ((cachedColor = XRBlack()))
    return cachedColor;

  return [super colorWithCalibratedWhite: NSBlack alpha: 1.0];
}

+ (NSColor *) whiteColor
{
  NSColor *cachedColor = nil;

  if ((cachedColor = XRWhite()))
    return cachedColor;

  return [super colorWithCalibratedWhite: NSWhite alpha: 1.0];
}

- (id) init
{
  be_color_reserved = malloc(sizeof(XGColor_struct));
  XRALLOC = NO;
	
  [super init];
	
  return self;
}

- (id) initWithCoder: (NSCoder*)aCoder
{
  [super initWithCoder: aCoder];
  be_color_reserved = malloc(sizeof(XGColor_struct));
  XRALLOC = NO;
  return self;
}

- (void) dealloc
{
  free(be_color_reserved);

  [super dealloc];
}

- (void) set
{
  if (self == XRLightGrayColor)
    PSsetgray(0.667);
  else if (self == XRDarkGrayColor)
    PSsetgray(0.333);
  else if (self == XRBlackColor)
    PSsetgray(0);
  else if (self == XRWhiteColor)
    PSsetgray(1.0);
  else
    {
      XColor fColor = [self xColor];
      PSsetrgbcolor((double)fColor.red/65536, (double)fColor.green/65536,
                    (double)fColor.blue/65536);
    }
}

- (Class) classForCoder: (NSCoder*)aCoder
{
  if ([self class] == [XGColor class])
    return [super class];
  return [self class];
}

@end
