/* 
   Browser.m

   NSBrowser and delegate for GNUstep Workspace

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Scott Christley <scottc@net-community.com>
   Date: March 1996
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: July 1998
   
   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#import <Foundation/NSFileManager.h>

#include <gnustep/xraw/XR.h>

#include "Browser.h"
#include "BrowserMatrix.h"

#define NUM_OF_COL 6				// number of columns displayed in browser


//*****************************************************************************
//
// 		Browser 
//
//*****************************************************************************

@implementation Browser

//
// Class methods
//
+ (void)initialize
{
	if (self == [Browser class])
		{
		NSDebugLog(@"Initialize xbrowser class\n");
		[self setVersion:1];								// Initial version
		}
}

//
// Instance methods
//
- (void)_performLoadOfColumn:(int)column
{													// use our custom matrix
	_browserMatrixClass = [BrowserMatrix class];	// class in Workspace
	[super _performLoadOfColumn:column];
}

- (void)_adjustMatrixOfColumn:(int)column
{
id bc;
NSScrollView *sc;
id matrix;
NSSize ms;

	if (column >= (int)[_browserColumns count])
		return;
	
	bc = [_browserColumns objectAtIndex: column];
	sc = [bc columnScrollView];
	matrix = [bc columnMatrix];

	if (sc && matrix && [bc isLoaded])			// adjust the matrix but only
		{										// if column has been loaded
		ms = [matrix cellSize];
		ms.width = 80;
		ms.height = 70;
		[matrix setCellSize: ms];
	
		[sc setDocumentView: matrix];
		}
}

//
// Event Handling 
//
- (void)doClick:(id)sender
{
	[self sendAction];								// Send action to target
}

- (void)doDoubleClick:(id)sender
{
int column = [self columnOfMatrix: sender];
NSArray *a;
BOOL shouldSelect = YES;

fprintf(stderr, " NSBrowser: doDoubleClick --- ");

	if (column == NSNotFound)						// If the matrix isn't ours
    	return;										// then just return

	if ([_browserDelegate respondsToSelector: 			// Ask the delegate if
			@selector(browser:selectRow:inColumn:)])	// the selection is ok
		{
		int row = [sender selectedRow];
		shouldSelect = [_browserDelegate browser: self 
										selectRow: row
										inColumn: column];
		}
	else											// Try the other delegate
		{											// method
		if ([_browserDelegate respondsToSelector:
				@selector(browser:selectCellWithString:inColumn:)])
			{
			id c = [sender selectedCell];
			shouldSelect = [_browserDelegate browser:self
					   			selectCellWithString:[c stringValue]
					   			inColumn:column];
			}
    	}

	if (!shouldSelect)						// If we should not select the cell
		{									// then select it and return
		[sender deselectSelectedCell];
		return;
		}

	a = [sender selectedCells];
  
	if ([a count] == 1)						// If only one cell is selected
		{
		id c = [a objectAtIndex: 0];
		if ([c isLeaf])						// If the cell is a leaf
			{								// then unload the columns after
			[self _unloadFromColumn: column + 1];
			[self setLastColumn: column];
			}
		else			// The cell is not a leaf so we need to load a column
			{					
			if (column == (int)([_browserColumns count] - 1))
	   	 		[self addColumn];			// If last column then add a column

			[self _performLoadOfColumn: column + 1];		// Load column
			[self setLastColumn: column + 1];
			[self _adjustMatrixOfColumn: column + 1];
			[self _unloadFromColumn: column + 2];
													// If this is the last 
			if (column == _lastVisibleColumn)		// visible column then
				[self scrollColumnsRightBy: 1];		// scroll right by one col
			}
    	}
	else										// If a multiple selection then
    	{										// we unload the columns after			
		[self _unloadFromColumn: column + 1];		
		[self setLastColumn: column];				
    	}
									// We have already handled the single click
									// so send the double action
	[self sendAction: _doubleAction to: [self target]];
}

@end

//*****************************************************************************
//
// 		ActiveBrowserDelegate 
//
//		an NSBrowser delegate which actively creates the rows
//
//*****************************************************************************

@implementation ActiveBrowserDelegate

- (void)browser:(NSBrowser *)sender createRowsForColumn:(int)column
      								inMatrix:(NSMatrix *)matrix
{
NSFileManager *fm = [NSFileManager defaultManager];
NSString *ptc = [sender pathToColumn: column];
NSArray *files = [fm directoryContentsAtPath: ptc];
int i, count = [files count];
int j, c, nrows;

	if (count == 0)
		return;
													// add upto NUM_OF_COL  
	c = count < NUM_OF_COL ? count : NUM_OF_COL;	// columns to the matrix
	for (j = 0; j < c; ++j)
		[matrix addColumn];

	nrows = count/c;
	for (i = 0; i < nrows; ++i)
		{
		for (j = 0; j < c; ++j)
			{
			id cell;
			BOOL exists = NO, is_dir = NO;
			NSMutableString *s = [[[NSMutableString alloc] initWithString: ptc]
															autorelease];
								 
			if (i != 0 && j == 0)					// First row is created
				[matrix insertRow: i];				// when a column is added
	
			cell = [matrix cellAtRow: i column: j];
		
			[cell setStringValue: [files objectAtIndex: i*c + j]];
		
			[s appendString: @"/"];
			[s appendString: [files objectAtIndex: i*c + j]];
			exists = [fm fileExistsAtPath: s isDirectory: &is_dir];
		
			if ((exists) && (is_dir))
				[cell setLeaf: NO];
			else
				[cell setLeaf: YES];
			}
		}
}

- (void)browser:(NSBrowser *)sender willDisplayCell:(id)cell
									atRow:(int)row
									column:(int)column
{
}

- (BOOL)browser:(NSBrowser *)sender selectCellWithString:(NSString *)title
       								inColumn:(int)column;
{
//NSFileManager *fm = [NSFileManager defaultManager];
NSString *ptc = [sender pathToColumn: column];
NSMutableString *s = [[[NSMutableString alloc] initWithString:ptc]autorelease];

 fprintf(stderr, " browser:sender selectCellWithString: %s ", [title cString]);

	if(column > 0)
		[s appendString: @"/"];
	[s appendString:title];

fprintf(stderr, " source Path: %s ", [s cString]);

// fprintf(stderr, " destination Path: %s ", [d cString]);
//      if([fm movePath:s toPath:d handler:self])

}

//
// @interface NSObject (NSFileManagerHandler)
//
- (BOOL)fileManager:(NSFileManager*)fileManager
		shouldProceedAfterError:(NSDictionary*)errorDictionary
{	
	return YES;	
}

- (void)fileManager:(NSFileManager*)fileManager
		willProcessPath:(NSString*)path
{
}

@end
