/*
   PXKMenu.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Ovidiu Predescu <ovidiu@net-community.com>
   Date: May 1997
   A completely rewritten version of the original source by Scott Christley.
   
   This file is part of the GNUstep GUI X/DPS Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <config.h>
#include <stdlib.h>

#include <Foundation/NSArray.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSUserDefaults.h>
#include <Foundation/NSNotification.h>

#include <AppKit/NSMatrix.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSApplication.h>
#include <AppKit/NSMenu.h>

#include <gnustep/xdps/PXKMenuWindow.h>
#include <gnustep/xdps/PXKDPSContext.h>
#include <gnustep/xdps/PXKMenu.h>

#define ASSIGN(variable, value) \
  [value retain]; \
  [variable release]; \
  variable = value;

NSString* NSMenuLocationsKey = @"NSMenuLocations";

//#define USE_NEEDS_DISPLAY

@implementation PXKMenuMatrix

- (BOOL)_processMouseEvent:(NSEvent*)theEvent level:(int)level
{
  unsigned eventMask = NSLeftMouseUpMask | NSLeftMouseDownMask
    | NSMouseMovedMask | NSLeftMouseDraggedMask | NSPeriodicMask;
  NSPoint lastLocation = [theEvent locationInWindow];
  NSApplication* theApp = [NSApplication sharedApplication];
  float height = [self frame].size.height;
  NSEvent* lastEvent = nil;
  id aCell;
  BOOL done = NO;
  BOOL newlySelected = (selectedCell == nil);
  int index;
  NSWindow* lastWindow;

  ASSIGN(lastEvent, theEvent);
#if !USE_NEEDS_DISPLAY
  [self lockFocus];
#endif

  while (!done) {
    BOOL shouldGetAnotherEvent = YES;

    /* Convert the point from the event's window to our window. For the main
       menu window they are the same. */
    if ((lastWindow = [lastEvent window]) != window) {
      lastLocation = [[lastEvent window] convertBaseToScreen:lastLocation];
      lastLocation = [window convertScreenToBase:lastLocation];
    }

    NSDebugLog (@"location = (%f, %f)", lastLocation.x, lastLocation.y);
    lastLocation = [self convertPoint:lastLocation fromView:nil];

    if (lastLocation.x >= 0) {

      /* The mouse can be inside our window or one of our attached menus */
      if (lastLocation.x < cellSize.width) {

	/* The mouse can be inside our window, check the y coordinate. */
	if (lastLocation.y >= 0 && lastLocation.y < height) {

	  /* The mouse is really inside our window. Determine the selected
	     cell. If the mouse is between two cells we simply assume it is
	     on the upper one. */
	  index = (height - lastLocation.y)
		  / (cellSize.height + INTERCELL_SPACE);
	  aCell = [cells objectAtIndex:index];

	  /* Now unselect the previous cell and select the new one. Ignore if
	     the cells are the same. If the previous cell has submenu, close
	     its associated window.
	   */
	  if (selectedCell != aCell) {
	    if (selectedCell) {
	      if ([selectedCell hasSubmenu]) {
		[[selectedCell target] close];
		[selectedCell setState:0];
	      }
	      [selectedCell highlight:NO
			    withFrame:selectedCellRect
			       inView:self];
	    }

	    ASSIGN(selectedCell, aCell);
	    selectedCellRect = [self cellFrameAtRow:index];
	    [selectedCell highlight:YES
			  withFrame:selectedCellRect
			  inView:self];

#if USE_NEEDS_DISPLAY
	    [self setNeedsDisplayInRect:selectedCellRect];
#else
	    [[self window] flushWindow];
#endif

	    /* If the current selected cell has a submenu open it */
	    if ([selectedCell hasSubmenu]) {
	      [[selectedCell target] display];
	      newlySelected = YES;
	    }
	  }
	}
	else {
	  /* The mouse is outside of menu window, unselect the previous cell */
	  if (selectedCell) {
	    [selectedCell highlight:NO
			  withFrame:selectedCellRect
			      inView:self];
	    if ([selectedCell hasSubmenu]) {
	      [[selectedCell target] close];
	      [selectedCell setState:0];
	    }
	    ASSIGN(selectedCell, nil);
#if USE_NEEDS_DISPLAY
	    [self setNeedsDisplayInRect:selectedCellRect];
#else
	    [[self window] flushWindow];
#endif
	  }
	}
      }
      else {
	/* The mouse can be in one of the right attached menus. */
	NSMenu* attachedMenu = [menu attachedMenu];
	id menuCells = [attachedMenu menuCells];

	if (attachedMenu
	    && [menuCells _processMouseEvent:lastEvent level:level + 1]) {
	  done = YES;
	  shouldGetAnotherEvent = NO;
	}
	if (selectedCell) {
	  [selectedCell highlight:NO
			withFrame:selectedCellRect
			inView:self];
	  if (attachedMenu) {
	    [attachedMenu close];
	    [selectedCell setState:0];
	  }
	  ASSIGN(selectedCell, nil);
#if USE_NEEDS_DISPLAY
	  [self setNeedsDisplayInRect:selectedCellRect];
#else
	  [[self window] flushWindow];
#endif
	}
      }
    }
    else {
      /* The mouse is on the left of menu. Close the current menu window. */
      if (level) {
	if (selectedCell) {
	    [selectedCell highlight:NO
			  withFrame:selectedCellRect
			      inView:self];
	    [selectedCell setState:0];
#if USE_NEEDS_DISPLAY
	    [self setNeedsDisplayInRect:selectedCellRect];
#else
	    [[self window] flushWindow];
#endif
	    [menu close];
	    ASSIGN(selectedCell, nil);
	}
#if !USE_NEEDS_DISPLAY
	[self unlockFocus];
#endif
	[lastEvent release];
	return NO;
      }
    }

    /* Get the next event */
    while (shouldGetAnotherEvent) {
      theEvent = [theApp nextEventMatchingMask:eventMask
				    untilDate:[NSDate distantFuture]
				       inMode:NSEventTrackingRunLoopMode
				      dequeue:YES];
      switch ([theEvent type]) {
	case NSPeriodic:
	  shouldGetAnotherEvent = NO;
	  break;
	case NSLeftMouseUp:
	  done = YES;
	  shouldGetAnotherEvent = NO;
	  ASSIGN(lastEvent, theEvent);
	  break;
	default:
	  ASSIGN(lastEvent, theEvent);
	  continue;
      }
    }
    lastLocation = [lastEvent locationInWindow];
  }

  /* Unselect the cell unless it has submenu. */
 if (![selectedCell hasSubmenu]) {
    /* Send the menu item action to its target */
    [menu performActionForItem:selectedCell];

    [selectedCell highlight:NO withFrame:selectedCellRect inView:self];
    ASSIGN(selectedCell, nil);
  }
  else {
    /* Set the state of the selected cell, if 1 this indicates the submenu is
       opened. */
    if ([[selectedCell target] hasAnAssociatedTornOffMenu]||newlySelected==NO) {
      [selectedCell setState:0];
      [selectedCell highlight:NO withFrame:selectedCellRect inView:self];
      [[selectedCell target] close];
      ASSIGN(selectedCell, nil);
    }
  }
#if USE_NEEDS_DISPLAY
  [self setNeedsDisplayInRect:selectedCellRect];
#else
  [[self window] flushWindow];
#endif

#if !USE_NEEDS_DISPLAY
  [self unlockFocus];
#endif

  [lastEvent release];

  return YES;
}

- (void)mouseDown:(NSEvent*)theEvent
{
  [NSEvent startPeriodicEventsAfterDelay:0.05 withPeriod:0.05];
  [self _processMouseEvent:theEvent level:0];
  [NSEvent stopPeriodicEvents];
}

@end /* PXKMenuMatrix */


typedef struct {
  PXKMenuWindow* window;
  PXKMenuWindowTitleView* titleView;
  BOOL isTornOff;
  BOOL hasAnAssociatedTornOffMenu;
  PXKMenu* isCopyOfMenu;
} PXKMenuPrivate;

#define WINDOW (((PXKMenuPrivate*)be_menu_reserved)->window)
#define TITLEVIEW (((PXKMenuPrivate*)be_menu_reserved)->titleView)
#define ISTORNOFF (((PXKMenuPrivate*)be_menu_reserved)->isTornOff)
#define HASANASSOCIATEDTORNOFFMENU \
    (((PXKMenuPrivate*)be_menu_reserved)->hasAnAssociatedTornOffMenu)
#define ISCOPYOFMENU (((PXKMenuPrivate*)be_menu_reserved)->isCopyOfMenu)

static titleHeight = 0;

@implementation PXKMenu

+ (void)initialize
{
	if (self == [PXKMenu class])
		[self setVersion:1];								// Initial version
}

- (void)_createWindowWithFrame:(NSRect)winRect
{
  be_menu_reserved = calloc (sizeof (PXKMenuPrivate), 1);
  WINDOW = [PXKMenuWindow alloc];
  titleHeight = [menuCells cellSize].height + 4;
  [WINDOW setTitleHeight:titleHeight];
  [WINDOW setMenu:self];
  [WINDOW initWithContentRect:winRect
	  styleMask:NSTitledWindowMask
	  backing:NSBackingStoreRetained
	  defer:NO];

  NSDebugLog (@"create menu with title '%@', frame = (%f, %f, %f, %f)",
	 [self title],
	 winRect.origin.x, winRect.origin.y,
	 winRect.size.width, winRect.size.height);
  TITLEVIEW = [PXKMenuWindowTitleView new];
  [[WINDOW contentView] addSubview:TITLEVIEW];
  [[WINDOW contentView] addSubview:menuCells];

  [WINDOW setContentSize:winRect.size];
  [TITLEVIEW setFrameOrigin:NSMakePoint(0, winRect.size.height)];
  [TITLEVIEW setFrameSize:NSMakeSize (winRect.size.width, titleHeight)];
}

- (id)initWithTitle:(NSString*)aTitle
{
  NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
  NSApplication* theApp = [NSApplication sharedApplication];
  NSRect winRect = {{0, 0}, {1, 1}};

  [super initWithTitle:aTitle];
  [self _createWindowWithFrame:winRect];

  [defaultCenter addObserver:self
		 selector:@selector(_showTornOffMenuIfAny:)
		 name:NSApplicationWillFinishLaunchingNotification
		 object:theApp];
  [defaultCenter addObserver:self
		 selector:@selector(_showTornOffMenuIfAny:)
		 name:NSApplicationWillBecomeActiveNotification
		 object:theApp];

  return self;
}

- (void)dealloc
{
  NSDebugLog (@"PXKMenu '%@' dealloc", [self title]);

  [WINDOW release];
  [TITLEVIEW release];
  free (be_menu_reserved);
  [super dealloc];
}

- (id)copyWithZone:(NSZone*)zone
{
  PXKMenu* copy = [super copyWithZone:zone];
  NSRect frame = [NSWindow frameRectForContentRect:[menuCells frame]
			    styleMask:[WINDOW styleMask]];

  [copy _createWindowWithFrame:frame];

  return copy;
}

/* This method is invoked when the user presses on the close button of a torn
   off menu window. */
- (void)_performMenuClose:sender
{
  NSUserDefaults* defaults = [NSUserDefaults standardUserDefaults];
  NSMutableDictionary* menuLocations
      = [[[defaults objectForKey:NSMenuLocationsKey]
		    mutableCopy]
		    autorelease];
  NSString* key;

  NSDebugLog (@"_performMenuClose");

  [self close];

  /* Inform our copy we are no longer available */
  (((PXKMenuPrivate*)((ISCOPYOFMENU)->be_menu_reserved))
		      ->hasAnAssociatedTornOffMenu) = NO;

  /* Remove from defaults our window position */
  key = [self title];
  if (key) {
    [menuLocations removeObjectForKey:key];
    [defaults setObject:menuLocations forKey:NSMenuLocationsKey];
    [defaults synchronize];
  }

  [self autorelease];
}

/* Show the torn off menus */
- (void)_showTornOffMenuIfAny:(NSNotification*)notification
{
  NSUserDefaults* defaults = [NSUserDefaults standardUserDefaults];
  NSDictionary* menuLocations = [defaults objectForKey:NSMenuLocationsKey];
  NSApplication* theApp = [NSApplication sharedApplication];
  NSString* key;
  NSArray* array;
  NSPoint origin;

  if ([theApp mainMenu] == self)
    key = nil; /* Ignore the main menu */
  else
    key = [self title];

  if (key) {
    array = [menuLocations objectForKey:key];
    if (array && [array isKindOfClass:[NSArray class]]) {
      [self windowBecomeTornOff];
      origin.x = [[array objectAtIndex:0] floatValue];
      origin.y = [[array objectAtIndex:1] floatValue];
      [WINDOW setFrameOrigin:origin];
      [self display];
    }
  }
}

- (id <NSMenuItem>)insertItemWithTitle:(NSString*)aString
				action:(SEL)aSelector
			 keyEquivalent:(NSString*)charCode
			       atIndex:(unsigned int)index
{
  NSRect winFrame = [WINDOW frame];
  float upperLeftWinPos = winFrame.origin.y + [menuCells frame].size.height;
  id cell;

  cell = [super insertItemWithTitle:aString
			     action:aSelector
		      keyEquivalent:charCode
			    atIndex:index];
  winFrame.origin.y = upperLeftWinPos - [menuCells frame].size.height;
  [WINDOW setFrameOrigin:winFrame.origin];

  return cell;
}

- (void)removeItem:(id <NSMenuItem>)anItem
{
  NSRect winFrame = [WINDOW frame];
  float upperLeftWinPos = winFrame.origin.y + [menuCells frame].size.height;

  [super removeItem:anItem];
  winFrame.origin.y = upperLeftWinPos - [menuCells frame].size.height;
  [WINDOW setFrameOrigin:winFrame.origin];
}

- (void)sizeToFit
{
  NSRect frame;

  [super sizeToFit];

  frame = [NSWindow frameRectForContentRect:[menuCells frame]
				  styleMask:[WINDOW styleMask]];
  frame.origin = [WINDOW frame].origin;

#if DEBUGLOG
  printf ("%p (%s): window frame (%4.2f %4.2f %4.2f %4.2f)\n", self,
	  [[self title] cString],
	  frame.origin.x, frame.origin.y, frame.size.width, frame.size.height);
#endif

  [WINDOW setContentSize:frame.size];
  [WINDOW setFrame:frame display:[WINDOW isVisible]];

  [TITLEVIEW setFrameOrigin:NSMakePoint (0, frame.size.height)];
  [TITLEVIEW setFrameSize:NSMakeSize (frame.size.width, titleHeight)];
}

- (void)submenuAction:(id)sender
{
}

- (void)windowBecomeTornOff
{
  PXKMenu* copy = [self copy]; /* The new menu is released when the user
				  presses its close button */
  id supermenuMenuCells = [supermenu menuCells];
  id supermenuSelectedCell = [supermenuMenuCells selectedCell];
  NSRect rect;

  if (!supermenuSelectedCell) {
    /* There is no selected cell in super menu, this happens when the method
       was invoked to create torn off menus. Identify the corresponding cell
       by comparing the cell targets from the upper menu cells array. */
    NSArray* cells = [supermenuMenuCells itemArray];
    int i, count = [cells count];

    for (i = 0; i < count; i++) {
      id cell = [cells objectAtIndex:i];

      if ([cell target] == self) {
	supermenuSelectedCell = cell;
	break;
      }
    }
  }

  /* Unselect the menu cell from super menu */
#if !USE_NEEDS_DISPLAY
  [supermenuMenuCells lockFocus];
#endif
  rect = [supermenuMenuCells selectedCellRect];
  [supermenuSelectedCell highlight:NO
			  withFrame:rect
			  inView:supermenuMenuCells];
  [supermenuSelectedCell setState:0];
#if USE_NEEDS_DISPLAY
  [supermenuMenuCells setNeedsDisplayInRect:rect];
#else
  [[supermenuMenuCells window] flushWindow];
#endif
  [supermenuMenuCells setSelectedCell:nil];
#if !USE_NEEDS_DISPLAY
  [supermenuMenuCells unlockFocus];
#endif

  [supermenuSelectedCell setTarget:copy];

  ((PXKMenu*)supermenu)->attachedMenu = nil;
  supermenu = nil;
  ((PXKMenuPrivate*)copy->be_menu_reserved)->hasAnAssociatedTornOffMenu = YES;

  [TITLEVIEW windowBecomeTornOff];
  ISTORNOFF = YES;
  ISCOPYOFMENU = copy;
}

- (BOOL)isTornOff
{
  return ISTORNOFF;
}

- (NSPoint)locationForSubmenu:(NSMenu*)aSubmenu
{
  NSRect frame = [WINDOW frame];
  NSRect submenuFrame = NSZeroRect;

  if (aSubmenu)
    submenuFrame = [((PXKMenuPrivate*)(((PXKMenu*)aSubmenu)->be_menu_reserved))
			->window frame];

  return NSMakePoint (frame.origin.x + frame.size.width + 1,
		      frame.origin.y + frame.size.height
			- submenuFrame.size.height);
}

- (void)update
{
  [super update];
  if ([WINDOW isVisible])
    [self display];
}

- (void)display
{
  /* Find out our position if we are not the main menu */
  if (supermenu) {
    NSPoint location = [supermenu locationForSubmenu:self];
    [WINDOW setFrameOrigin:location];
    ((PXKMenu*)supermenu)->attachedMenu = self;
  }
  else {
    NSUserDefaults* defaults = [NSUserDefaults standardUserDefaults];
    NSDictionary* menuLocations = [defaults objectForKey:NSMenuLocationsKey];
    NSApplication* theApp = [NSApplication sharedApplication];
    NSString* key;
    NSArray* array;
    NSPoint origin;

    if ([theApp mainMenu] == self)
      key = @"Main menu";
    else
      key = [self title];

    if (key) {
      array = [menuLocations objectForKey:key];
      if (array && [array isKindOfClass:[NSArray class]]) {
	origin.x = [[array objectAtIndex:0] floatValue];
	origin.y = [[array objectAtIndex:1] floatValue];
	[WINDOW setFrameOrigin:origin];
      }
    }
  }

  /* Invoke the -submenuAction: method so subclasses can use it. */
  [self submenuAction:nil];

  [WINDOW display];
  [WINDOW orderFront:nil];
}

- (void)close
{
  PXKDPSContext *context = [NSDPSContext currentContext];
  id selectedCell = [menuCells selectedCell];

  /* Also close the attached menu. This is recursive, if the attached menu
     has an attached menu this will be closed too. */
  if (attachedMenu)
    [attachedMenu close];

  [context setXDrawable:WINDOW];

  if (selectedCell && [selectedCell isHighlighted]) {
    NSRect selectedCellRect = [menuCells selectedCellRect];

    [selectedCell highlight:NO withFrame:selectedCellRect inView:menuCells];
    [selectedCell setState:0];
    [WINDOW flushWindow];
    [menuCells setSelectedCell:nil];
  }

  /* Invoke the -submenuAction: method so subclasses can use it. */
  [self submenuAction:nil];

  [WINDOW orderOut:nil];
  if (supermenu) {
    ((PXKMenu*)supermenu)->attachedMenu = nil;
    [context setXDrawable:[(PXKMenu*)supermenu menuWindow]];
  }

}

- (PXKMenuWindow*)menuWindow
{
  return WINDOW;
}

- (PXKMenuWindowTitleView*)titleView
{
  return TITLEVIEW;
}

- (BOOL)hasAnAssociatedTornOffMenu
{
  return (((PXKMenuPrivate*)be_menu_reserved)->hasAnAssociatedTornOffMenu);
}

- (Class) classForCoder: aCoder
{
  if ([self class] == [PXKMenu class])
    return [super class];
  return [self class];
}

@end /* PXKMenu */
