/*
   AFMFileFontInfo.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: February 1997
   
   This file is part of the GNUstep GUI X/DPS Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <float.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <sys/file.h>
#include <fcntl.h>

#include <Foundation/NSDictionary.h>
#include <Foundation/NSString.h>
#include <Foundation/NSFileManager.h>
#include <Foundation/NSAutoreleasePool.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSDate.h>

#include <AppKit/NSFont.h>
#include "AFMFileFontInfo.h"
#include "fonts.h"

#ifndef MAX
#define MAX(a, b) \
    ({typedef _ta = (a), _tb = (b); \
	_ta _a = (a); _tb _b = (b); \
	_a > _b ? _a : _b; })
#endif

#ifndef MIN
#define MIN(a, b) \
    ({typedef _ta = (a), _tb = (b); \
	_ta _a = (a); _tb _b = (b); \
	_a < _b ? _a : _b; })
#endif

/* This dictionary keeps global font info objects indexed by AFM file names.
   A global font info instance is the representation of AFM file; all the
   dimensions are in the 1x1 space not in the 1000x1000 space of the AFM file.
   Each font object maintains a copy of this instance with all the dimensions
   scaled to the font size. */
static NSMutableDictionary* globalFontInfoDictionary = nil;

#ifndef LIB_FOUNDATION_LIBRARY
/* Define callbacks for a map table that has cStrings as keys */

/* hpjw from Aho, Sethi & Ullman: Principles of compiler design. */
static unsigned __NSHashCString(void *table, const void *aString)
{
    register const char* p = (char*)aString;
    register unsigned hash = 0, hash2;
    register int i, n = strlen((char*)aString);

    for(i=0; i < n; i++) {
        hash <<= 4;
        hash += *p++;
        if((hash2 = hash & 0xf0000000))
            hash ^= (hash2 >> 24) ^ hash2;
    }
    return hash;
}

static BOOL __NSCompareCString(void *table, 
    const void *anObject1, const void *anObject2)
{
    return strcmp((char*)anObject1, (char*)anObject2) == 0;
}

static void __NSRetainNothing(void *table, const void *anObject)
{
}

static void __NSReleaseNothing(void *table, void *anObject)
{
}

static NSString* __NSDescribePointers(void *table, const void *anObject)
{
    return [NSString stringWithCString:anObject];
}

static const NSMapTableKeyCallBacks NSNonOwnedCStringMapKeyCallBacks = {
    (unsigned(*)(NSMapTable*, const void*))__NSHashCString,
    (BOOL(*)(NSMapTable*, const void*, const void*))__NSCompareCString,
    (void (*)(NSMapTable*, const void* anObject))__NSRetainNothing,
    (void (*)(NSMapTable*, void* anObject))__NSReleaseNothing,
    (NSString *(*)(NSMapTable*, const void*))__NSDescribePointers,
    (const void *)NULL
}; 
#endif /* LIB_FOUNDATION_LIBRARY */


@implementation AFMFileFontInfo

+ (void)initialize
{
  static BOOL initialized = NO;

  if (!initialized) {
    initialized = YES;
    globalFontInfoDictionary = [NSMutableDictionary new];
  }
}

+ (AFMFileFontInfo*)fontInfoForFont:(NSFont*)font
  afmFilename:(NSString*)afmFileName
{
  AFMFileFontInfo* fontInfo;
  const float* matrix = [font matrix];

  /* Check to see if the font was previously requested. */
  if ((fontInfo = [globalFontInfoDictionary objectForKey:afmFileName]))
    return [fontInfo newTransformedFontInfoForMatrix:matrix];

  /* The font was not previously requested. Parse the AFM file and create a
     new font info for it. */
  fontInfo = [self createFontInfoFromAFMFileName:afmFileName];

  if (!fontInfo)
    return nil;

  /* Record into the global font info dictionary that we've parsed the AFM
     file. */
  [globalFontInfoDictionary setObject:fontInfo forKey:afmFileName];

  return [fontInfo newTransformedFontInfoForMatrix:matrix];
}

+ (AFMFileFontInfo*)createFontInfoFromAFMFileName:(NSString*)fileName
{
  AFMFileFontInfo* fontInfo = [[self new] autorelease];
  NSMutableDictionary* afmDict = fontInfo->afmDictionary;
  NSAutoreleasePool* pool = nil;
  AFMFontInfo* cFontInfo;
  AFMGlobalFontInfo* gfi;
  const char* fileNameCString;
  int fontType;
  FILE* file;
  int i, code;

/* Do this until gnustep-base will contain the NSFileManager class. Not very
   accurate since NeXT Foundation library does have NSFileManager. On NT or
   Win95 probably the 'else' code should be modified such that the cstring
   passed to access() contains \ instead of / (in case of NSFileManager this
   is hidden inside its implementation). */
#if defined(LIB_FOUNDATION_LIBRARY) || defined(NeXT_PDO)
  if (![[NSFileManager defaultManager] isReadableFileAtPath:fileName])
    return nil;
#else
  if (access([fileName cString], R_OK) < 0)
    return nil;
#endif

  fileNameCString = [fileName cString];
  file = fopen (fileNameCString, "r");
  if (!file) {
    NSLog (@"cannot open AFM file %@", fileName);
    abort ();
  }

  code = AFMParseFile (file, &cFontInfo, AFM_G | AFM_M | AFM_P);
  if (code != afm_ok) {
    switch (code) {
      case afm_parseError:
	NSLog (@"parse error in AFM file %@", fileName);
	break;
      case afm_earlyEOF:
	NSLog (@"unexpected EOF in AFM file %@", fileName);
	break;
      case afm_storageProblem:
	NSLog (@"memory allocation problem while parsing the AFM file %@",
		fileName);
	break;
    }
    return nil;
  }
  fclose (file);

  gfi = cFontInfo->gfi;

  pool = [NSAutoreleasePool new];

  /*
   * Set the font information in instance variables and in the AFM dictionary.
   */

  fontInfo->fontName = [[NSString stringWithCString:gfi->fontName] retain];
  [afmDict setObject:fontInfo->fontName forKey:NSAFMFontName];

  fontInfo->familyName = [[NSString stringWithCString:gfi->familyName] retain];
  [afmDict setObject:fontInfo->familyName forKey:NSAFMFamilyName];

  fontInfo->italicAngle = gfi->italicAngle;
  [afmDict setObject:[[NSNumber numberWithFloat:gfi->italicAngle] stringValue]
	   forKey:NSAFMItalicAngle];

  fontInfo->weight = [[NSString stringWithCString:gfi->weight] retain];
  [afmDict setObject:fontInfo->weight forKey:NSAFMWeight];

  fontInfo->underlinePosition = ((float)gfi->underlinePosition) / 1000;
  [afmDict setObject:[[NSNumber numberWithFloat:fontInfo->underlinePosition]
				stringValue]
	   forKey:NSAFMUnderlinePosition];

  fontInfo->underlineThickness = ((float)gfi->underlineThickness) / 1000;
  [afmDict setObject:[[NSNumber numberWithFloat:fontInfo->underlineThickness]
				stringValue]
	   forKey:NSAFMUnderlineThickness];

  fontInfo->capHeight = ((float)gfi->capHeight) / 1000;
  [afmDict setObject:[[NSNumber numberWithFloat:fontInfo->capHeight]
				stringValue]
	   forKey:NSAFMCapHeight];

  fontInfo->xHeight = ((float)gfi->xHeight) / 1000;
  [afmDict setObject:[[NSNumber numberWithFloat:fontInfo->xHeight] stringValue]
	   forKey:NSAFMXHeight];

  fontInfo->descender = ((float)gfi->descender) / 1000;
  [afmDict setObject:[[NSNumber numberWithFloat:fontInfo->descender]
				stringValue]
	   forKey:NSAFMDescender];

  fontInfo->ascender = ((float)gfi->ascender) / 1000;
  [afmDict setObject:[[NSNumber numberWithFloat:fontInfo->ascender]
				stringValue]
	   forKey:NSAFMAscender];

  fontInfo->encodingScheme = [[NSString stringWithCString:gfi->encodingScheme]
					retain];
  [afmDict setObject:fontInfo->encodingScheme forKey:NSAFMEncodingScheme];

  [afmDict setObject:[NSString stringWithCString:gfi->afmVersion]
	   forKey:NSAFMFormatVersion];
  [afmDict setObject:[NSString stringWithCString:gfi->notice]
	   forKey:NSAFMNotice];
  [afmDict setObject:[NSString stringWithCString:gfi->version]
	   forKey:NSAFMVersion];

  /* Setup bbox as expected by NSFont */
  fontInfo->fontBBox.origin.x = ((float)gfi->fontBBox.llx) / 1000;
  fontInfo->fontBBox.origin.y = ((float)gfi->fontBBox.lly) / 1000;
  fontInfo->fontBBox.size.width
      = ((float)(gfi->fontBBox.urx - gfi->fontBBox.llx)) / 1000;
  fontInfo->fontBBox.size.height
      = ((float)(gfi->fontBBox.ury - gfi->fontBBox.lly)) / 1000;

  fontInfo->isFixedPitch = gfi->isFixedPitch;

  /* Get the font type from the DGS server */
  PSWGetFontType ([fontInfo->fontName cString], &fontType);
  fontInfo->isBaseFont = (fontType == 1);

  fontInfo->maximumAdvancement.width = FLT_MIN;
  fontInfo->maximumAdvancement.height = FLT_MIN;
  fontInfo->minimumAdvancement.width = FLT_MAX;
  fontInfo->minimumAdvancement.height = FLT_MAX;

  /* Fill in the widths and glyphs arrays. */
  for (i = 0; i < cFontInfo->numOfChars; i++) {
    AFMCharMetricInfo charMetricInfo = cFontInfo->cmi[i];
    AFMGlyphInfo* glyph
	= [AFMGlyphInfo glyphFromAFMCharMetricInfo:&charMetricInfo];
    NSSize glyphAdvancement = [glyph advancement];

    NSMapInsert (fontInfo->glyphsByName, [[glyph name] cString], glyph);
    fontInfo->maximumAdvancement.width
	= MAX (fontInfo->maximumAdvancement.width, glyphAdvancement.width);
    fontInfo->maximumAdvancement.height
	= MAX (fontInfo->maximumAdvancement.height, glyphAdvancement.height);
    fontInfo->minimumAdvancement.width
	= MIN (fontInfo->minimumAdvancement.width, glyphAdvancement.width);
    fontInfo->minimumAdvancement.height
	= MIN (fontInfo->minimumAdvancement.height, glyphAdvancement.height);
    if (charMetricInfo.code > 0) {
      fontInfo->widths[charMetricInfo.code] = charMetricInfo.wx;
      fontInfo->glyphs[charMetricInfo.code] = [glyph retain];
    }
  }

  /* Set the entries in the kerning array for all the glyphs in the pair
     kerning array. */
  {
    AFMGlyphInfo* glyph1 = nil;
    AFMGlyphInfo* glyph2 = nil;
    const char* previousGlyphName = "";
    AFMPairKernData pkd;
    NSSize advancement;

    /* First compute the numbers of kern pairs for each glyph. This is used to
       avoid unnecessary allocations of kern pair arrays in glyph objects. */
    for (i = 0; i < cFontInfo->numOfPairs; i++) {
      pkd = cFontInfo->pkd[i];

      /* Check the name of the first glyph. Use a hint here: the kern pairs are
         grouped on the first glyph. */
      if (strcmp (pkd.name1, previousGlyphName)) {
	previousGlyphName = pkd.name1;
	glyph1 = NSMapGet (fontInfo->glyphsByName, pkd.name1);
      }
      [glyph1 incrementNumberOfKernPairs];
    }

    /* Now set the pair kerns in glyphs. */
    for (i = 0; i < cFontInfo->numOfPairs; i++) {
      pkd = cFontInfo->pkd[i];

      if (strcmp (pkd.name1, previousGlyphName)) {
	previousGlyphName = pkd.name1;
	glyph1 = NSMapGet (fontInfo->glyphsByName, pkd.name1);
      }

      glyph2 = NSMapGet (fontInfo->glyphsByName, pkd.name2);

      if (!glyph1) {
	NSLog (@"unknown glyph name %s in AFM file %@", pkd.name1, fileName);
	continue;
      }
      if (!glyph2) {
	NSLog (@"unknown glyph name %s in AFM file %@", pkd.name2, fileName);
	continue;
      }

      advancement = NSMakeSize (pkd.xamt, pkd.yamt);
      [glyph1 addPairKerningForGlyph:[glyph2 code] advancement:advancement];
    }
  }

  /* Free the cFontInfo structure */
  free (gfi->afmVersion);
  free (gfi->fontName);
  free (gfi->fullName);
  free (gfi->familyName);
  free (gfi->weight);
  free (gfi->version);
  free (gfi->notice);
  free (gfi->encodingScheme);
  for (i = 0; i < cFontInfo->numOfChars; i++) {
    AFMCharMetricInfo cmi = cFontInfo->cmi[i];
    free (cmi.name);
  }
  for (i = 0; i < cFontInfo->numOfPairs; i++) {
    AFMPairKernData pkd = cFontInfo->pkd[i];
    free (pkd.name1);
    free (pkd.name2);
  }
  free (cFontInfo);

  [pool release];

  return fontInfo;
}

- init
{
  int i;

  self = [super init];

  afmDictionary = [[NSMutableDictionary dictionaryWithCapacity:25] retain];
  glyphsByName = NSCreateMapTable (NSNonOwnedCStringMapKeyCallBacks,
				   NSObjectMapValueCallBacks, 256);

  for (i = 0; i < 256; i++)
    widths[i] = 0.0;

  return self;
}

- (void)dealloc
{
  int i;

  /* Don't free the glyphsByName map table. It is owned by the first font info
     object that is never freed. */
  [afmDictionary release];
  [fontName release];
  [familyName release];
  [weight release];
  [encodingScheme release];

  for (i = 0; i < 256; i++)
    [glyphs[i] release];

  [super dealloc];
}

- (AFMFileFontInfo*)newTransformedFontInfoForMatrix:(const float*)matrix
{
  AFMFileFontInfo* new = [[self copy] autorelease];

  [new transformUsingMatrix:matrix];
  return new;
}

- (void)transformUsingMatrix:(const float*)matrix
{
  float a = matrix[0];
  float b = matrix[1];
  float c = matrix[2];
  float d = matrix[3];
  float tx = matrix[4];
  float ty = matrix[5];
  float x1, y1, width1, height1;
  int i;

  /* The doc says the result of -widths is not affected by font's matrix. */
  for (i = 0; i < 256; i++)
    [glyphs[i] transformUsingMatrix:matrix];

  x1 = fontBBox.origin.x;
  y1 = fontBBox.origin.y;
  width1 = fontBBox.size.width;
  height1 = fontBBox.size.height;
  fontBBox.origin.x = a * x1 + c * y1 + tx;
  fontBBox.origin.y = b * x1 + d * y1 + ty;
  fontBBox.size.width = a * width1 + c * height1;
  fontBBox.size.height = b * width1 + d * height1;

  width1 = maximumAdvancement.width;
  height1 = maximumAdvancement.height;
  maximumAdvancement.width = a * width1 + c * height1;
  maximumAdvancement.height = b * width1 + d * height1;

  width1 = minimumAdvancement.width;
  height1 = minimumAdvancement.height;
  minimumAdvancement.width = a * width1 + c * height1;
  minimumAdvancement.height = b * width1 + d * height1;

  underlinePosition = a * underlinePosition + tx;
  underlineThickness = a * underlineThickness + tx;
  capHeight = a * capHeight + tx;
  ascender = a * ascender + tx;
  descender = a * descender + tx;

  xHeight = a * xHeight + tx;
}

- copy
{
  AFMFileFontInfo* new = [isa alloc];
  int i;

  new->afmDictionary = [afmDictionary retain];
  new->glyphsByName = glyphsByName;
  new->maximumAdvancement.width = maximumAdvancement.width;
  new->maximumAdvancement.height = maximumAdvancement.height;
  new->minimumAdvancement.width = minimumAdvancement.width;
  new->minimumAdvancement.height = minimumAdvancement.height;

  for (i = 0; i < 256; i++) {
    new->widths[i] = widths[i];
    new->glyphs[i] = [glyphs[i] copy];
  }

  new->fontName = [fontName retain];
  new->familyName = [familyName retain];
  new->italicAngle = italicAngle;
  new->weight = [weight retain];
  new->underlinePosition = underlinePosition;
  new->underlineThickness = underlineThickness;
  new->capHeight = capHeight;
  new->xHeight = xHeight;
  new->descender = descender;
  new->ascender = ascender;
  new->encodingScheme = [encodingScheme retain];

  new->fontBBox.origin.x = fontBBox.origin.x;
  new->fontBBox.origin.y = fontBBox.origin.y;
  new->fontBBox.size.width = fontBBox.size.width;
  new->fontBBox.size.height = fontBBox.size.height;

  new->isFixedPitch = isFixedPitch;
  new->isBaseFont = isBaseFont;

  return new;
}

- (NSString*)encodingScheme			{ return encodingScheme; }
- (BOOL)isBaseFont				{ return isBaseFont; }
- (BOOL)isFixedPitch				{ return isFixedPitch; }
- (NSDictionary*)afmDictionary			{ return afmDictionary; }
- (NSRect)boundingRectForFont			{ return fontBBox; }
- (NSString*)displayName			{ return familyName; }
- (NSString*)familyName				{ return familyName; }
- (NSString*)fontName				{ return fontName; }
- (float)ascender				{ return ascender; }
- (float)descender				{ return descender; }
- (float)capHeight				{ return capHeight; }
- (float)italicAngle				{ return italicAngle; }
- (NSSize)maximumAdvancement			{ return maximumAdvancement; }
- (NSSize)minimumAdvancement			{ return minimumAdvancement; }
- (float)underlinePosition			{ return underlinePosition; }
- (float)underlineThickness			{ return underlineThickness; }
- (float)xHeight				{ return xHeight; }
- (float*)widths				{ return widths; }

- (NSSize)advancementForGlyph:(NSGlyph)glyph
{
  return [glyphs[glyph] advancement];
}

- (NSRect)boundingRectForGlyph:(NSGlyph)glyph
{
  return [glyphs[glyph] boundingRect];
}

- (BOOL)glyphIsEncoded:(NSGlyph)glyph
{
  return [glyphs[glyph] isEncoded];
}

- (NSGlyph)glyphWithName:(NSString*)glyphName
{
  AFMGlyphInfo* glyph = NSMapGet (glyphsByName, [glyphName cString]);

  return glyph ? [glyph code] : -1;
}

- (NSPoint)positionOfGlyph:(NSGlyph)curGlyph
  precededByGlyph:(NSGlyph)prevGlyph
  isNominal:(BOOL*)nominal
{
  NSPoint point;
  NSSize size;

  if (curGlyph == NSControlGlyph || prevGlyph == NSControlGlyph
      || curGlyph < 0 || curGlyph > 255 || prevGlyph < 0 || prevGlyph >255) {
    if (nominal)
      *nominal = NO;
    return NSZeroPoint;
  }

  if (curGlyph == NSNullGlyph) {
    size = [glyphs[prevGlyph] advancement];
    point.x = size.width;
    point.y = size.height;
    if (nominal)
      *nominal = NO;
    return point;
  }

  if (glyphs[prevGlyph]) {
    if (!glyphs[curGlyph]) {
      point.x = maximumAdvancement.width;
      point.y = maximumAdvancement.height;
      if (nominal)
	*nominal = NO;
      return point;
    }
    size = [glyphs[prevGlyph] advancementIfFollowedByGlyph:
				      [glyphs[curGlyph] code]
			      isNominal:nominal];
    point.x = size.width;
    point.y = size.height;
    return point;
  }
  return NSZeroPoint;
}

- (float)widthOfString:(NSString*)string
{
  /* TODO: We should really map here the characters from string to
     series of glyphs. Until then we consider the glyphs to be
     equivalent with characters. */

  int i, length = [string length];
  const char* cString = [string cString];
  float width = 0;

  for (i = 0; i < length; i++)
    width += [glyphs[(int)cString[i]] advancement].width;

  return width;
}

@end /* AFMFileFontInfo */


@implementation AFMGlyphInfo

+ (AFMGlyphInfo*)glyphFromAFMCharMetricInfo:(AFMCharMetricInfo*)mi
{
  AFMGlyphInfo* glyph = [[self new] autorelease];

  glyph->name = [[NSString stringWithCString:mi->name] retain];
  glyph->code = mi->code;

  /* Setup bbox as defined by NSRect */
  glyph->bbox.origin.x = ((float)mi->charBBox.llx) / 1000;
  glyph->bbox.origin.y = ((float)mi->charBBox.lly) / 1000;
  glyph->bbox.size.width = ((float)mi->charBBox.urx - mi->charBBox.llx) / 1000;
  glyph->bbox.size.height = ((float)mi->charBBox.ury - mi->charBBox.lly)/ 1000;

  glyph->advancement.width = ((float)mi->wx) / 1000;
  glyph->advancement.height = ((float)mi->wy) / 1000;

  return glyph;
}

- (void)transformUsingMatrix:(const float*)matrix;
{
  float a = matrix[0];
  float b = matrix[1];
  float c = matrix[2];
  float d = matrix[3];
  float tx = matrix[4];
  float ty = matrix[5];
  float x1 = bbox.origin.x;
  float y1 = bbox.origin.y;
  float width1 = bbox.size.width;
  float height1 = bbox.size.height;
  int i;

  bbox.origin.x = a * x1 + c * y1 + tx;
  bbox.origin.y = b * x1 + d * y1 + ty;
  bbox.size.width = a * width1 + c * height1;
  bbox.size.height = b * width1 + d * height1;

  x1 = advancement.width;
  y1 = advancement.height;
  advancement.width = a * x1 + c * y1 + tx;
  advancement.height = b * x1 + d * y1 + ty;

  for (i = 0; i < numOfPairs; i++) {
    x1 = kerning[i].advancement.width;
    y1 = kerning[i].advancement.height;
    kerning[i].advancement.width = a * x1 + c * y1 + tx;
    kerning[i].advancement.height = b * x1 + d * y1 + ty;
  }
}

- (void)incrementNumberOfKernPairs
{
  numOfPairs++;
}

- (void)addPairKerningForGlyph:(NSGlyph)glyph advancement:(NSSize)_advancement
{
  tPairKerningInfo kernInfo = {
      glyph,
      { advancement.width + ((float)_advancement.width) / 1000,
        advancement.height + ((float)_advancement.height) / 1000
      }
  };
  int i;

  if (kerning == NULL)
    kerning = malloc (numOfPairs * sizeof (tPairKerningInfo));

  /* Insert the glyph in the proper position in the kerning array so this will
     be sorted ascending after the glyph code. */
  for (i = 0; i < lastKernPair; i++)
    if (kerning[i].glyph > glyph) {
      /* Make room for a new kerning pair in this position. */
      memmove (&kerning[i + 1], &kerning[i],
	       (lastKernPair - i) * sizeof (tPairKerningInfo));
      break;
    }

  kerning[i] = kernInfo;
  lastKernPair++;
}

- (NSSize)advancementIfFollowedByGlyph:(NSGlyph)glyph
  isNominal:(BOOL*)nominal
{
  /* Search for the glyph using a binary search algorithm */
  int lower = 0;
  int upper = numOfPairs;
  int midpoint;

  if (!kerning) {
    if (nominal)
      *nominal = NO;
    return advancement;
  }

  while (upper >= lower) {
    midpoint = (lower + upper) / 2;
    if (kerning[midpoint].glyph == glyph) {
      if (nominal)
	*nominal = YES;
      return kerning[midpoint].advancement;
    }
    else if (kerning[midpoint].glyph > glyph)
      upper = midpoint - 1;
    else if (kerning[midpoint].glyph < glyph)
      lower = midpoint + 1;
  }

  /* The glyph was not found in the kernings array. Return the advancement of
     receiver. */
  if (nominal)
    *nominal = NO;
  return advancement;
}

- copy
{
  AFMGlyphInfo* new = [isa alloc];

  new->name = [name retain];
  new->code = code;
  new->bbox = bbox;
  new->advancement = advancement;

  if (kerning) {
    new->numOfPairs = numOfPairs;
    new->kerning = malloc (numOfPairs * sizeof (tPairKerningInfo));
    memcpy (new->kerning, kerning, numOfPairs * sizeof (tPairKerningInfo));
  }

  return new;
}

- (void)dealloc
{
  [name release];
  [super dealloc];
  if (kerning)
    free (kerning);
}

- (NSString*)name	{ return name; }
- (NSGlyph)code		{ return code; }
- (NSRect)boundingRect	{ return bbox; }
- (NSSize)advancement	{ return advancement; }
- (BOOL)isEncoded	{ return YES; }

@end /* AFMGlyphInfo */
