/* -*-C-*-
*******************************************************************************
*
* File:         plx9080.h
* Description:  PLX 9080 registers
*
*******************************************************************************
*/
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * References:
 *
 *  PLX 9080 PCI Data Book version 1.06
 *	http://www.plxtech.com/download/9080/databook/9080db-106.pdf
 */

/*
 * There are *lots* of registers in the PLX 9080.  I'm only defining the
 * one's I'm using...
 */

#ifndef _PLX9080_H_
#define _PLX9080_H_

/*
 * pci configuration registers are standard.
 * see linux/pci.h
 */

/*
 * Local configuration registers.
 * Offsets are relative to PCI side, add PLX_LCL_OFFSET for local address
 */

#define	PLX_LCL_OFFSET		0x80	// offset of regs from local side


#define	PLX_LAS0RR		0x00	// Local Address Space 0 Range Register PCI-to-local
#define	PLX_LAS0BA		0x04	// Local Address Space 0 Local Base Address (Remap) Register
#define	PLX_MARBR		0x08	// Mode/Arbitration Register
#define	PLX_BIGEND		0x0c	// Big/Little Endian Register
#define	PLX_EROMRR		0x10	// Expansion ROM Range Register
#define	PLX_EROMBA		0x14	// Expansion ROM Local Base Address (Remap) Register
#define	PLX_LBRD0		0x18	// Local Address Space 0 / Expn ROM Bus Region Desc
#define	PLX_DMRR		0x1c	// Local Range Register for Direct Master to PCI
#define	PLX_DMLBAM		0x20	// Local Bus Base Addr Reg for Direct Master to PCI Memory
#define	PLX_DMLBAI		0x24	// Local Base Addr Reg for Direct Master to PCI IO/CFG
#define	PLX_DMPBAM		0x28	// PCI Base Addr (Remap) Reg for Direct Master to PCI Memory
#define	PLX_DMCFGA		0x2c	// PCI Config Addr Reg for Direct Master to PCI IO/CFG

#define	PLX_LAS1RR		0xf0	// Local Addr Space 1 Range Reg for PCI-to-Local Bus
#define	PLX_LAS1BA		0xf4	// Local Addr Space 1 Local Base Addr (Remap) Register
#define	PLX_LBRD1		0xf8	// Local Addr Space 1 Bus Region Desc Reg

/*
 * Runtime registers.
 * Offsets are relative to PCI side.
 */

#define	PLX_MBOX0		0x40
#define	PLX_MBOX1		0x44
#define	PLX_MBOX2		0x48
#define	PLX_MBOX3		0x4c
#define	PLX_MBOX4		0x50
#define	PLX_MBOX5		0x54
#define	PLX_MBOX6		0x58
#define	PLX_MBOX7		0x5c

#define	PLX_P2LDBELL		0x60	// PCI-to-Local Doorbell Reg
#define	PLX_L2PDBELL		0x64	// Local-to-PCI Doorbell Reg

#define	PLX_INTCSR		0x68	// Interrupt Control/Status

#	define	PLX_LSERR_ENABLE	0x00000001
#	define	PLX_LSERR_PE		0x00000002
#	define	PLX_SERR		0x00000004
#	define  PLX_MBOX_IE 		0x00000008			
#	undef  PLX_UNUSED /*		0x00000010			*/
#	undef  PLX_UNUSED /*		0x00000020			*/
#	undef  PLX_UNUSED /*		0x00000040			*/
#	undef  PLX_UNUSED /*		0x00000080			*/
#	define PLX_PCI_IE		0x00000100
#	define	PLX_PCI_DOORBELL_IE	0x00000200
#	define	PLX_PCI_ABORT_IE	0x00000400
#	define	PLX_PCI_LOCAL_IE	0x00000800
#	define	PLX_RETRY_ABORT_ENABLE	0x00001000
#	define	PLX_PCI_DOORBELL_INT	0x00002000
#	define	PLX_PCI_ABORT_INT	0x00004000
#	define	PLX_PCI_LOCAL_INT	0x00008000
#	define	PLX_LCL_IE		0x00010000
#	define	PLX_LCL_DOORBELL_IE	0x00020000
#	define	PLX_LCL_DMA0_IE		0x00040000
#	define	PLX_LCL_DMA1_IE		0x00080000
#	define	PLX_LCL_DOORBELL_INT	0x00100000
#	define	PLX_LCL_DMA0_INT	0x00200000
#	define	PLX_LCL_DMA1_INT	0x00400000
#	define	PLX_LCL_BIST_INT	0x00800000
#	define	PLX_BM_DIRECT_		0x01000000
#	define	PLX_BM_DMA0_		0x02000000
#	define	PLX_BM_DMA1_		0x04000000
#	define	PLX_BM_ABORT_		0x08000000
#	define  PLX_MBOX0_INT 		0x10000000
#	define	PLX_MBOX1_INT		0x20000000
#	define  PLX_MBOX2_INT 		0x40000000
#	define  PLX_MBOX3_INT 		0x80000000


#define	PLX_CNTRL		0x6c	/* EEPROM,PCI,User,Init Control/Status */

#	define PLX_PCI_DMA_RD_CMD(x)	(((x) & 0xf) <<  0)
#	define PLX_PCI_DMA_WR_CMD(x)	(((x) & 0xf) <<  4)
#	define PLX_PCI_DIRM_RD_CMD(x)	(((x) & 0xf) <<  8)
#	define PLX_PCI_DIRM_WR_CMD(x)	(((x) & 0xf) << 12)
#	define PLX_USEROUT		0x00010000
#	define PLX_USERIN		0x00020000
#	define PLX_EECK			0x01000000
#	define PLX_EECS			0x02000000
#	define PLX_EEWD			0x04000000
#	define PLX_EERD			0x08000000
#	define PLX_EEPRESENT		0x10000000
#	define PLX_RELOAD_CONFIG	0x20000000
#	define PLX_PCI_SW_RESET		0x40000000
#	define PLX_LCL_INIT_STATUS	0x80000000


#define	PLX_PCIHIDR	0x70		// PCI Permanent Configuration ID Reg
#define	PLX_PCIHREV	0x74		// PCI Permanent Revision ID Reg

/*
 *	DMA registers.  Offset is from PCI side
 */
#define	PLX_DMA0_MODE		0x80

#	define PLX_DMA_MODE_WIDTH32	0x00000003
#	define PLX_DMA_MODE_WAITSTATES(x)	(((x) & 0xf) << 2)
#	define PLX_DMA_MODE_NOREADY	0x00000000
#	define PLX_DMA_MODE_READY	0x00000040
#	define PLX_DMA_MODE_NOBTERM	0x00000000
#	define PLX_DMA_MODE_BTERM	0x00000080
#	define PLX_DMA_MODE_NOBURST	0x00000000
#	define PLX_DMA_MODE_BURST	0x00000100
#	define PLX_DMA_MODE_NOCHAIN	0x00000000
#	define PLX_DMA_MODE_CHAIN	0x00000200
#	define PLX_DMA_MODE_DONE_IE	0x00000400
#	define PLX_DMA_MODE_ADDR_HOLD	0x00000800
#	define PLX_DMA_MODE_DEMAND	0x00001000
#	define PLX_DMA_MODE_WR_AND_INVL	0x00002000	// write and invalidate
#	define PLX_DMA_MODE_EOT_EN	0x00004000
#	define PLX_DMA_MODE_STOP	0x00008000
#	define PLX_DMA_MODE_CLR_CNT_EN	0x00010000
#	define PLX_DMA_MODE_INTR_PCI	0x00020000	// deliver DMA intr to PCI side
#	define PLX_DMA_MODE_INTR_LOCAL	0x00000000	// deliver DMA intr to Local side


#define	PLX_DMA0_PCI_ADDR	0x84	/* non-chaining mode PCI address */

#define	PLX_DMA0_LCL_ADDR	0x88	/* non-chaining mode local address */

#define	PLX_DMA0_SIZE		0x8C	/* non-chaining mode length */

#define	PLX_DMA0_DESCRIPTOR	0x90
#	define	PLX_DMA_DESC_IS_PCI	0x00000001	// desc is in PCI addr space
#	define	PLX_DMA_DESC_IS_LCL	0x00000000 	// desc is in Local addr space
#	define	PLX_DMA_DESC_EOC	0x00000002
#	define	PLX_DMA_DESC_TC_IE	0x00000004
#	define	PLX_DMA_DESC_TO_HOST	0x00000008
#	define	PLX_DMA_DESC_TO_BOARD	0x00000000
#	define	PLX_DMA_DESC_NEXTADDR	0xFFFFfff0


// chaining mode dma descriptor entry
// must be 4 u32 aligned

struct plx_dma_entry {
  u32		pci_addr;		// PCI bus address
  u32		local_addr;		// local (board) bus address
  u32		length;			// bytes
  u32		next_and_flags;		// low 4 bits same as PLX_DMA_DESC_xxxx
};


#define	PLX_DMA1_MODE		0x94
#define	PLX_DMA1_PCI_ADDR	0x98
#define	PLX_DMA1_LCL_ADDR	0x9c
#define	PLX_DMA1_SIZE		0xa0
#define	PLX_DMA1_DESCRIPTOR	0xa4

#define	PLX_DMA0_CSR		0xa8	// N.B., 8 bit register
#define	PLX_DMA1_CSR		0xa9	// N.B., 8 bit register

#	define PLX_DMA_CSR_ENABLE	0x00000001
#	define PLX_DMA_CSR_START	0x00000002
#	define PLX_DMA_CSR_ABORT	0x00000004
#	define PLX_DMA_CSR_CLR_INTR	0x00000008
#	define PLX_DMA_CSR_DONE		0x00000010


#define	PLX_DMA_ARB		0xac	// same as PLX_MARBR

#define	PLX_DMA_THRESH		0xb0

// message queue registers (not defined)

#endif /* _PLX9080_H_ */
