/*
     This file is part of GNUnet.
     (C) 2010 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/gnunet-setup-options.c
 * @brief configuration details
 * @author Christian Grothoff
 */
#include "gnunet-setup-options.h"
#include <gnunet/gnunet_util_lib.h>
#include <gdk/gdkkeysyms.h>

/**
 * Regular expression for YES
 */
#define REX_YES "^YES$"

/**
 * Regular expression for NO
 */
#define REX_NO "^NO$"


/**
 * Initialize a toggle button based on an options 'yes/no' value.
 *
 * @param cls closure
 * @param section section with the value
 * @param option option name
 * @param value value as a string
 * @param widget widget to initialize
 * @param cfg configuration handle
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
load_yes_no (const void *cls,
	     const char *section,
	     const char *option,
	     const char *value,
	     GObject *widget,
	     const struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkToggleButton *button;

  button = GTK_TOGGLE_BUTTON (widget);
  if (button == NULL)
    return GNUNET_SYSERR;
  gtk_toggle_button_set_active (button,
				(0 == strcasecmp (value, "YES")) ? TRUE : FALSE);
  return GNUNET_OK;
}


/**
 * Set a yes/no option based on a toggle button.
 *
 * @param cls closure
 * @param section section with the value
 * @param option option name
 * @param widget widget to initialize
 * @param cfg configuration handle to update
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
save_yes_no (const void *cls,
	     const char *section,
	     const char *option,
	     GObject *widget,
	     struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkToggleButton *button;		       
  gboolean mode;

  button = GTK_TOGGLE_BUTTON (widget);
  if (button == NULL)
    return GNUNET_SYSERR;
  mode = gtk_toggle_button_get_active (button);
  GNUNET_CONFIGURATION_set_value_string (cfg, 
					 section, option,
					 mode == TRUE ? "YES" : "NO");
  return GNUNET_OK;
}


/**
 * Initialize a GtkFileChooser based on a filename option.
 *
 * @param cls closure
 * @param section section with the value
 * @param option option name
 * @param value value as a string
 * @param widget widget to initialize
 * @param cfg configuration handle
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
load_filename (const void *cls,
	       const char *section,
	       const char *option,
	       const char *value,
	       GObject *widget,
	       const struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkFileChooser *chooser;

  chooser = GTK_FILE_CHOOSER (widget);
  if (chooser == NULL)
    return GNUNET_SYSERR;
  gtk_file_chooser_set_filename (chooser,
				 value);
  return GNUNET_OK;
}


/**
 * Set filename option based on a file chooser.
 *
 * @param cls closure
 * @param section section with the value
 * @param option option name
 * @param widget widget to initialize
 * @param cfg configuration handle to update
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
save_filename (const void *cls,
	       const char *section,
	       const char *option,
	       GObject *widget,
	       struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkFileChooser *chooser;
  gchar *fn;

  chooser = GTK_FILE_CHOOSER (widget);
  if (chooser == NULL)
    return GNUNET_SYSERR;
  fn = gtk_file_chooser_get_filename (chooser);
  if (fn == NULL)
    fn = g_strdup ("");
  GNUNET_CONFIGURATION_set_value_string (cfg, 
					 section, option,
					 fn);
  g_free (fn);
  return GNUNET_OK;
}


/**
 * Initialize a GtkEntry based on a text option.
 *
 * @param cls closure
 * @param section section with the value
 * @param option option name
 * @param value value as a string
 * @param widget widget to initialize
 * @param cfg configuration handle
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
load_text (const void *cls,
	   const char *section,
	   const char *option,
	   const char *value,
	   GObject *widget,
	   const struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkEntry *entry;

  entry = GTK_ENTRY (widget);
  if (entry == NULL)
    return GNUNET_SYSERR;
  gtk_entry_set_text (entry,
		      value);
  return GNUNET_OK;
}


/**
 * Set text option based on a GtkEntry.
 *
 * @param cls closure
 * @param section section with the value
 * @param option option name
 * @param widget widget to initialize
 * @param cfg configuration handle to update
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
save_text (const void *cls,
	   const char *section,
	   const char *option,
	   GObject *widget,
	   struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkEntry *entry;
  const gchar *txt;

  entry = GTK_ENTRY (widget);
  if (entry == NULL)
    return GNUNET_SYSERR;
  txt = gtk_entry_get_text (entry);
  GNUNET_CONFIGURATION_set_value_string (cfg, 
					 section, option,
					 txt);
  return GNUNET_OK;
}



/**
 * Initialize a GtkComboBox based on a text option.
 *
 * @param cls closure
 * @param section section with the value
 * @param option option name
 * @param value value as a string
 * @param widget widget to initialize
 * @param cfg configuration handle
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
load_combo_text (const void *cls,
		 const char *section,
		 const char *option,
		 const char *value,
		 GObject *widget,
		 const struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkTreeIter iter;
  GtkComboBox *cb;
  GtkListStore *ls;

  ls = GTK_LIST_STORE (gtk_builder_get_object (builder, 
					       "GNUNET_setup_transport_tcp_internal_ip_liststore"));
  if (ls == NULL)
    {
      GNUNET_break (0);
      return GNUNET_SYSERR;
    }
  gtk_list_store_insert_before (ls, &iter, NULL);
  gtk_list_store_set (ls,
		      &iter,
		      0, value,
		      -1);
  cb = GTK_COMBO_BOX (gtk_builder_get_object (builder, 
					      "GNUNET_setup_transport_tcp_internal_ip_comboboxentry"));
  gtk_combo_box_set_active_iter (cb, &iter);
  return GNUNET_OK;
}


/**
 * Set text option based on a GtkComboBox.
 *
 * @param cls closure
 * @param section section with the value
 * @param option option name
 * @param widget widget to initialize
 * @param cfg configuration handle to update
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
save_combo_text (const void *cls,
		 const char *section,
		 const char *option,
		 GObject *widget,
		 struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkComboBox *cb;
  gchar *c;

  cb = GTK_COMBO_BOX (gtk_builder_get_object (builder, 
					      "GNUNET_setup_transport_tcp_internal_ip_comboboxentry"));
  c = gtk_combo_box_get_active_text (cb);
  GNUNET_CONFIGURATION_set_value_string (cfg, section, option, c);
  g_free (c);
  return GNUNET_OK;
}


/**
 * Initialize a GtkSpinButton based on a numeric option.
 *
 * @param cls closure
 * @param section section with the value
 * @param option option name
 * @param value value as a string
 * @param widget widget to initialize
 * @param cfg configuration handle
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
load_number (const void *cls,
	     const char *section,
	     const char *option,
	     const char *value,
	     GObject *widget,
	     const struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkSpinButton *spin;
  unsigned int number;

  spin = GTK_SPIN_BUTTON (widget);
  if (spin == NULL)
    return GNUNET_SYSERR;
  if (1 != sscanf (value, "%u", &number))
    return GNUNET_SYSERR;
  gtk_spin_button_set_value (spin,
			     number);
  return GNUNET_OK;
}


/**
 * Set numeric option based on a spin button.
 *
 * @param cls closure
 * @param section section with the value
 * @param option option name
 * @param widget widget to initialize
 * @param cfg configuration handle to update
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
save_number (const void *cls,
	     const char *section,
	     const char *option,
	     GObject *widget,
	     struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkSpinButton *spin;

  spin = GTK_SPIN_BUTTON (widget);
  if (spin == NULL)
    return GNUNET_SYSERR;
  GNUNET_CONFIGURATION_set_value_number (cfg, 
					 section, option,
					 gtk_spin_button_get_value_as_int (spin));
  return GNUNET_OK;
}


/**
 * Initialize a toggle button based on the existence of a word
 * in an option value.
 *
 * @param cls word to test for (conat char *)
 * @param section section with the value
 * @param option option name
 * @param value value as a string
 * @param widget widget to initialize
 * @param cfg configuration handle
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
load_option_list (const void *cls,
		  const char *section,
		  const char *option,
		  const char *value,
		  GObject *widget,
		  const struct GNUNET_CONFIGURATION_Handle *cfg)
{
  const char *word = cls;
  char *t;
  char *w;
  GtkToggleButton *button;
  int found;

  button = GTK_TOGGLE_BUTTON (widget);
  if (button == NULL)
    return GNUNET_SYSERR;
  found = GNUNET_NO;
  t = GNUNET_strdup (value);
  w = strtok (t, " ");
  while (w != NULL)
    {
      if (0 == strcmp (w, word))
	{
	  found = GNUNET_YES;
	  break;
	}
      w = strtok (NULL, " ");
    }
  GNUNET_free (t);
  gtk_toggle_button_set_active (button,
				found);
  return GNUNET_OK;
}


/**
 * Add or remove a word from a sentence based on a toggle button's yes/no value.
 *
 * @param cls  word to add or remove for (conat char *)
 * @param section section with the value
 * @param option option name
 * @param widget widget to initialize
 * @param cfg configuration handle to update
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
save_option_list (const void *cls,
		  const char *section,
		  const char *option,
		  GObject *widget,
		  struct GNUNET_CONFIGURATION_Handle *cfg)
{
  const char *word = cls;
  GtkToggleButton *button;
  gboolean mode;

  button = GTK_TOGGLE_BUTTON (widget);
  if (button == NULL)
    return GNUNET_SYSERR;
  mode = gtk_toggle_button_get_active (button);
  if (mode == TRUE)
    GNUNET_CONFIGURATION_append_value_filename (cfg,
						section,
						option,
						word);
  else
    GNUNET_CONFIGURATION_remove_value_filename (cfg,
						section,
						option,
						word);
  return GNUNET_OK;
}



/**
 * User pressed a key in a sensitive tree view with a list store.
 * Check if it was the 'delete' key and if so remove the selected
 * row.
 *
 * @param tv tree view emitting the signal
 * @param event key stroke data
 * @param user_data not used
 * @return TRUE to stop other handlers from being invoked
 */
gboolean
GNUNET_setup_treeview_key_press_event_cb (GtkTreeView *tv,
					  GdkEventKey *event,
					  gpointer user_data)
{
  GtkListStore *ls;
  GtkTreeModel *tm;
  GtkTreeSelection *sel;
  GtkTreeIter iter;
  gboolean editable;

  if ( (event->type != GDK_KEY_PRESS) ||
       (event->state != 0) ||
       (event->keyval != GDK_Delete) )
    return FALSE;
  ls = GTK_LIST_STORE (gtk_tree_view_get_model (tv));
  if (ls == NULL)
    {
      GNUNET_break (0);
      return FALSE;
    }
  sel = gtk_tree_view_get_selection (tv);
  if (TRUE != 
      gtk_tree_selection_get_selected (sel,
				       &tm,
				       &iter))    
    return FALSE;
  gtk_tree_model_get (tm, &iter, 1, &editable, -1);
  if (TRUE == editable)
    return FALSE; /* likely currently editing... */
  gtk_list_store_remove (ls, &iter);
  gtk_tree_model_get_iter_first (tm, &iter);
  gtk_tree_selection_select_iter (sel, &iter);
  return FALSE;
}


/**
 * Initialize a GtkListStore by tokenizing the value into strings.
 *
 * @param cls closure (unused)
 * @param section section with the value
 * @param option option name
 * @param value value as a string
 * @param widget widget to initialize
 * @param cfg configuration handle
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
load_string_list_store (const void *cls,
			const char *section,
			const char *option,
			const char *value,
			GObject *widget,
			const struct GNUNET_CONFIGURATION_Handle *cfg)
{
  char *t;
  char *w;
  GtkListStore *ls;
  GtkTreeIter iter;

  ls = GTK_LIST_STORE (widget);
  if (ls == NULL)
    return GNUNET_SYSERR;
  t = GNUNET_strdup (value);
  w = strtok (t, " ");
  while (w != NULL)
    {
      gtk_list_store_insert_with_values (ls,
					 &iter,
					 G_MAXINT, 
					 0, w,
					 1, FALSE,
					 -1);
      w = strtok (NULL, " ");
    }
  GNUNET_free (t);
  gtk_list_store_insert_with_values (ls,
				     &iter,
				     G_MAXINT, 
				     0, "",
				     1, TRUE,
				     -1);
  return GNUNET_OK;
}


/**
 * Create an option string by merging strings from
 * a GtkListStore.
 *
 * @param cls closure (unused)
 * @param section section with the value
 * @param option option name
 * @param widget widget to initialize
 * @param cfg configuration handle to update
 * @return GNUNET_OK on success, GNUNET_SYSERR if there was a problem
 */
static int
save_string_list_store (const void *cls,
			const char *section,
			const char *option,
			GObject *widget,
			struct GNUNET_CONFIGURATION_Handle *cfg)
{
  GtkTreeModel *tm;
  GtkTreeIter iter;
  char *value;
  char *n;
  gchar *val;

  tm = GTK_TREE_MODEL (widget);
  if (tm == NULL)
    return GNUNET_SYSERR;
  value = NULL;
  if (TRUE == gtk_tree_model_get_iter_first (tm, &iter))
    {
      do
	{
	  gtk_tree_model_get (tm,
			      &iter,
			      0, &val,
			      -1);
	  if (value == NULL)
	    {
	      value = GNUNET_strdup (val);
	    }
	  else
	    {
	      GNUNET_asprintf (&n,
			       "%s %s",			       
			       value,
			       val);
	      GNUNET_free (value);
	      value = n;
	    }
	  g_free (val);
	}
      while (TRUE == gtk_tree_model_iter_next (tm, &iter));
    }
  if (value == NULL)
    value = GNUNET_strdup ("");
  GNUNET_CONFIGURATION_set_value_string (cfg,
					 section,
					 option,
					 value);
  GNUNET_free (value);  
  return GNUNET_OK;
}


/**
 * Hide "min connected friends" option if in F2F-only mode.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_min_connected_friends[] =
{
  { "GNUNET_setup_minimum_friends_label", NULL, REX_YES },
  { "GNUNET_setup_minimum_friends_spinbutton", NULL, REX_YES },
  { NULL, NULL, NULL }
};


/**
 * Hide "hostlist" options if hostlist is not in use.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_hostlist_options[] =
{
  { "GNUNET_setup_hostlist_client_enable_checkbutton", "(^| )hostlist($| )", NULL },
  { "GNUNET_setup_hostlist_client_learn_checkbutton",  "(^| )hostlist($| )", NULL },
  { "GNUNET_setup_hostlist_hbox",  "(^| )hostlist($| )", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "hostlist" server options if hostlist server is not in use.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_hostlist_server_options[] =
{
  { "GNUNET_setup_hostlist_advertise_checkbutton",  "(^| )-p($| )", NULL },
  { "GNUNET_setup_hostlist_port_label",  "(^| )-p($| )", NULL },
  { "GNUNET_setup_hostlist_server_port_spin_button", "(^| )-p($| )", NULL },
  { NULL, NULL, NULL }
};



/**
 * Hide "fs tab" if FS not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_fs_tab[] =
{
  { "GNUNET_setup_fs_scrolledwindow", "(^| )fs($| )", NULL },
  { NULL, NULL, NULL }
};



/**
 * Hide "tcp tab" if TCP not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_tcp_tab[] =
{
  { "GNUNET_setup_transport_tcp_vbox", "(^| )tcp($| )", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "udp tab" if UDP not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_udp_tab[] =
{
  { "GNUNET_setup_transport_udp_vbox", "(^| )udp($| )", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "http tab" if HTTP not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_http_tab[] =
{
  { "GNUNET_setup_transport_http_vbox", "(^| )http($| )", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "https tab" if HTTPS not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_https_tab[] =
{
  { "GNUNET_setup_transport_https_vbox", "(^| )https($| )", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "dv tab" if DV not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_dv_tab[] =
{
  { "GNUNET_setup_transport_dv_vbox", "(^| )dv($| )", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "wlan tab" if WLAN not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_wlan_tab[] =
{
  { "GNUNET_setup_transport_wlan_vbox", "(^| )wlan($| )", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "sqlite datastore" tab if sqlite not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_sqlite_datastore_tab[] =
{
  { "GNUNET_setup_fs_datastore_sqlite_label", "^sqlite$", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "mysql datastore" tab if mysql not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_mysql_datastore_tab[] =
{
  { "GNUNET_setup_datastore_mysql_vbox", "^mysql$", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "postgres datastore" tab if postgres not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_postgres_datastore_tab[] =
{
  { "GNUNET_setup_datastore_postgres_hbox", "^postgres$", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "sqlite datacache" tab if sqlite not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_sqlite_datacache_tab[] =
{
  { "GNUNET_setup_fs_datacache_sqlite_label", "^sqlite$", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "mysql datacache" tab if mysql not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_mysql_datacache_tab[] =
{
  { "GNUNET_setup_datacache_mysql_vbox", "^mysql$", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide "postgres datacache" tab if postgres not active.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_postgres_datacache_tab[] =
{
  { "GNUNET_setup_datacache_postgres_hbox", "^postgres$", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide most TCP options if port is zero.
 */
static struct GNUNET_SETUP_VisibilitySpecification hide_all_tcp_options[] =
{
  { "GNUNET_setup_transport_tcp_adv_port_hbox", NULL, "^0$" },
  { "GNUNET_setup_transport_tcp_autoconfig_hbox", NULL, "^0$" },
  { "GNUNET_setup_transport_tcp_nat_hbox", NULL, "^0$" },
  { "GNUNET_setup_transport_tcp_nat_options_hbox", NULL, "^0$" },
  { NULL, NULL, NULL }
};


/**
 * Hide NATed peer options.
 */
static struct GNUNET_SETUP_VisibilitySpecification toggle_tcp_nat_options[] =
{
  { "GNUNET_setup_transport_tcp_hole_punched_checkbutton", "^YES$", NULL },
  { "GNUNET_setup_transport_tcp_upnp_enable_checkbutton", "^YES$", NULL },
  { "GNUNET_setup_transport_tcp_icmp_server_enable_checkbutton", "^YES$", NULL },
  { "GNUNET_setup_transport_tcp_external_ip_hbox", "^YES$", NULL },
  { NULL, NULL, NULL }
};

/**
 * Hide hole-punched NATed peer options.
 */
static struct GNUNET_SETUP_VisibilitySpecification toggle_tcp_nat_punched_options[] =
{
  { "GNUNET_setup_transport_tcp_upnp_enable_checkbutton", "^NO$", NULL },
  { "GNUNET_setup_transport_tcp_icmp_server_enable_checkbutton", "^NO$", NULL },
  { NULL, NULL, NULL }
};


/**
 * Hide internal IP options.
 */
static struct GNUNET_SETUP_VisibilitySpecification toggle_internal_ip[] =
{
  { "GNUNET_setup_transport_tcp_internal_ip_hbox", "^YES$", NULL },
  { NULL, NULL, NULL }
};


/**
 * Option specification data.
 */
const struct GNUNET_SETUP_OptionSpecification option_specifications[] =
  {

    /* GENERAL TAB */

    { 
      "GNUNET_setup_friends_only_checkbutton",
      "toggled",
      "topology",
      "FRIENDS-ONLY",
      gettext_noop ("Should GNUnet exclusively connect to friends?"),
      "https://gnunet.org/configuration-f2f",
      &load_yes_no,
      &save_yes_no, NULL,
      hide_min_connected_friends
    },    

    { 
      "GNUNET_setup_friends_filechooserbutton",
      "selection-changed",
      "topology",
      "FRIENDS",
      gettext_noop ("Friends file containing the list of friendly peers"),
      "https://gnunet.org/configuration-f2f",
      &load_filename,
      &save_filename, NULL,
      NULL
    },

    { 
      "GNUNET_setup_minimum_friends_spinbutton",
      "value-changed",
      "topology",
      "MINIMUM-FRIENDS",
      gettext_noop ("Minimum number of friendly connections"),
      "https://gnunet.org/configuration-f2f",
      &load_number,
      &save_number, NULL,
      NULL
    },    

    {
      "GNUNET_setup_general_services_topology_checkbutton",
      "toggled",
      "arm",
      "DEFAULTSERVICES",
      gettext_noop ("Topology should always be loaded"),
      "https://gnunet.org/configuration-topology",
      &load_option_list,
      &save_option_list, "topology",
      NULL
    },

    {
      "GNUNET_setup_general_services_hostlist_checkbutton",
      "toggled",
      "arm",
      "DEFAULTSERVICES",
      gettext_noop ("Should hostlist support be started automatically on startup?"),
      "https://gnunet.org/configuration-hostlist",
      &load_option_list,
      &save_option_list, "hostlist",
      hide_hostlist_options
    },


    {
      "GNUNET_setup_general_services_fs_checkbutton",
      "toggled",
      "arm",
      "DEFAULTSERVICES",
      gettext_noop ("Should file-sharing be started automatically on startup?"),
      "https://gnunet.org/configuration-fs",
      &load_option_list,
      &save_option_list, "fs",
      hide_fs_tab
    },

    {
      "GNUNET_setup_general_services_vpn_checkbutton",
      "toggled",
      "arm",
      "DEFAULTSERVICES",
      gettext_noop ("Should the VPN be started automatically on startup?"),
      "https://gnunet.org/configuration-vpn",
      &load_option_list,
      &save_option_list, "vpn",
      NULL /* hide_vpn_tab */
    },

    {
      "GNUNET_setup_hostlist_client_enable_checkbutton",
      "toggled",
      "hostlist",
      "OPTIONS",
      gettext_noop ("Should GNUnet learn about other peers using hostlists"),
      "https://gnunet.org/configuration-hostlist",
      &load_option_list,
      &save_option_list, "-b",
      NULL
    },

    {
      "GNUNET_setup_hostlist_client_learn_checkbutton",
      "toggled",
      "hostlist",
      "OPTIONS",
      gettext_noop ("Should GNUnet learn hostlists from other peers"),
      "https://gnunet.org/configuration-hostlist",
      &load_option_list,
      &save_option_list, "-e",
      NULL
    },

    {
      "GNUNET_setup_hostlist_offer_hostlist_checkbutton",
      "toggled",
      "hostlist",
      "OPTIONS",
      gettext_noop ("Should this peer offer a hostlist to other peers"),
      "https://gnunet.org/configuration-hostlist-server",
      &load_option_list,
      &save_option_list, "-p",
      hide_hostlist_server_options
    },

    {
      "GNUNET_setup_hostlist_advertise_checkbutton",
      "toggled",
      "hostlist",
      "OPTIONS",
      gettext_noop ("Should this peer advertise its hostlist to other peers"),
      "https://gnunet.org/configuration-hostlist-server",
      &load_option_list,
      &save_option_list, "-a",
      NULL
    },

    {
      "GNUNET_setup_hostlist_server_port_spin_button",
      "value-changed",
      "hostlist",
      "HTTPPORT",
      gettext_noop ("Port this peers hostlist should be offered on"),
      "https://gnunet.org/configuration-hostlist-server",
      &load_number,
      &save_number, NULL,
      NULL
    },

    {
      "GNUNET_setup_hostlist_url_liststore",
      "row-changed",
      "hostlist",
      "SERVERS",
      NULL, NULL,
      &load_string_list_store,
      &save_string_list_store, NULL,
      NULL
    },

    {
      "GNUNET_setup_hostlist_url_treeview",
      NULL, NULL, NULL,
      gettext_noop ("Known hostlist URLs"),
      "https://gnunet.org/configuration-hostlist",
      NULL, NULL, NULL, 
      NULL
    },

    {
      "GNUNET_setup_bandwidth_out_spinbutton",
      "value-changed",
      "core",
      "TOTAL_QUOTA_OUT",
      gettext_noop ("How many bytes per second are we allowed to transmit?"),
      "https://gnunet.org/configuration-bandwidth",
      &load_number,
      &save_number, NULL,
      NULL
    },

    {
      "GNUNET_setup_bandwidth_in_spinbutton",
      "value-changed",
      "core",
      "TOTAL_QUOTA_IN",
      gettext_noop ("How many bytes per second are we allowed to receive?"),
      "https://gnunet.org/configuration-bandwidth",
      &load_number,
      &save_number, NULL,
      NULL
    },
   
    /* Transport TAB */

    {
      "GNUNET_setup_transport_tcp_checkbutton",
      "toggled",
      "transport",
      "PLUGINS",
      gettext_noop ("Enable communication via TCP"),
      "https://gnunet.org/configuration-tcp",
      &load_option_list,
      &save_option_list, "tcp",
      hide_tcp_tab
    },

    {
      "GNUNET_setup_transport_udp_checkbutton",
      "toggled",
      "transport",
      "PLUGINS",
      gettext_noop ("Enable communication via UDP"),
      "https://gnunet.org/configuration-udp",
      &load_option_list,
      &save_option_list, "udp",
      hide_udp_tab
    },

    {
      "GNUNET_setup_transport_http_checkbutton",
      "toggled",
      "transport",
      "PLUGINS",
      gettext_noop ("Enable communication via HTTP"),
      "https://gnunet.org/configuration-http",
      &load_option_list,
      &save_option_list, "http",
      hide_http_tab
    },

    {
      "GNUNET_setup_transport_https_checkbutton",
      "toggled",
      "transport",
      "PLUGINS",
      gettext_noop ("Enable communication via HTTPS"),
      "https://gnunet.org/configuration-https",
      &load_option_list,
      &save_option_list, "https",
      hide_https_tab
    },

    {
      "GNUNET_setup_transport_dv_checkbutton",
      "toggled",
      "transport",
      "PLUGINS",
      gettext_noop ("Enable communication via DV"),
      "https://gnunet.org/configuration-dv",
      &load_option_list,
      &save_option_list, "dv",
      hide_dv_tab
    },

    {
      "GNUNET_setup_transport_wlan_checkbutton",
      "toggled",
      "transport",
      "PLUGINS",
      gettext_noop ("Enable communication via WLAN"),
      "https://gnunet.org/configuration-wlan",
      &load_option_list,
      &save_option_list, "wlan",
      hide_wlan_tab
    },

    {
      "GNUNET_setup_transport_tcp_port_spinbutton",
      "value-changed",
      "transport-tcp",
      "PORT",
      gettext_noop ("Port we bind to for TCP"),
      "https://gnunet.org/configuration-tcp",
      &load_number,
      &save_number, NULL,
      hide_all_tcp_options
    },

    {
      "GNUNET_setup_transport_tcp_adv_port_spinbutton",
      "value-changed",
      "transport-tcp",
      "ADVERTISED_PORT",
      gettext_noop ("Port visible to other peers"),
      "https://gnunet.org/configuration-tcp",
      &load_number,
      &save_number, NULL,
      NULL
    },

    {
      "GNUNET_setup_transport_tcp_nat_checkbutton",
      "toggled",
      "transport-tcp",
      "BEHIND_NAT",
      gettext_noop ("Check if this peer is behind a NAT"),
      "https://gnunet.org/configuration-nat",
      &load_yes_no,
      &save_yes_no, NULL,
      toggle_tcp_nat_options
    },

    {
      "GNUNET_setup_transport_tcp_hole_punched_checkbutton",
      "toggled",
      "transport-tcp",
      "PUNCHED_NAT",
      gettext_noop ("Check if the NAT has been hole-punched manually"),
      "https://gnunet.org/configuration-nat",
      &load_yes_no,
      &save_yes_no, NULL,
      toggle_tcp_nat_punched_options
    },

    {
      "GNUNET_setup_transport_tcp_upnp_enable_checkbutton",
      "toggled",
      "transport-tcp",
      "ENABLE_UPNP",
      gettext_noop ("Enable NAT traversal with UPnP/PMP"),
      "https://gnunet.org/configuration-nat",
      &load_yes_no,
      &save_yes_no, NULL,
      NULL,
    },

    {
      "GNUNET_setup_transport_tcp_icmp_server_enable_checkbutton",
      "toggled",
      "transport-tcp",
      "ENABLE_ICMP_SERVER",
      gettext_noop ("Enable NAT traversal with ICMP as server"),
      "https://gnunet.org/configuration-nat",
      &load_yes_no,
      &save_yes_no, NULL,
      NULL,
    },

    {
      "GNUNET_setup_transport_tcp_external_ip_address_entry",
      "changed",
      "transport-tcp",
      "EXTERNAL_ADDRESS",
      gettext_noop ("External (public) IP address of the NAT"),
      "https://gnunet.org/configuration-nat",
      &load_text,
      &save_text, NULL,
      NULL
    },

    {
      "GNUNET_setup_transport_tcp_icmp_client_enable_checkbutton",
      "toggled",
      "transport-tcp",
      "ENABLE_ICMP_CLIENT",
      gettext_noop ("Enable NAT traversal with ICMP as client"),
      "https://gnunet.org/configuration-nat",
      &load_yes_no,
      &save_yes_no, NULL,
      toggle_internal_ip
    },

    {
      "GNUNET_setup_transport_tcp_internal_ip_comboboxentry",
      "changed",
      "transport-tcp",
      "INTERNAL_ADDRESS",
      gettext_noop ("Internal (private) IP address of the NAT"),
      "https://gnunet.org/configuration-nat",
      &load_combo_text,
      &save_combo_text, NULL,
      NULL
    },

    {
      "GNUNET_setup_transport_tcp_disable_ipv6_checkbutton",
      "toggled",
      "transport-tcp",
      "DISABLEV6",
      gettext_noop ("Disable IPv6 support"),
      "https://gnunet.org/configuration-ipv6",
      &load_yes_no,
      &save_yes_no, NULL,
      NULL,
    },

    {
      "GNUNET_setup_transport_udp_port_spinbutton",
      "value-changed",
      "transport-udp",
      "PORT",
      gettext_noop ("Port for inbound UDP packets, use 0 if behind NAT"),
      "https://gnunet.org/configuration-udp",
      &load_number,
      &save_number, NULL,
      NULL
    },

    {
      "GNUNET_setup_transport_http_port_spinbutton",
      "value-changed",
      "transport-http",
      "PORT",
      gettext_noop ("Port for inbound HTTP connections, use 0 if behind NAT"),
      "https://gnunet.org/configuration-http",
      &load_number,
      &save_number, NULL,
      NULL
    },

    {
      "GNUNET_setup_transport_https_port_spinbutton",
      "value-changed",
      "transport-https",
      "PORT",
      gettext_noop ("Port for inbound HTTPS connections, use 0 if behind NAT"),
      "https://gnunet.org/configuration-https",
      &load_number,
      &save_number, NULL,
      NULL
    },

    /* FS TAB */
    
    {
      "GNUNET_setup_fs_datastore_sqlite_radiobutton",
      "toggled",
      "datastore",
      "DATABASE",
      gettext_noop ("Use sqLite to store file-sharing content"),
      "https://gnunet.org/configuration-datastore",
      &load_option_list /* abuse! */,
      &save_option_list /* abuse! */, "sqlite",
      hide_sqlite_datastore_tab
    },

    {
      "GNUNET_setup_fs_datastore_mysql_radiobutton",
      "toggled",
      "datastore",
      "DATABASE",
      gettext_noop ("Use MySQL to store file-sharing content"),
      "https://gnunet.org/configuration-datastore",
      &load_option_list /* abuse! */,
      &save_option_list /* abuse! */, "mysql",
      hide_mysql_datastore_tab
    },

    {
      "GNUNET_setup_fs_datastore_postgres_radiobutton",
      "toggled",
      "datastore",
      "DATABASE",
      gettext_noop ("Use Postgres to store file-sharing content"),
      "https://gnunet.org/configuration-datastore",
      &load_option_list /* abuse! */,
      &save_option_list /* abuse! */, "postgres",
      hide_postgres_datastore_tab
    },

    {
      "GNUNET_setup_datastore_mysql_database_name_entry",
      "changed",
      "datastore-mysql",
      "DATABASE",
      gettext_noop ("Name for the MySQL database"),
      "https://gnunet.org/configuration-datastore",
      &load_text,
      &save_text, NULL,
      NULL
    },

    { 
      "GNUNET_setup_datastore_mysql_config_file_filechooserbutton",
      "selection-changed",
      "datastore-mysql",
      "CONFIG",
      gettext_noop ("Configuration file for MySQL access"),
      "http://dev.mysql.com/doc/refman/5.5/en/option-files.html",
      &load_filename,
      &save_filename, NULL,
      NULL
    },

    {
      "GNUNET_setup_datastore_mysql_username_entry",
      "changed",
      "datastore-mysql",
      "USER",
      gettext_noop ("Username for MySQL access"),
      "https://gnunet.org/configuration-datastore",
      &load_text,
      &save_text, NULL,
      NULL
    },

    {
      "GNUNET_setup_datastore_mysql_password_entry",
      "changed",
      "datastore-mysql",
      "PASSWORD",
      gettext_noop ("Password for MySQL access"),
      "https://gnunet.org/configuration-datastore",
      &load_text,
      &save_text, NULL,
      NULL
    },

    {
      "GNUNET_setup_datastore_mysql_hostname_entry",
      "changed",
      "datastore-mysql",
      "HOST",
      gettext_noop ("Name of host running MySQL database"),
      "https://gnunet.org/configuration-datastore",
      &load_text,
      &save_text, NULL,
      NULL
    },

    {
      "GNUNET_setup_datastore_mysql_port_spinbutton",
      "value-changed",
      "datastore-mysql",
      "PORT",
      gettext_noop ("Port of MySQL database"),
      "https://gnunet.org/configuration-datastore",
      &load_number,
      &save_number, NULL,
      NULL
    },

    {
      "GNUNET_setup_datastore_postgres_config_entry",
      "changed",
      "datastore-postgres",
      "CONFIG",
      gettext_noop ("Configuration for Postgres (passed to PQconnectdb)"),
      "http://www.postgresql.org/docs/8.1/static/libpq.html#LIBPQ-CONNECT",
      &load_text,
      &save_text, NULL,
      NULL
    },

    
    { 
      "GNUNET_setup_fs_migration_from_checkbutton",
      "toggled",
      "fs",
      "CONTENT_PUSHING",
      gettext_noop ("Should we try to push our content to other peers?"),
      "https://gnunet.org/configuration-fs",
      &load_yes_no,
      &save_yes_no, NULL,
      NULL
    },    

    { 
      "GNUNET_setup_fs_migration_to_checkbutton",
      "toggled",
      "fs",
      "CONTENT_CACHING",
      gettext_noop ("Are we allowed to cache content received from other peers?"),
      "https://gnunet.org/configuration-fs",
      &load_yes_no,
      &save_yes_no, NULL,
      NULL
    },    
    
    {
      "GNUNET_setup_fs_datacache_sqlite_radiobutton",
      "toggled",
      "dhtcache",
      "DATABASE",
      gettext_noop ("Use sqLite to cache DHT data"),
      "https://gnunet.org/configuration-datacache",
      &load_option_list /* abuse! */,
      &save_option_list /* abuse! */, "sqlite",
      hide_sqlite_datacache_tab
    },

    {
      "GNUNET_setup_fs_datacache_mysql_radiobutton",
      "toggled",
      "dhtcache",
      "DATABASE",
      gettext_noop ("Use MySQL to cache DHT data"),
      "https://gnunet.org/configuration-datacache",
      &load_option_list /* abuse! */,
      &save_option_list /* abuse! */, "mysql",
      hide_mysql_datacache_tab
    },

    {
      "GNUNET_setup_fs_datacache_postgres_radiobutton",
      "toggled",
      "dhtcache",
      "DATABASE",
      gettext_noop ("Use Postgres to cache DHT data"),
      "https://gnunet.org/configuration-datacache",
      &load_option_list /* abuse! */,
      &save_option_list /* abuse! */, "postgres",
      hide_postgres_datacache_tab
    },
  
    {
      "GNUNET_setup_datacache_mysql_database_name_entry",
      "changed",
      "datacache-mysql",
      "DATABASE",
      gettext_noop ("Name for the MySQL database"),
      "https://gnunet.org/configuration-datacache",
      &load_text,
      &save_text, NULL,
      NULL
    },

    { 
      "GNUNET_setup_datacache_mysql_config_file_filechooserbutton",
      "selection-changed",
      "datacache-mysql",
      "CONFIG",
      gettext_noop ("Configuration file for MySQL access"),
      "http://dev.mysql.com/doc/refman/5.5/en/option-files.html",
      &load_filename,
      &save_filename, NULL,
      NULL
    },

    {
      "GNUNET_setup_datacache_mysql_username_entry",
      "changed",
      "datacache-mysql",
      "USER",
      gettext_noop ("Username for MySQL access"),
      "https://gnunet.org/configuration-datacache",
      &load_text,
      &save_text, NULL,
      NULL
    },

    {
      "GNUNET_setup_datacache_mysql_password_entry",
      "changed",
      "datacache-mysql",
      "PASSWORD",
      gettext_noop ("Password for MySQL access"),
      "https://gnunet.org/configuration-datacache",
      &load_text,
      &save_text, NULL,
      NULL
    },

    {
      "GNUNET_setup_datacache_mysql_hostname_entry",
      "changed",
      "datacache-mysql",
      "HOST",
      gettext_noop ("Name of host running MySQL database"),
      "https://gnunet.org/configuration-datacache",
      &load_text,
      &save_text, NULL,
      NULL
    },

    {
      "GNUNET_setup_datacache_mysql_port_spinbutton",
      "value-changed",
      "datacache-mysql",
      "PORT",
      gettext_noop ("Port of MySQL database"),
      "https://gnunet.org/configuration-datacache",
      &load_number,
      &save_number, NULL,
      NULL
    },

    {
      "GNUNET_setup_datacache_postgres_config_entry",
      "changed",
      "datacache-postgres",
      "CONFIG",
      gettext_noop ("Configuration for Postgres (passed to PQconnectdb)"),
      "http://www.postgresql.org/docs/8.1/static/libpq.html#LIBPQ-CONNECT",
      &load_text,
      &save_text, NULL,
      NULL
    },
    
    /* END of list */

    { NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL }
  };

/* end of gnunet-setup-options.c */
