/*
     This file is part of GNUnet
     (C) 2005, 2006, 2010 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/main_window_file_search.c
 * @author Christian Grothoff
 */
#include "common.h"

/**
 * Builder used for the search dialog.
 */
static GtkBuilder *builder;


/**
 * User double-clicked on namespace or pressed enter;
 * move namespace root to the 'keywords' line.
 */
void 
GNUNET_GTK_search_dialog_namespace_tree_view_row_activated_cb (GtkTreeView       *tree_view,
							       GtkTreePath       *path,
							       GtkTreeViewColumn *column,
							       gpointer           user_data)
{
  GtkTreeModel *ls;
  GtkTreeSelection *sel;
  GtkEntry *query;
  GtkTreeIter iter;
  char *root;

  query = GTK_ENTRY (gtk_builder_get_object (builder,
					     "GNUNET_GTK_search_dialog_keyword_entry"));
  sel = gtk_tree_view_get_selection (tree_view); 
  if (TRUE ==
      gtk_tree_selection_get_selected (sel,
				       &ls,
				       &iter))
    {
      gtk_tree_model_get (ls, &iter,
			  1, &root,
			  -1);
      gtk_entry_set_text (query,
			  root);
      GNUNET_free (root);
    }
}


void
GNUNET_GTK_search_dialog_search_button_clicked_cb (GtkWidget * dummy, 
						   gpointer data)
{
  GtkEntry *query;
  GtkSpinButton *anonymity;
  GtkTreeView *namespace;
  const char *keywords;
  char *emsg;
  struct GNUNET_FS_Uri *uri;
  GNUNET_HashCode *nsid;
  GtkTreeModel *ls;
  GtkTreeView *tv;
  GtkTreeSelection *sel;
  GtkTreeIter iter;

  query = GTK_ENTRY (gtk_builder_get_object (builder,
					     "GNUNET_GTK_search_dialog_keyword_entry"));
  anonymity = GTK_SPIN_BUTTON (gtk_builder_get_object (builder,
						 "GNUNET_GTK_search_dialog_anonymity_spin_button"));
  namespace = GTK_TREE_VIEW (gtk_builder_get_object (builder,
						     "GNUNET_GTK_search_dialog_namespace_tree_view"));
  keywords = gtk_entry_get_text (query);
  nsid = NULL;
  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_search_dialog_namespace_tree_view"));
  sel = gtk_tree_view_get_selection (tv); 
  if (TRUE ==
      gtk_tree_selection_get_selected (sel,
				       &ls,
				       &iter))
    gtk_tree_model_get (ls, &iter,
			2, &nsid,
			-1);
  if (nsid != NULL)
    {
      uri = GNUNET_FS_uri_sks_create_from_nsid (nsid,
						keywords);    
      GNUNET_assert (uri != NULL);
    }
  else    
    {
      emsg = NULL;
      uri = GNUNET_FS_uri_ksk_create (keywords, &emsg);    
      if (uri == NULL)
	{
	  GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
		      _("Invalid keyword string `%s': %s"),
		      keywords,
		      emsg);
	  GNUNET_free_non_null (emsg);
	  return;
	}
    }
  GNUNET_FS_search_start (GNUNET_GTK_get_fs_handle (),
			  uri,
			  gtk_spin_button_get_value_as_int (anonymity),
			  GNUNET_FS_SEARCH_OPTION_NONE,
			  NULL);
  GNUNET_FS_uri_destroy (uri);
}



/**
 * Add pseudonym data to list store
 *
 * @param cls closure (the 'GtkListStore')
 * @param pseudonym hash code of public key of pseudonym
 * @param md meta data known about the pseudonym
 * @param rating the local rating of the pseudonym
 * @return GNUNET_OK to continue iteration, GNUNET_SYSERR to abort
 */
static int
add_namespace_to_ls (void *cls,
		     const GNUNET_HashCode *
		     pseudonym,
		     const struct
		     GNUNET_CONTAINER_MetaData * md,
		     int rating)
{
  GtkListStore *ls = cls;
  char *root;
  char *ns_name;
  GNUNET_HashCode *nsid;
  char *description;
  char *uris;
  char *emsg;
  struct GNUNET_FS_Uri *uri;
  GtkTreeIter iter;
  
  ns_name = GNUNET_PSEUDONYM_id_to_name (GNUNET_GTK_get_configuration (),
					 pseudonym);
  nsid = GNUNET_malloc (sizeof (GNUNET_HashCode));
  *nsid = *pseudonym;
  root = NULL;
  uris = GNUNET_CONTAINER_meta_data_get_by_type (md,
						 EXTRACTOR_METATYPE_URI);
  if (uris != NULL)
    {
      emsg = NULL;
      uri = GNUNET_FS_uri_parse (uris, &emsg);
      if (uri == NULL)
	GNUNET_free (emsg);
      root = GNUNET_FS_uri_sks_get_content_id (uri);
      GNUNET_FS_uri_destroy (uri);
    }
  description = GNUNET_CONTAINER_meta_data_get_first_by_types (md,
							       EXTRACTOR_METATYPE_TITLE,
							       EXTRACTOR_METATYPE_BOOK_TITLE,
							       EXTRACTOR_METATYPE_DESCRIPTION,
							       EXTRACTOR_METATYPE_SUMMARY,
							       EXTRACTOR_METATYPE_ALBUM,
							       EXTRACTOR_METATYPE_COMMENT,
							       EXTRACTOR_METATYPE_SUBJECT,
							       EXTRACTOR_METATYPE_KEYWORDS,
							       -1);
  gtk_list_store_insert_with_values (ls,
				     &iter,
				     G_MAXINT,
				     0, ns_name,
				     1, root,
				     2, nsid,
				     3, description,
				     -1);
  GNUNET_free (ns_name);
  GNUNET_free_non_null (root);
  GNUNET_free_non_null (description);
  return GNUNET_OK;
}


/**
 * Search selected in 'file' menu.
 */
void
GNUNET_GTK_main_menu_file_search_activate_cb (GtkWidget * dummy, 
					      gpointer data)
{
  GtkWidget *ad;
  GtkListStore *ls;
  GtkTreeIter iter;
  GNUNET_HashCode *nsid;
  
  GNUNET_assert (builder == NULL);
  builder = GNUNET_GTK_get_new_builder ("search_dialog.glade");
  if (builder == NULL)
    return;
  ad = GTK_WIDGET (gtk_builder_get_object (builder,
					   "GNUNET_GTK_search_dialog"));
  ls = GTK_LIST_STORE (gtk_builder_get_object (builder,
					       "GNUNET_GTK_namespace_list_store"));
  GNUNET_PSEUDONYM_list_all (GNUNET_GTK_get_configuration (),
			     &add_namespace_to_ls,
			     ls);
  gtk_dialog_run (GTK_DIALOG (ad));

  /* free nsids from 'ls' */
  if (TRUE ==
      gtk_tree_model_get_iter_first (GTK_TREE_MODEL (ls), 
				     &iter))
    {
      do
	{
	  gtk_tree_model_get (GTK_TREE_MODEL (ls), 
			      &iter,
			      2, &nsid,
			      -1);
	  GNUNET_free (nsid);
	}
      while (TRUE == gtk_tree_model_iter_next (GTK_TREE_MODEL (ls), 
					       &iter));
    }
  gtk_widget_destroy (ad);
  g_object_unref (G_OBJECT (builder));
  builder = NULL;
}

/* end of main_window_file_search.c */
