# GNU Enterprise Forms - wx 2.6 UI Driver - Box widget
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: box.py 9956 2009-10-11 18:54:57Z reinhard $
"""
A <box> is a visual container for positioned layouts.  All child widgets of a
box use relative positions where the top-left position within the box is 0/0.
"""

import wx

from gnue.forms.uidrivers.wx26.widgets import _base

__all__ = ['UIBox']


# =============================================================================
# Interface implementation for a box widget
# =============================================================================

class UIBox(_base.UIHelper):
    """
    Interface implementation for the <box> tag
    """

    # -------------------------------------------------------------------------
    # Create a wx box widget
    # -------------------------------------------------------------------------

    def _create_widget_(self, event, spacer):
        """
        Create the wx.StaticBoxSizer for the box and add it to the owner.
        The spacer is ignored for box tags.

        @param event: the creation-event instance carrying information like
            container (parent-widget)
        @param spacer: not used for pages

        @returns: the wx.BoxSizer instance containing the StaticBoxSizer of
            this box which in turn contains the GridBagSizer for all the child
            widgets.
        """

        top, left = self.chr_pos
        height, width = self.chr_span

        owner = self.getParent()
        parent = event.container

        self._container = parent

        # the border
        box_title = wx.StaticBox(parent, -1, self._gfObject.label)

        # a vbox linked to the border
        box = wx.StaticBoxSizer(box_title, wx.VERTICAL)

        # a gridbag that will contain all the widgets inside the box
        self._sizer = wx.GridBagSizer()
        self._sizer.SetEmptyCellSize((self._uiDriver.cellWidth,
                self._uiDriver.cellHeight))
        # make the gridbag as big as we need it, independent from the number of
        # occupied rows and columns
        self._sizer.SetMinSize((self._uiDriver.cellWidth * (width - 2),
                self._uiDriver.cellHeight * (height - 2)))
        # leave some border on top
        half = self._uiDriver.control_border('label')
        box.Add((half, half))
        box.Add(self._sizer, 1)

        # the vertical box which will contain the box
        outer = wx.BoxSizer(wx.VERTICAL)
        outer.Add((half, half))
        outer.Add(box, 1, wx.ALIGN_CENTER_HORIZONTAL)
        outer.Add((1, self._uiDriver.cellHeight / 2))

        self.widget = outer
        owner.add_widgets(self, 0)

        return self._container


    # -------------------------------------------------------------------------
    # Add child widgets to the sizer
    # -------------------------------------------------------------------------

    def add_widgets(self, ui_widget, spacer, border=0):
        """
        Add a given UI widget to the page.

        @param ui_widget: widget to add to the page
        @param spacer: not used for boxes
        """

        item = ui_widget.widget
        flags = wx.EXPAND

        if not ui_widget.is_growable():
            box = wx.BoxSizer(wx.HORIZONTAL)
            box.Add(item, 1, wx.ALIGN_CENTER_VERTICAL)
            item = box

        if border:
            flags |= wx.TOP | wx.BOTTOM

        self._sizer.Add(item, ui_widget.chr_pos, ui_widget.chr_span, flags,
                border)


# =============================================================================
# Configuration data
# =============================================================================

configuration = {
  'baseClass': UIBox,
  'provides' : 'GFBox',
  'container': 0
}
