/* cmd-archives.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/cmd/main.h"
#include "tla/libfsutils/ensure-dir.h"
#include "tla/libarch/my.h"
#include "tla/libarch/namespace.h"
#include "tla/libarch/archives.h"
#include "tla/libarch/cmd-archives.h"



static t_uchar * usage = "[options] [search regular expression]";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
                                   "\n"
                                   "Copyright 2003 Tom Lord\n"
                                   "\n"
                                   "This is free software; see the source for copying conditions.\n"
                                   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
                                   "PARTICULAR PURPOSE.\n"
                                   "\n"
                                   "Report bugs to " cfg__tla_bug_mail ".\n"
                                   "\n"
                                   cfg__std__release_id_string
                                   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.") \
  OP (opt_names, "n", "names", 0, \
      "print archive names only")

t_uchar arch_cmd_archives_help[] = ("Report registered archives and their locations.\n"
				    "Print a list of registered archives and their locations\n"
				    "\n"
				    "If [search regex] is given then only archives with names\n"
				    "that match [search regex] will be shown\n");

enum options
{
  OPTS (OPT_ENUM, OPT_IGN)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_archives (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  int names_only;

  names_only = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, arch_cmd_archives_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_names:
          {
            names_only = 1;
            break;
          }

        case opt_double_dash:
          {
            goto no_more_args;
          }
        }
    }

 no_more_args:

  if (argc > 2)
    goto usage_error;

  {
    rel_table list = 0;
    int x;
    int re_error;
    regex_t archive_needle;

    list = arch_registered_archives ();
    rel_sort_table_by_field (0, list, 0);

    if (argc == 2)
      re_error = regcomp(&archive_needle, argv[1], REG_EXTENDED);
    else
      re_error = regcomp(&archive_needle, ".*", REG_EXTENDED);

    if (re_error)
              panic("Invalid regular expression given.");

    for (x = 0; x < rel_n_records (list); ++x)
      {
        if (!regexec (&archive_needle, list[x][0], 0, 0, 0))
          {
            safe_printfmt (1, "%s\n", list[x][0]);
            if (!names_only)
              safe_printfmt (1, "    %s\n", list[x][1]);
          }
      }
    regfree(&archive_needle);
  }

  return 0;
}




/* tag: Tom Lord Sun May 18 22:09:05 2003 (ls-archives.c)
 */
