/* cmd-update.c:
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */




#include "config-options.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/fs/cwd.h"
#include "hackerlab/vu/vu-dash.h"
#include "tla/libfsutils/tmp-files.h"
#include "tla/libfsutils/rmrf.h"
#include "tla/libfsutils/copy-file.h"
#include "tla/libfsutils/dir-as-cwd.h"
#include "tla/libarch/namespace.h"
#include "tla/libarch/project-tree.h"
#include "tla/libarch/my.h"
#include "tla/libarch/copy-project-tree.h"
#include "tla/libarch/archive.h"
#include "tla/libarch/patch-logs.h"
#include "tla/libarch/local-cache.h"
#include "tla/libarch/cmd.h"
#include "tla/libarch/cmd-replay.h"
#include "tla/libarch/cmd-deltapatch.h"
#include "tla/libarch/cmd-undo-changes.h"
#include "tla/libarch/cmd-redo-changes.h"
#include "tla/libarch/cmd-update.h"



static t_uchar * usage = "[options] dir [newdir] [version/revision]";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
                                   "\n"
                                   "Copyright 2003 Tom Lord\n"
                                   "\n"
                                   "This is free software; see the source for copying conditions.\n"
                                   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
                                   "PARTICULAR PURPOSE.\n"
                                   "\n"
                                   "Report bugs to " cfg__tla_bug_mail ".\n"
                                   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.") \
  OP (opt_archive, "A", "archive", 1, \
      "Override `my-default-archive'") \
  OP (opt_in_place, 0, "in-place", 0, \
      "modify DIR directly")



t_uchar arch_cmd_update_help[] = ("update a project tree to reflect recent archived changes\n"
                                  "Construct a new project tree NEW-DIR for the working directory\n"
                                  "containing OLD-DIR.\n"
                                  "\n"
                                  "The new project tree is formed by getting the latest revision\n"
                                  "of VERSION (or the default version of OLD-DIR) and then applying a\n"
                                  "patch set of the differences between OLD-DIR and the highest revision\n"
                                  "with which OLD-DIR is up-to-date.\n"
                                  "\n"
                                  "If the merge involves conflicts, a warning message is printed, and the\n"
                                  "new project tree will contain \".rej\" files.\n");

enum options
{
  OPTS (OPT_ENUM, OPT_IGN)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_update (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;
  int in_place = 0;
  int exit_status = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, arch_cmd_update_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_archive:
          {
            lim_free (0, default_archive);
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_in_place:
          {
            in_place = 1;
            break;
          }
        }
    }

  if (in_place)
    {
      if ((argc < 2) || (argc > 3))
        goto usage_error;
    }
  else
    {
      if ((argc < 3) || (argc > 4))
        goto usage_error;
    }

  if (default_archive && !arch_valid_archive_name (default_archive))
    {
      safe_printfmt (2, "update: invalid archive name (%s)\n", default_archive);
      exit (1);
    }


  {
    t_uchar * upon;
    t_uchar * upon_root = 0;
    t_uchar * dest;
    t_uchar * vsnspec = 0;
    t_uchar * archive = 0;
    t_uchar * version = 0;
    struct arch_archive * arch;
    t_uchar * upon_level = 0;
    rel_table archive_levels = 0;
    t_uchar * archive_level = 0;

    upon = argv[1];
    upon_root = arch_tree_root (0, upon, 0);

    if (!upon_root)
      {
        safe_printfmt (2, "update: dir not in a project tree (%s)\n", upon);
        exit (1);
      }

    if (in_place)
      {
        dest = upon;
        if (argc > 2)
          vsnspec = str_save (0, argv[2]);
      }
    else
      {
        dest = argv[2];
        if (argc > 3)
          vsnspec = str_save (0, argv[3]);
      }

    if (!vsnspec)
      {
        vsnspec = arch_tree_version (upon_root);
        if (!vsnspec)
          {
            safe_printfmt (2, "update: tree has no default version (%s)\n", upon_root);
            exit (1);
          }
      }

    if (!arch_valid_package_name (vsnspec, arch_maybe_archive, arch_req_version, 0))
      {
        safe_printfmt (2, "update: illegal version name (%s)\n", vsnspec);
        exit (1);
      }

    archive = arch_parse_package_name (arch_ret_archive, default_archive, vsnspec);
    version = arch_parse_package_name (arch_ret_non_archive, 0, vsnspec);

    arch = arch_archive_connect (archive, 0);
    invariant (!!arch);

    upon_level = arch_highest_patch_level (upon_root, archive, version);

    if (!upon_level)
      {
        safe_printfmt (2, "update: tree has no common history with version\n    tree: %s\n    version: %s\n", upon_root, vsnspec);
        exit (1);
      }


    archive_levels = arch_archive_revisions (arch, version, 0);
    if (!archive_levels)
      {
        safe_printfmt (2, "update: (WEIRD ERROR!) archive has no revisions for version (%s/%s)\n", archive, version);
        exit (1);
      }

    if (!in_place)
      {
        safe_printfmt (1, "* copying %s to %s\n", upon_root, dest);
        safe_flush (1);
        arch_copy_project_tree (upon_root, dest, 1, 1);
      }

    if (!str_cmp (upon_level, archive_levels[rel_n_records (archive_levels) - 1][0]))
      {
        safe_printfmt (1, "* tree is already up to date\n");
        safe_flush (1);
      }
    else
      {
        t_uchar * ancestor_rev = 0;
        t_uchar * latest_rev = 0;
        t_uchar * local_ancestor = 0;
        t_uchar * local_latest = 0;
        int use_replay_strategy = 1;

        ancestor_rev = str_alloc_cat_many (0, version, "--", upon_level, str_end);
        latest_rev = str_alloc_cat_many (0, version, "--", archive_levels[rel_n_records (archive_levels) - 1][0], str_end);

        local_ancestor = arch_find_local_copy (upon_root, 0, archive, ancestor_rev);
        local_latest = arch_find_local_copy (upon_root, 0, archive, latest_rev);

        if (local_ancestor && local_latest)
          use_replay_strategy = 0;

        if (use_replay_strategy)
          {
            enum arch_revision_type type;
            int x;

            /* are we sure we really can?
             */

            for (x = 0; x < rel_n_records (archive_levels); ++x)
              if (!str_cmp (upon_level, archive_levels[x][0]))
                break;
            invariant (x < rel_n_records (archive_levels));

            while (use_replay_strategy && (x < rel_n_records (archive_levels)))
              {
                t_uchar * this_rev = 0;

                this_rev = str_alloc_cat_many (0, version, "--", archive_levels[x][0], str_end);
                arch_revision_type (&type, 0 , arch, this_rev);

                if (type != arch_simple_revision)
                  {
                    use_replay_strategy = 0;
                    break;
                  }

                lim_free (0, this_rev);
                ++x;
              }
          }

        safe_printfmt (1, "* setting aside local changes temporarily\n");
        safe_flush (1);
        arch_call_cmd (arch_cmd_undo_changes, "update", "-A", archive, "-d", dest, ancestor_rev, 0);

        safe_printfmt (1, "* updating for new patches in archive\n");
        safe_flush (1);
        if (use_replay_strategy)
          {
            /* could be fancy here and binary search for local copies :-)
             */
            arch_call_cmd (arch_cmd_replay, "update", "--in-place", "--new", "-A", archive, dest, version, 0);
          }
        else
          {
            arch_call_cmd (arch_cmd_deltapatch, "update", "--in-place", "-A", archive, ancestor_rev, latest_rev, dest, 0);
          }

        safe_printfmt (1, "* reapplying local changes\n");
        safe_flush (1);
        arch_call_cmd (arch_cmd_redo_changes, "update", "-d", dest, 0);


        lim_free (0, ancestor_rev);
        lim_free (0, latest_rev);
      }

    lim_free (0, upon_root);
    lim_free (0, vsnspec);
    lim_free (0, archive);
    lim_free (0, version);
    arch_archive_close (arch);
    lim_free (0, upon_level);
    rel_free_table (archive_levels);
    lim_free (0, archive_level);
  }


  lim_free (0, default_archive);

  if (exit_status)
    {
      safe_printfmt (2, "\nupdate: conflicts occured during update\n");
    }

  return exit_status;
}




/* tag: Tom Lord Wed Jun  4 22:39:12 2003 (cmd-update.c)
 */
