#!/bin/sh
# Check the database for old lock files or index inconsistencies
# Copyright (C) 1993 Free Software Foundation, Inc.
# Contributed by Jonathan Kamens (jik@security.ov.com).
#
# This file is part of GNU GNATS.
#
# GNU GNATS is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# GNU GNATS is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with GNU GNATS; see the file COPYING.  If not, write to
# the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

# This script takes no arguments.  It attempts to lock the GNATS
# database for five minutes; if it fails, it sends a mail message
# notifying the administrator of the failure and exits.
# 
# Once the database is locked, the script searches the database for
# lock files that are more than 24 hours old.  Any old lock files are
# reported to the administrator in a mail message.
# 
# After checking for old lock files, it calls gen-index and compares
# the results with gnats-adm/index; any inconsistencies are reported
# to the administrators in a mail message.
# 
# After checking the index file for inconsistencies, the script
# unlocks the database and exits.

GNATS_ROOT=xGNATS_ROOTx
GNATS_ADMIN=xGNATS_ADMINx
LIBDIR=xLIBDIRx/gnats
MAIL_AGENT="xMAIL_AGENTx"

PATH=$LIBDIR:$GNATS_ROOT/gnats-bin:/bin:/usr/bin; export PATH
TMPDIR=${TMPDIR-/tmp}
TMPFILE=$TMPDIR/gnats-check-db-$$

# 
# First, try to lock the database
#

i=0
NOTLOCKED=true
while [ $i -lt 30 ]; do
	if $LIBDIR/pr-edit --lockdb; then
		NOTLOCKED=false
		break
	fi
	i=`expr $i + 1`
	sleep 10
done

if $NOTLOCKED; then
	$MAIL_AGENT<<EOF
To: $GNATS_ADMIN
Subject: $0: can't lock database

Unable to continue database check, because database in
$GNATS_ROOT could not be locked for five minutes.
EOF
	exit 1
fi

#
# Now, check for old lock files
#

find $GNATS_ROOT -type f -name '[0-9]*.lock' -mtime +1 -print > $TMPFILE
if [ -s $TMPFILE ]; then
	cat - $TMPFILE <<EOF | $MAIL_AGENT
To: $GNATS_ADMIN
Subject: $0: found old lock files

The following lock files in the database $GNATS_ROOT
are more than a day old:

EOF
fi

#
# Now, check for inconsistencies in the index file
#

sort -t/ +1n $GNATS_ROOT/gnats-adm/index > $TMPFILE
gen-index --numerical > $TMPFILE.2
if diff $TMPFILE $TMPFILE.2 > $TMPFILE.3; then
	true
else
	cat - $TMPFILE.3 <<EOF | $MAIL_AGENT
To: $GNATS_ADMIN
Subject: $0: possible inconsistencies in database index

The following possible inconsistencies were found in
$GNATS_ROOT/gnats-adm/index.

Lines prefixed by '<' are from the current index file.  Lines
prefixed by '>' are from a fresh index generated with
$LIBDIR/gen-index.

EOF
fi

rm -f ${TMPFILE}*
$LIBDIR/pr-edit --unlockdb
exit 0
