/*
 * gleem -- OpenGL Extremely Easy-To-Use Manipulators.
 * Copyright (C) 1998 Kenneth B. Russell (kbrussel@media.mit.edu)
 * See the file LICENSE.txt in the doc/ directory for licensing terms.
 */

#include <math.h>
#include <gleem/HandleBoxManip.h>
#include <gleem/ManipPartTransform.h>
#include <gleem/ManipPartHollowCubeFace.h>
#include <gleem/ManipPartCube.h>
#include <gleem/ManipPartSquare.h>

GLEEM_USE_NAMESPACE

HandleBoxManip::HandleBoxManip()
{
  translation.setValue(0, 0, 0);
  scale.setValue(1, 1, 1);
  geometryScale.setValue(1, 1, 1);
  // FIXME
  dragState = INACTIVE;
  createGeometry();
  recalc();
}

HandleBoxManip::~HandleBoxManip()
{
  deleteGeometry();
}

void
HandleBoxManip::setTranslation(const GleemV3f &translation)
{
  this->translation = translation;
  recalc();
}

const GleemV3f &
HandleBoxManip::getTranslation() const
{
  return translation;
}

void
HandleBoxManip::setRotation(const GleemRot &rotation)
{
  this->rotation = rotation;
  recalc();
}

const GleemRot &
HandleBoxManip::getRotation() const
{
  return rotation;
}

void
HandleBoxManip::setScale(const GleemV3f &scale)
{
  this->scale = scale;
  recalc();
}

const GleemV3f &
HandleBoxManip::getScale() const
{
  return scale;
}

void
HandleBoxManip::setGeometryScale(const GleemV3f &geometryScale)
{
  this->geometryScale = geometryScale;
  recalc();
}

const GleemV3f &
HandleBoxManip::getGeometryScale() const
{
  return geometryScale;
}

void
HandleBoxManip::render()
{
  // FIXME: Add scale handles
  int i;
  for (i = 0; i < faces.size(); i++)
    faces[i].geometry->render();

  for (i = 0; i < rotateHandles.size(); i++)
    rotateHandles[i].geometry->render();
}

void
HandleBoxManip::intersectRay(const GleemV3f &rayStart,
			     const GleemV3f &rayDirection,
			     vector<HitPoint> &results)
{
  // FIXME: add scale handles
  int i;
  for (i = 0; i < faces.size(); i++)
    faces[i].geometry->intersectRay(rayStart, rayDirection, results);
  for (i = 0; i < rotateHandles.size(); i++)
    rotateHandles[i].geometry->intersectRay(rayStart, rayDirection, results);
}

void
HandleBoxManip::highlight(const HitPoint &hit)
{
  highlightedGeometry = hit.manipPart;
  assert(highlightedGeometry != NULL);
  highlightedGeometry->highlight();
}

void
HandleBoxManip::clearHighlight()
{
  highlightedGeometry->clearHighlight();
  highlightedGeometry = NULL;
}

void
HandleBoxManip::makeActive(const HitPoint &hit)
{
  // Find which piece of geometry it is
  int i;
  for (i = 0; i < faces.size(); i++)
    {
      if (faces[i].geometry == hit.manipPart)
	{
	  dragState = TRANSLATE;
	  dragPlane.setPoint(hit.intPt);
	  dragPlane.setNormal(faces[i].normal);
	  dragOffset = translation - hit.intPt;
	  draggedGeometry = faces[i].geometry;
	  draggedGeometry->highlight();
	  return;
	}      
    }

  for (i = 0; i < rotateHandles.size(); i++)
    {
      RotateHandleInfo &rotInfo = rotateHandles[i];
      if (rotInfo.geometry == hit.manipPart)
	{
	  dragState = ROTATE;
	  // Determine which direction we're rotating by taking dot
	  // products of the ray direction with the rotating planes'
	  // normals
	  float dotp0 =
	    fabs(hit.rayDirection.dot(faces[rotInfo.faceIdx0].normal));
	  float dotp1 =
	    fabs(hit.rayDirection.dot(faces[rotInfo.faceIdx1].normal));
	  int faceIdx;
	  if (dotp0 > dotp1)
	    faceIdx = rotInfo.faceIdx0;
	  else
	    faceIdx = rotInfo.faceIdx1;
	  FaceInfo &face = faces[faceIdx];
	  // Set up the rotation plane
	  rotatePlane.setOrigin(translation);
	  rotatePlane.setNormal(face.normal);
	  GleemV3f dummy;
	  GleemV2f startUV;
	  rotatePlane.projectPoint(hit.intPt, dummy, startUV);
	  startAngle = atan2f(startUV[1], startUV[0]);
	  startRot = rotation;
	  draggedGeometry = rotInfo.geometry;
	  draggedGeometry->highlight();
	  return;
	}
    }
  assert(0);
}

void
HandleBoxManip::drag(const GleemV3f &rayStart,
		     const GleemV3f &rayDirection)
{
  if (dragState == TRANSLATE)
    {
      // Algorithm: Find intersection of ray with dragPlane. Add
      // dragOffset to this point to get new translation.
      GleemV3f intPt;
      float t;
      if (dragPlane.intersectRay(rayStart,
				 rayDirection,
				 intPt,
				 t) == false)
	// Ray is parallel to plane. Punt.
	return;
      translation = intPt + dragOffset;
      recalc();
    }
  else if (dragState == ROTATE)
    {
      GleemV3f intPt;
      float t;
      GleemV2f uvCoords;
      if (rotatePlane.intersectRay(rayStart,
				   rayDirection,
				   intPt,
				   t,
				   uvCoords) == false)
	// Ray is parallel to plane. Punt.
	return;
      // Compute offset rotation angle
      GleemRot offsetRot(rotatePlane.getNormal(),
			atan2f(uvCoords[1], uvCoords[0]) - startAngle);
      GleemRot::mult(offsetRot, startRot, rotation);
      recalc();
    }
  else
    {
      cerr << "HandleBoxManip::drag: ERROR: Unexpected drag state" << endl;
      return;
    }
  Manip::drag(rayStart, rayDirection);
}

void
HandleBoxManip::makeInactive()
{
  dragState = INACTIVE;
  draggedGeometry->clearHighlight();
  draggedGeometry = NULL;
}

void
HandleBoxManip::deleteGeometry()
{
  int i;
  for (i = 0; i < faces.size(); i++)
    {
      FaceInfo &foo = faces[i];
      delete foo.geometry;
    }
  faces.erase(faces.begin(), faces.end());
  
  for (i = 0; i < rotateHandles.size(); i++)
    {
      RotateHandleInfo &foo = rotateHandles[i];
      delete foo.geometry;
    }
  rotateHandles.erase(rotateHandles.begin(), rotateHandles.end());
}

void
HandleBoxManip::createGeometry()
{
  deleteGeometry();

  //
  // Faces
  //

  FaceInfo info;
  // Front face (index 0)
  info.origNormal.setValue(0, 0, 1);
  info.geometry =
    createFace(info.origNormal, info.origNormal, GleemV3f(0, 1, 0));
  faces.push_back(info);
  // Right face (index 1)
  info.origNormal.setValue(1, 0, 0);
  info.geometry =
    createFace(info.origNormal, info.origNormal, GleemV3f(0, 1, 0));
  faces.push_back(info);
  // Back face (index 2)
  info.origNormal.setValue(0, 0, -1);
  info.geometry =
    createFace(info.origNormal, info.origNormal, GleemV3f(0, 1, 0));
  faces.push_back(info);
  // Left face (index 3)
  info.origNormal.setValue(-1, 0, 0);
  info.geometry =
    createFace(info.origNormal, info.origNormal, GleemV3f(0, 1, 0));
  faces.push_back(info);
  // Top face (index 4)
  info.origNormal.setValue(0, 1, 0);
  info.geometry =
    createFace(info.origNormal, info.origNormal, GleemV3f(0, 0, -1));
  faces.push_back(info);
  // Bottom face (index 5)
  info.origNormal.setValue(0, -1, 0);
  info.geometry =
    createFace(info.origNormal, info.origNormal, GleemV3f(0, 0, 1));
  faces.push_back(info);

  //
  // Rotation handles
  //

  RotateHandleInfo rotInfo;
  // Front handle. Rotates about top/bottom and left/right faces.
  // Maintain references to top and right faces.
  rotInfo.faceIdx0 = 4;
  rotInfo.faceIdx1 = 1;
  rotInfo.geometry = createRotateHandle(GleemV3f(0, 0, 1));
  rotateHandles.push_back(rotInfo);
  // Right handle. Rotates about top/bottom and front/back faces.
  // Maintain references to top and front faces.
  rotInfo.faceIdx0 = 4;
  rotInfo.faceIdx1 = 0;
  rotInfo.geometry = createRotateHandle(GleemV3f(1, 0, 0));
  rotateHandles.push_back(rotInfo);
  // Back handle. Rotates about top/bottom and left/right faces.
  // Maintain references to top and right faces.
  rotInfo.faceIdx0 = 4;
  rotInfo.faceIdx1 = 1;
  rotInfo.geometry = createRotateHandle(GleemV3f(0, 0, -1));
  rotateHandles.push_back(rotInfo);
  // Left handle. Rotates about top/bottom and front/back faces.
  // Maintain references to top and front faces.
  rotInfo.faceIdx0 = 4;
  rotInfo.faceIdx1 = 0;
  rotInfo.geometry = createRotateHandle(GleemV3f(-1, 0, 0));
  rotateHandles.push_back(rotInfo);
  // Top handle. Rotates about front/back and left/right faces.
  // Maintain references to front and right faces.
  rotInfo.faceIdx0 = 0;
  rotInfo.faceIdx1 = 1;
  rotInfo.geometry = createRotateHandle(GleemV3f(0, 1, 0));
  rotateHandles.push_back(rotInfo);
  // Bottom handle. Rotates about front/back and left/right faces.
  // Maintain references to front and right faces.
  rotInfo.faceIdx0 = 0;
  rotInfo.faceIdx1 = 1;
  rotInfo.geometry = createRotateHandle(GleemV3f(0, -1, 0));
  rotateHandles.push_back(rotInfo);
}

void
HandleBoxManip::recalc()
{
  // Construct local to world transform for geometry.
  // Scale, Rotation, Translation. Since we're right multiplying
  // column vectors, the actual matrix composed is TRS.
  GleemMat4f scaleMat;
  GleemMat4f rotMat;
  GleemMat4f xlateMat;
  GleemMat4f tmpMat;
  scaleMat.makeIdent();
  scaleMat[0][0] = scale[0] * geometryScale[0];
  scaleMat[1][1] = scale[1] * geometryScale[1];
  scaleMat[2][2] = scale[2] * geometryScale[2];
  rotMat.makeIdent();
  rotMat.setRotation(rotation);
  xlateMat.makeIdent();
  xlateMat[0][3] = translation[0];
  xlateMat[1][3] = translation[1];
  xlateMat[2][3] = translation[2];
  GleemMat4f::mult(xlateMat, rotMat, tmpMat);
  GleemMat4f::mult(tmpMat, scaleMat, xform);
  // FIXME: Add scale handles
  for (int i = 0; i < faces.size(); i++)
    {
      FaceInfo &face = faces[i];
      face.geometry->setTransform(xform);
      xform.xformDir(face.origNormal, face.normal);
      face.normal.normalize();
      
      RotateHandleInfo &rotInfo = rotateHandles[i];
      rotInfo.geometry->setTransform(xform);
    }
}

ManipPart *
HandleBoxManip::createFace(const GleemV3f &translation,
			   const GleemV3f &normal,
			   const GleemV3f &up)
{
  ManipPartTransform *xform = new ManipPartTransform(this);
  ManipPartHollowCubeFace *face = new ManipPartHollowCubeFace(this);
  ManipPartSquare *square = new ManipPartSquare(this);
  square->setVisible(false);
  xform->addPart(face);
  xform->addPart(square);
  GleemMat4f offset;
  offset.makeIdent();
  GleemV3f right;
  GleemV3f::cross(up, normal, right);
  offset[0][0] = right[0];
  offset[1][0] = right[1];
  offset[2][0] = right[2];
  offset[0][1] = up[0];
  offset[1][1] = up[1];
  offset[2][1] = up[2];
  offset[0][2] = normal[0];
  offset[1][2] = normal[1];
  offset[2][2] = normal[2];
  offset[0][3] = translation[0];
  offset[1][3] = translation[1];
  offset[2][3] = translation[2];
  xform->setOffsetTransform(offset);
  return xform;
}

ManipPart *
HandleBoxManip::createRotateHandle(const GleemV3f &direction)
{
  ManipPartCube *handle = new ManipPartCube(this);
  GleemMat4f offset;
  offset.makeIdent();
  offset[0][0] = offset[1][1] = offset[2][2] = 0.1f;
  offset.setTranslation(2.0f * direction);
  ManipPartTransform *xform = new ManipPartTransform(this);
  xform->addPart(handle);
  xform->setOffsetTransform(offset);
  return xform;
}
