/*
 * SieveBenchmark.cs - Implementation of the "SieveBenchmark" class.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace PNetMark
{

using System;

/// <summary>
/// <para>The <see cref="T:PNetMark.SieveBenchmark"/> class implements
/// the <see cref="T:PNetMark.IBenchmark"/> interface, and runs a
/// prime number sieve.</para>
/// </summary>
public class SieveBenchmark : IBenchmark
{
	// Internal state.
	public int maxPrime;
	public int[] primes;

	/// <summary>
	/// <para>Construct a new <see cref="T:PNetMark.SieveBenchmark"/>
	/// instance.</para>
	/// </summary>
	public SieveBenchmark()
	{
		maxPrime = 512;
	}

	/// <summary>
	/// <para>Initialize the benchmark to run on this system.</para>
	/// </summary>
	///
	/// <param name="size">
	/// <para>A value that indicates the size of the benchmark to run.
	/// This should be zero to use the default value.</para>
	/// </param>
	///
	/// <returns>
	/// <para>Returns <see langword="true"/> if the benchmark could be
	/// initialized, or <see langword="false"/> otherwise.</para>
	/// </returns>
	public bool Initialize(int size)
	{
		if(size != 0)
		{
			maxPrime = size;
		}
		primes = new int [maxPrime];
		return true;
	}

	/// <summary>
	/// <para>Get the name of the benchmark.</para>
	/// </summary>
	///
	/// <value>
	/// <para>A <see cref="T:System.String"/> that contains the name
	/// of the benchmark.</para>
	/// </value>
	public String Name
	{
		get
		{
			return "Sieve ";
		}
	}

	/// <summary>
	/// <para>Get the magnification factor for test scoring.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The magnification factor.</para>
	/// </value>
	public int Magnification
	{
		get
		{
			return 2771;
		}
	}

	/// <summary>
	/// <para>Run the benchmark.</para>
	/// </summary>
	public void Run()
	{
		int temp1;
		int temp3;
		int temp4;
		bool isPrime;
		primes[0] = 1;
		primes[1] = 2;
		temp3 = 2;
		for(temp1 = 3; temp1 < maxPrime; ++temp1)
		{
			isPrime = true;
			for(temp4 = 1; temp4 < temp3 && isPrime; ++temp4)
			{
				if(primes[temp4] > 0 && primes[temp4] <= (temp1 / 2))
				{
					if((temp1 % primes[temp4]) == 0)
					{
						isPrime = false;
					}
				}
			}
			if(isPrime)
			{
				++temp3;
				primes[temp3 - 1] = temp1;
			}
		}
	}

	/// <summary>
	/// <para>Clean up after the benchmark has been run.</para>
	/// </summary>
	public void CleanUp()
	{
		// Nothing needs to be done here.
	}

} // class SieveBenchmark

} // namespace PNetMark
