/*
 * DrawingHatchBrush.cs - Implementation of hatch brushes for System.Drawing.
 *
 * Copyright (C) 2003  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace System.Drawing.Toolkit
{

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Toolkit;
using Xsharp;

internal sealed class DrawingHatchBrush : IToolkitBrush
{
	// Internal state.
	private HatchStyle style;
	private System.Drawing.Color foreColor;
	private System.Drawing.Color backColor;
	private static Xsharp.Bitmap[] hatchBitmaps;

	// Constructor.
	public DrawingHatchBrush(HatchStyle style,
							 System.Drawing.Color foreColor,
							 System.Drawing.Color backColor)
			{
				this.style = style;
				this.foreColor = foreColor;
				this.backColor = backColor;
			}

	// Select this brush into a graphics object.
	public void Select(IToolkitGraphics _graphics)
			{
				DrawingGraphics graphics = (_graphics as DrawingGraphics);
				if(graphics != null)
				{
					Xsharp.Graphics g = graphics.graphics;
					Xsharp.Bitmap bitmap;
					lock(typeof(DrawingHatchBrush))
					{
						bitmap = GetBitmap(style);
					}
					if(bitmap != null)
					{
						// Use an opaque stipple to fill the region.
						g.Foreground =
							DrawingToolkit.DrawingToXColor(foreColor);
						g.Background =
							DrawingToolkit.DrawingToXColor(backColor);
						g.SetFillOpaqueStippled(bitmap, 0, 0);
					}
					else
					{
						// We don't recognize this hatch style, so use a
						// solid brush with the foreground color.
						g.Foreground =
							DrawingToolkit.DrawingToXColor(foreColor);
						g.SetFillSolid();
					}
				}
			}

	// Dispose of this brush.
	public void Dispose()
			{
				// Nothing to do here in this implementation.
			}

	// Get the bitmap corresponding to a particular hatch style.
	private static Xsharp.Bitmap GetBitmap(HatchStyle style)
			{
				Xsharp.Bitmap bitmap;

				// See if we have a cached bitmap for this style.
				if(((int)style) >= 0 && ((int)style) <= 52)
				{
					if(hatchBitmaps == null)
					{
						hatchBitmaps = new Xsharp.Bitmap [53];
					}
					bitmap = hatchBitmaps[(int)style];
					if(bitmap != null)
					{
						return bitmap;
					}
				}
				else
				{
					return null;
				}

				// Get the raw bits for the hatch bitmap.
				byte[] bits;
				switch(style)
				{
					case HatchStyle.BackwardDiagonal:
						bits = BackwardDiagonal_bits; break;
					case HatchStyle.DarkDownwardDiagonal:
						bits = DarkDownwardDiagonal_bits; break;
					case HatchStyle.DarkHorizontal:
						bits = DarkHorizontal_bits; break;
					case HatchStyle.DarkUpwardDiagonal:
						bits = DarkUpwardDiagonal_bits; break;
					case HatchStyle.DarkVertical:
						bits = DarkVertical_bits; break;
					case HatchStyle.DashedDownwardDiagonal:
						bits = DashedDownwardDiagonal_bits; break;
					case HatchStyle.DashedHorizontal:
						bits = DashedHorizontal_bits; break;
					case HatchStyle.DashedUpwardDiagonal:
						bits = DashedUpwardDiagonal_bits; break;
					case HatchStyle.DashedVertical:
						bits = DashedVertical_bits; break;
					case HatchStyle.DiagonalBrick:
						bits = DiagonalBrick_bits; break;
					case HatchStyle.DiagonalCross:
						bits = DiagonalCross_bits; break;
					case HatchStyle.Divot:
						bits = Divot_bits; break;
					case HatchStyle.DottedDiamond:
						bits = DottedDiamond_bits; break;
					case HatchStyle.DottedGrid:
						bits = DottedGrid_bits; break;
					case HatchStyle.ForwardDiagonal:
						bits = ForwardDiagonal_bits; break;
					case HatchStyle.Horizontal:
						bits = Horizontal_bits; break;
					case HatchStyle.HorizontalBrick:
						bits = HorizontalBrick_bits; break;
					case HatchStyle.LargeCheckerBoard:
						bits = LargeCheckerBoard_bits; break;
					case HatchStyle.LargeConfetti:
						bits = LargeConfetti_bits; break;
					case HatchStyle.LargeGrid:
						bits = LargeGrid_bits; break;
					case HatchStyle.LightDownwardDiagonal:
						bits = LightDownwardDiagonal_bits; break;
					case HatchStyle.LightHorizontal:
						bits = LightHorizontal_bits; break;
					case HatchStyle.LightUpwardDiagonal:
						bits = LightUpwardDiagonal_bits; break;
					case HatchStyle.LightVertical:
						bits = LightVertical_bits; break;
					case HatchStyle.NarrowHorizontal:
						bits = NarrowHorizontal_bits; break;
					case HatchStyle.NarrowVertical:
						bits = NarrowVertical_bits; break;
					case HatchStyle.OutlinedDiamond:
						bits = OutlinedDiamond_bits; break;
					case HatchStyle.Percent05:
						bits = Percent05_bits; break;
					case HatchStyle.Percent10:
						bits = Percent10_bits; break;
					case HatchStyle.Percent20:
						bits = Percent20_bits; break;
					case HatchStyle.Percent25:
						bits = Percent25_bits; break;
					case HatchStyle.Percent30:
						bits = Percent30_bits; break;
					case HatchStyle.Percent40:
						bits = Percent40_bits; break;
					case HatchStyle.Percent50:
						bits = Percent50_bits; break;
					case HatchStyle.Percent60:
						bits = Percent60_bits; break;
					case HatchStyle.Percent70:
						bits = Percent70_bits; break;
					case HatchStyle.Percent75:
						bits = Percent75_bits; break;
					case HatchStyle.Percent80:
						bits = Percent80_bits; break;
					case HatchStyle.Percent90:
						bits = Percent90_bits; break;
					case HatchStyle.Plaid:
						bits = Plaid_bits; break;
					case HatchStyle.Shingle:
						bits = Shingle_bits; break;
					case HatchStyle.SmallCheckerBoard:
						bits = SmallCheckerBoard_bits; break;
					case HatchStyle.SmallConfetti:
						bits = SmallConfetti_bits; break;
					case HatchStyle.SmallGrid:
						bits = SmallGrid_bits; break;
					case HatchStyle.SolidDiamond:
						bits = SolidDiamond_bits; break;
					case HatchStyle.Sphere:
						bits = Sphere_bits; break;
					case HatchStyle.Trellis:
						bits = Trellis_bits; break;
					case HatchStyle.Vertical:
						bits = Vertical_bits; break;
					case HatchStyle.Wave:
						bits = Wave_bits; break;
					case HatchStyle.Weave:
						bits = Weave_bits; break;
					case HatchStyle.WideDownwardDiagonal:
						bits = WideDownwardDiagonal_bits; break;
					case HatchStyle.WideUpwardDiagonal:
						bits = WideUpwardDiagonal_bits; break;
					case HatchStyle.ZigZag:
						bits = ZigZag_bits; break;
					default: return null;
				}

				// Create the bitmap, cache it for later, and then return it.
				bitmap = new Xsharp.Bitmap(16, 16, bits);
				hatchBitmaps[(int)style] = bitmap;
				return bitmap;
			}

	// Bitmap data for all of the hatch styles.
	private static readonly byte[] BackwardDiagonal_bits = {
		0x80,0x80,0x40,0x40,0x20,0x20,0x10,0x10,0x08,0x08,0x04,0x04,
		0x02,0x02,0x01,0x01,0x80,0x80,0x40,0x40,0x20,0x20,0x10,0x10,
		0x08,0x08,0x04,0x04,0x02,0x02,0x01,0x01};
	private static readonly byte[] DarkDownwardDiagonal_bits = {
		0x33,0x33,0x66,0x66,0xcc,0xcc,0x99,0x99,0x33,0x33,0x66,0x66,
		0xcc,0xcc,0x99,0x99,0x33,0x33,0x66,0x66,0xcc,0xcc,0x99,0x99,
		0x33,0x33,0x66,0x66,0xcc,0xcc,0x99,0x99};
	private static readonly byte[] DarkHorizontal_bits = {
		0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,
		0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,
		0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00};
	private static readonly byte[] DarkUpwardDiagonal_bits = {
		0xcc,0xcc,0x66,0x66,0x33,0x33,0x99,0x99,0xcc,0xcc,0x66,0x66,
		0x33,0x33,0x99,0x99,0xcc,0xcc,0x66,0x66,0x33,0x33,0x99,0x99,
		0xcc,0xcc,0x66,0x66,0x33,0x33,0x99,0x99};
	private static readonly byte[] DarkVertical_bits = {
		0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,
		0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,
		0x33,0x33,0x33,0x33,0x33,0x33, 0x33,0x33};
	private static readonly byte[] DashedDownwardDiagonal_bits = {
		0x00,0x00,0x00,0x00,0x11,0x11,0x22,0x22,0x44,0x44,0x88,0x88,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x11,0x11,0x22,0x22,
		0x44,0x44,0x88,0x88,0x00,0x00,0x00,0x00};
	private static readonly byte[] DashedHorizontal_bits = {
		0x0f,0x0f,0x00,0x00,0x00,0x00,0x00,0x00,0xf0,0xf0,0x00,0x00,
		0x00,0x00,0x00,0x00,0x0f,0x0f,0x00,0x00,0x00,0x00,0x00,0x00,
		0xf0,0xf0,0x00,0x00,0x00,0x00,0x00,0x00};
	private static readonly byte[] DashedUpwardDiagonal_bits = {
		0x00,0x00,0x00,0x00,0x88,0x88,0x44,0x44,0x22,0x22,0x11,0x11,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x88,0x88,0x44,0x44,
		0x22,0x22,0x11,0x11,0x00,0x00,0x00,0x00};
	private static readonly byte[] DashedVertical_bits = {
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x10,0x10,0x10,0x10,
		0x10,0x10,0x10,0x10,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
		0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10};
	private static readonly byte[] DiagonalBrick_bits = {
		0x80,0x80,0x40,0x40,0x20,0x20,0x10,0x10,0x18,0x18,0x24,0x24,
		0x42,0x42,0x81,0x81,0x80,0x80,0x40,0x40,0x20,0x20,0x10,0x10,
		0x18,0x18,0x24,0x24,0x42,0x42,0x81,0x81};
	private static readonly byte[] DiagonalCross_bits = {
		0x81,0x81,0x42,0x42,0x24,0x24,0x18,0x18,0x18,0x18,0x24,0x24,
		0x42,0x42,0x81,0x81,0x81,0x81,0x42,0x42,0x24,0x24,0x18,0x18,
		0x18,0x18,0x24,0x24,0x42,0x42,0x81,0x81};
	private static readonly byte[] Divot_bits = {
		0x00,0x00,0x08,0x08,0x10,0x10,0x08,0x08,0x00,0x00,0x01,0x01,
		0x80,0x80,0x01,0x01,0x00,0x00,0x08,0x08,0x10,0x10,0x08,0x08,
		0x00,0x00,0x01,0x01,0x80,0x80,0x01,0x01};
	private static readonly byte[] DottedDiamond_bits = {
		0x01,0x01,0x00,0x00,0x44,0x44,0x00,0x00,0x10,0x10,0x00,0x00,
		0x44,0x44,0x00,0x00,0x01,0x01,0x00,0x00,0x44,0x44,0x00,0x00,
		0x10,0x10,0x00,0x00,0x44,0x44,0x00,0x00};
	private static readonly byte[] DottedGrid_bits = {
		0x55,0x55,0x00,0x00,0x01,0x01,0x00,0x00,0x01,0x01,0x00,0x00,
		0x01,0x01,0x00,0x00,0x55,0x55,0x00,0x00,0x01,0x01,0x00,0x00,
		0x01,0x01,0x00,0x00,0x01,0x01,0x00,0x00};
	private static readonly byte[] ForwardDiagonal_bits = {
		0x01,0x01,0x02,0x02,0x04,0x04,0x08,0x08,0x10,0x10,0x20,0x20,
		0x40,0x40,0x80,0x80,0x01,0x01,0x02,0x02,0x04,0x04,0x08,0x08,
		0x10,0x10,0x20,0x20,0x40,0x40,0x80,0x80};
	private static readonly byte[] Horizontal_bits = {
		0xff,0xff,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0xff,0xff,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
	private static readonly byte[] HorizontalBrick_bits = {
		0xff,0xff,0x01,0x01,0x01,0x01,0x01,0x01,0xff,0xff,0x10,0x10,
		0x10,0x10,0x10,0x10,0xff,0xff,0x01,0x01,0x01,0x01,0x01,0x01,
		0xff,0xff,0x10,0x10,0x10,0x10,0x10,0x10};
	private static readonly byte[] LargeCheckerBoard_bits = {
		0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0xf0,0xf0,0xf0,0xf0,
		0xf0,0xf0,0xf0,0xf0,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
		0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0};
	private static readonly byte[] LargeConfetti_bits = {
		0x8d,0x8d,0x0c,0x0c,0xc0,0xc0,0xd8,0xd8,0x1b,0x1b,0x03,0x03,
		0x30,0x30,0xb1,0xb1,0x8d,0x8d,0x0c,0x0c,0xc0,0xc0,0xd8,0xd8,
		0x1b,0x1b,0x03,0x03,0x30,0x30,0xb1,0xb1};
	private static readonly byte[] LargeGrid_bits = {
		0xff,0xff,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0x01,0xff,0xff,0x01,0x01,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01};
	private static readonly byte[] LightDownwardDiagonal_bits = {
		0x11,0x11,0x22,0x22,0x44,0x44,0x88,0x88,0x11,0x11,0x22,0x22,
		0x44,0x44,0x88,0x88,0x11,0x11,0x22,0x22,0x44,0x44,0x88,0x88,
		0x11,0x11,0x22,0x22,0x44,0x44,0x88,0x88};
	private static readonly byte[] LightHorizontal_bits = {
		0xff,0xff,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0xff,0x00,0x00,
		0x00,0x00,0x00,0x00,0xff,0xff,0x00,0x00,0x00,0x00,0x00,0x00,
		0xff,0xff,0x00,0x00,0x00,0x00,0x00,0x00};
	private static readonly byte[] LightUpwardDiagonal_bits = {
		0x88,0x88,0x44,0x44,0x22,0x22,0x11,0x11,0x88,0x88,0x44,0x44,
		0x22,0x22,0x11,0x11,0x88,0x88,0x44,0x44,0x22,0x22,0x11,0x11,
		0x88,0x88,0x44,0x44,0x22,0x22,0x11,0x11};
	private static readonly byte[] LightVertical_bits = {
		0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
		0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11,
		0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11};
	private static readonly byte[] NarrowHorizontal_bits = {
		0xff,0xff,0x00,0x00,0xff,0xff,0x00,0x00,0xff,0xff,0x00,0x00,
		0xff,0xff,0x00,0x00,0xff,0xff,0x00,0x00,0xff,0xff,0x00,0x00,
		0xff,0xff,0x00,0x00,0xff,0xff,0x00,0x00};
	private static readonly byte[] NarrowVertical_bits = {
		0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
		0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
		0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa};
	private static readonly byte[] OutlinedDiamond_bits = {
		0x41,0x41,0x22,0x22,0x14,0x14,0x08,0x08,0x14,0x14,0x22,0x22,
		0x41,0x41,0x80,0x80,0x41,0x41,0x22,0x22,0x14,0x14,0x08,0x08,
		0x14,0x14,0x22,0x22,0x41,0x41,0x80,0x80};
	private static readonly byte[] Percent05_bits = {
		0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x10,0x00,0x00,
		0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
		0x10,0x10,0x00,0x00,0x00,0x00,0x00,0x00};
	private static readonly byte[] Percent10_bits = {
		0x01,0x01,0x00,0x00,0x10,0x10,0x00,0x00,0x01,0x01,0x00,0x00,
		0x10,0x10,0x00,0x00,0x01,0x01,0x00,0x00,0x10,0x10,0x00,0x00,
		0x01,0x01,0x00,0x00,0x10,0x10,0x00,0x00};
	private static readonly byte[] Percent20_bits = {
		0x11,0x11,0x00,0x00,0x44,0x44,0x00,0x00,0x11,0x11,0x00,0x00,
		0x44,0x44,0x00,0x00,0x11,0x11,0x00,0x00,0x44,0x44,0x00,0x00,
		0x11,0x11,0x00,0x00,0x44,0x44,0x00,0x00};
	private static readonly byte[] Percent25_bits = {
		0x11,0x11,0x44,0x44,0x11,0x11,0x44,0x44,0x11,0x11,0x44,0x44,
		0x11,0x11,0x44,0x44,0x11,0x11,0x44,0x44,0x11,0x11,0x44,0x44,
		0x11,0x11,0x44,0x44,0x11,0x11,0x44,0x44};
	private static readonly byte[] Percent30_bits = {
		0x55,0x55,0x22,0x22,0x55,0x55,0x88,0x88,0x55,0x55,0x22,0x22,
		0x55,0x55,0x88,0x88,0x55,0x55,0x22,0x22,0x55,0x55,0x88,0x88,
		0x55,0x55,0x22,0x22,0x55,0x55,0x88,0x88};
	private static readonly byte[] Percent40_bits = {
		0x55,0x55,0xaa,0xaa,0x55,0x55,0x8a,0x8a,0x55,0x55,0xaa,0xaa,
		0x55,0x55,0xa8,0xa8,0x55,0x55,0xaa,0xaa,0x55,0x55,0x8a,0x8a,
		0x55,0x55,0xaa,0xaa,0x55,0x55,0xa8,0xa8};
	private static readonly byte[] Percent50_bits = {
		0x55,0x55,0xaa,0xaa,0x55,0x55,0xaa,0xaa,0x55,0x55,0xaa,0xaa,
		0x55,0x55,0xaa,0xaa,0x55,0x55,0xaa,0xaa,0x55,0x55,0xaa,0xaa,
		0x55,0x55,0xaa,0xaa,0x55,0x55,0xaa,0xaa};
	private static readonly byte[] Percent60_bits = {
		0x77,0x77,0xaa,0xaa,0xdd,0xdd,0xaa,0xaa,0x77,0x77,0xaa,0xaa,
		0xdd,0xdd,0xaa,0xaa,0x77,0x77,0xaa,0xaa,0xdd,0xdd,0xaa,0xaa,
		0x77,0x77,0xaa,0xaa,0xdd,0xdd,0xaa,0xaa};
	private static readonly byte[] Percent70_bits = {
		0xee,0xee,0xbb,0xbb,0xee,0xee,0xbb,0xbb,0xee,0xee,0xbb,0xbb,
		0xee,0xee,0xbb,0xbb,0xee,0xee,0xbb,0xbb,0xee,0xee,0xbb,0xbb,
		0xee,0xee,0xbb,0xbb,0xee,0xee,0xbb,0xbb};
	private static readonly byte[] Percent75_bits = {
		0xee,0xee,0xff,0xff,0xbb,0xbb,0xff,0xff,0xee,0xee,0xff,0xff,
		0xbb,0xbb,0xff,0xff,0xee,0xee,0xff,0xff,0xbb,0xbb,0xff,0xff,
		0xee,0xee,0xff,0xff,0xbb,0xbb,0xff,0xff};
	private static readonly byte[] Percent80_bits = {
		0xf7,0xf7,0xff,0xff,0x7f,0x7f,0xff,0xff,0xf7,0xf7,0xff,0xff,
		0x7f,0x7f,0xff,0xff,0xf7,0xf7,0xff,0xff,0x7f,0x7f,0xff,0xff,
		0xf7,0xf7,0xff,0xff,0x7f,0x7f,0xff,0xff};
	private static readonly byte[] Percent90_bits = {
		0xff,0xff,0xff,0xff,0xff,0xff,0xef,0xef,0xff,0xff,0xff,0xff,
		0xff,0xff,0xfe,0xfe,0xff,0xff,0xff,0xff,0xff,0xff,0xef,0xef,
		0xff,0xff,0xff,0xff,0xff,0xff,0xfe,0xfe};
	private static readonly byte[] Plaid_bits = {
		0x55,0x55,0xaa,0xaa,0x55,0x55,0xaa,0xaa,0x0f,0x0f,0x0f,0x0f,
		0x0f,0x0f,0x0f,0x0f,0x55,0x55,0xaa,0xaa,0x55,0x55,0xaa,0xaa,
		0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f};
	private static readonly byte[] Shingle_bits = {
		0xc0,0xc0,0x21,0x21,0x12,0x12,0x0c,0x0c,0x30,0x30,0x40,0x40,
		0x80,0x80,0x80,0x80,0xc0,0xc0,0x21,0x21,0x12,0x12,0x0c,0x0c,
		0x30,0x30,0x40,0x40,0x80,0x80,0x80,0x80};
	private static readonly byte[] SmallCheckerBoard_bits = {
		0x99,0x99,0x66,0x66,0x66,0x66,0x99,0x99,0x99,0x99,0x66,0x66,
		0x66,0x66,0x99,0x99,0x99,0x99,0x66,0x66,0x66,0x66,0x99,0x99,
		0x99,0x99,0x66,0x66,0x66,0x66,0x99,0x99};
	private static readonly byte[] SmallConfetti_bits = {
		0x01,0x01,0x10,0x10,0x02,0x02,0x40,0x40,0x08,0x08,0x80,0x80,
		0x04,0x04,0x20,0x20,0x01,0x01,0x10,0x10,0x02,0x02,0x40,0x40,
		0x08,0x08,0x80,0x80,0x04,0x04,0x20,0x20};
	private static readonly byte[] SmallGrid_bits = {
		0xff,0xff,0x11,0x11,0x11,0x11,0x11,0x11,0xff,0xff,0x11,0x11,
		0x11,0x11,0x11,0x11,0xff,0xff,0x11,0x11,0x11,0x11,0x11,0x11,
		0xff,0xff,0x11,0x11,0x11,0x11,0x11,0x11};
	private static readonly byte[] SolidDiamond_bits = {
		0x08,0x08,0x1c,0x1c,0x3e,0x3e,0x7f,0x7f,0x3e,0x3e,0x1c,0x1c,
		0x08,0x08,0x00,0x00,0x08,0x08,0x1c,0x1c,0x3e,0x3e,0x7f,0x7f,
		0x3e,0x3e,0x1c,0x1c,0x08,0x08,0x00,0x00};
	private static readonly byte[] Sphere_bits = {
		0xee,0xee,0x91,0x91,0xf1,0xf1,0xf1,0xf1,0xee,0xee,0x19,0x19,
		0x1f,0x1f,0x1f,0x1f,0xee,0xee,0x91,0x91,0xf1,0xf1,0xf1,0xf1,
		0xee,0xee,0x19,0x19,0x1f,0x1f,0x1f,0x1f};
	private static readonly byte[] Trellis_bits = {
		0xff,0xff,0x66,0x66,0xff,0xff,0x99,0x99,0xff,0xff,0x66,0x66,
		0xff,0xff,0x99,0x99,0xff,0xff,0x66,0x66,0xff,0xff,0x99,0x99,
		0xff,0xff,0x66,0x66,0xff,0xff,0x99,0x99};
	private static readonly byte[] Vertical_bits = {
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01};
	private static readonly byte[] Wave_bits = {
		0x00,0x00,0x18,0x18,0xa4,0xa4,0x03,0x03,0x00,0x00,0x18,0x18,
		0xa4,0xa4,0x03,0x03,0x00,0x00,0x18,0x18,0xa4,0xa4,0x03,0x03,
		0x00,0x00,0x18,0x18,0xa4,0xa4,0x03,0x03};
	private static readonly byte[] Weave_bits = {
		0x11,0x11,0x2a,0x2a,0x44,0x44,0xa2,0xa2,0x11,0x11,0x28,0x28,
		0x44,0x44,0x8a,0x8a,0x11,0x11,0x2a,0x2a,0x44,0x44,0xa2,0xa2,
		0x11,0x11,0x28,0x28,0x44,0x44,0x8a,0x8a};
	private static readonly byte[] WideDownwardDiagonal_bits = {
		0x83,0x83,0x07,0x07,0x0e,0x0e,0x1c,0x1c,0x38,0x38,0x70,0x70,
		0xe0,0xe0,0xc1,0xc1,0x83,0x83,0x07,0x07,0x0e,0x0e,0x1c,0x1c,
		0x38,0x38,0x70,0x70,0xe0,0xe0,0xc1,0xc1};
	private static readonly byte[] WideUpwardDiagonal_bits = {
		0xc1,0xc1,0xe0,0xe0,0x70,0x70,0x38,0x38,0x1c,0x1c,0x0e,0x0e,
		0x07,0x07,0x83,0x83,0xc1,0xc1,0xe0,0xe0,0x70,0x70,0x38,0x38,
		0x1c,0x1c,0x0e,0x0e,0x07,0x07,0x83,0x83};
	private static readonly byte[] ZigZag_bits = {
		0x81,0x81,0x42,0x42,0x24,0x24,0x18,0x18,0x81,0x81,0x42,0x42,
		0x24,0x24,0x18,0x18,0x81,0x81,0x42,0x42,0x24,0x24,0x18,0x18,
		0x81,0x81,0x42,0x42,0x24,0x24,0x18,0x18};

}; // class DrawingHatchBrush

}; // namespace System.Drawing.Toolkit
