/*
 * Window.cs - Curses window abstraction.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Curses
{

using System;
using System.Collections;

public class Window : IDisposable, IEnumerable
{
	// Cached copy of the screen window.
	private static Window screen;

	// Internal state.
	private IntPtr   handle;
	private Window   parent;
	private int      x;
	private int      y;
	private int      width;
	private int      height;
	private Window[] children;
	private String	 name;

	/// <summary>
	/// <para>Constructs a new <see cref="T:Curses.Window"/> instance,
	/// as a sub-window of another window.</para>
	/// </summary>
	///
	/// <param name="parent">
	/// <para>The parent window to create the new window within.
	/// If this value is <see langword="null"/>, then the window
	/// will be created as a child of the screen.</para>
	/// </param>
	///
	/// <param name="x">
	/// <para>The X co-ordinate of the top-left corner of the window.</para>
	/// </param>
	///
	/// <param name="y">
	/// <para>The Y co-ordinate of the top-left corner of the window.</para>
	/// </param>
	///
	/// <param name="width">
	/// <para>The width of the new window.  If this value is zero or
	/// negative, then the window will stretch to the right-hand side
	/// of its parent.</para>
	/// </param>
	///
	/// <param name="height">
	/// <para>The height of the new window.  If this value is zero or
	/// negative, then the window will stretch to the bottom side
	/// of its parent.</para>
	/// </param>
	public Window(Window parent, int x, int y, int width, int height)
			{
				// Get the default parent, if necessary.
				if(parent == null)
				{
					parent = Screen;
				}

				// Adjust the width and height.
				if(width <= 0)
				{
					width = parent.Width - x;
					if(width <= 0)
					{
						width = 1;
					}
				}
				if(height <= 0)
				{
					height = parent.Height - y;
					if(height <= 0)
					{
						height = 1;
					}
				}

				// Create the underlying curses window.
				if(parent == Screen && parent.handle != (IntPtr)0)
				{
					// Create a new top-level window.
					handle = Curses.Native.newwin(height, width, y, x);
				}
				else if(parent.handle != (IntPtr)0)
				{
					// Create a sub-window of an existing window.
					handle = Curses.Native.derwin(parent.handle,
												  height, width, y, x);
				}
				else
				{
					// The parent is invalid, so this window is too.
					handle = (IntPtr)0;
				}

				// Initialize the window state.
				this.parent = parent;
				this.x = x;
				this.y = y;
				this.width = width;
				this.height = height;
				children = null;

				// Add ourselves to the parent's child list.
				if(parent.children == null)
				{
					parent.children = new Window [1];
					parent.children[0] = this;
				}
				else
				{
					int length = parent.children.Length;
					Window[] newList = new Window [length + 1];
					Array.Copy(parent.children, 0, newList, 0, length);
					newList[length] = this;
					parent.children = newList;
				}
			}

	/// <summary>
	/// <para>Constructs a new <see cref="T:Curses.Window"/> instance
	/// that corresponds to the screen.</para>
	/// </summary>
	///
	/// <param name="handle">
	/// <para>The handle for the curses screen window, or <c>(IntPtr)0</c>
	/// if curses could not be initialized for some reason.</para>
	/// </param>
	private Window(IntPtr handle)
			{
				this.handle = handle;
				parent = null;
				x = 0;
				y = 0;
				if(handle != (IntPtr)0)
				{
					width = Curses.Native.CursesHelpGetCols();
					height = Curses.Native.CursesHelpGetLines();
				}
				else
				{
					width = 80;
					height = 24;
				}
				children = null;
			}

	/// <summary>
	/// <para>Finalize this window.</para>
	/// </summary>
	~Window()
			{
				Close(false);
			}

	/// <summary>
	/// <para>Get the window that corresponds to the entire screen.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The window object corresponding to the screen.</para>
	/// </value>
	public static Window Screen
			{
				get
				{
					lock(typeof(Window))
					{
						if(screen != null)
						{
							return screen;
						}
						if(Setup.Initialize())
						{
							screen = new Window
								(Curses.Native.CursesHelpGetStdScr());
						}
						else
						{
							screen = new Window((IntPtr)0);
						}
						return screen;
					}
				}
			}

	/// <summary>
	/// <para>Get the parent of this window.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The parent window, or <see langword="null"/> if this
	/// window is the screen.</para>
	/// </value>
	public Window Parent
			{
				get
				{
					return parent;
				}
			}

	/// <summary>
	/// <para>Get the X co-ordinate of the top-level corner of
	/// this window.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The X co-ordinate.</para>
	/// </value>
	public int X
			{
				get
				{
					return x;
				}
			}

	/// <summary>
	/// <para>Get the Y co-ordinate of the top-level corner of
	/// this window.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The Y co-ordinate.</para>
	/// </value>
	public int Y
			{
				get
				{
					return y;
				}
			}

	/// <summary>
	/// <para>Get the width of this window.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The width.</para>
	/// </value>
	public int Width
			{
				get
				{
					return width;
				}
			}

	/// <summary>
	/// <para>Get the height of this window.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The height.</para>
	/// </value>
	public int Height
			{
				get
				{
					return height;
				}
			}

	/// <summary>
	/// <para>Get or set the name associated with this window.</para>
	/// <para>This can be used with the <c>GetChildByName</c> and
	/// <c>GetWindowByName</c> methods to a locate window by name.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The window's name.  Returns <see langword="null"/> if
	/// there is no name currently assigned.</para>
	/// </value>
	public String Name
			{
				get
				{
					return name;
				}
				set
				{
					name = value;
				}
			}

	/// <summary>
	/// <para>Get or set the foreground color of this window.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The foreground color.</para>
	/// </value>
	public Color Foreground
			{
				get
				{
					if(Curses.Native.has_colors() != 0 &&
					   handle != (IntPtr)0)
					{
						Attributes bg =
							(Attributes)(Curses.Native.getbkgd(handle));
						short pair = (short)(((int)(bg & ~Attributes.A_COLOR))
												>> 8);
						Color f, g;
						DrawingContext.PairContent(pair, out f, out g);
						return f;
					}
					else
					{
						return Color.Default;
					}
				}
				set
				{
					if(Curses.Native.has_colors() == 0 ||
					   handle == (IntPtr)0)
					{
						return;
					}
					Attributes bg = (Attributes)(Curses.Native.getbkgd(handle));
					short pair = (short)(((int)(bg & ~Attributes.A_COLOR))
												>> 8);
					Color f, g;
					DrawingContext.PairContent(pair, out f, out g);
					pair = DrawingContext.AllocColorPair(value, g);
					Attributes newbg = bg & ~(Attributes.A_COLOR);
					newbg |= (Attributes)(((int)pair) << 8);
					if(bg != newbg)
					{
						Curses.Native.wbkgd
							(handle, ((uint)newbg) | ((uint)0x20));
					}
				}
			}

	/// <summary>
	/// <para>Get or set the background color of this window.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The background color.</para>
	/// </value>
	public Color Background
			{
				get
				{
					if(Curses.Native.has_colors() != 0 &&
					   handle != (IntPtr)0)
					{
						Attributes bg =
							(Attributes)(Curses.Native.getbkgd(handle));
						short pair = (short)(((int)(bg & ~Attributes.A_COLOR))
												>> 8);
						Color f, g;
						Curses.DrawingContext.PairContent(pair, out f, out g);
						return g;
					}
					else
					{
						return Color.Default;
					}
				}
				set
				{
					if(Curses.Native.has_colors() == 0 ||
					   handle == (IntPtr)0)
					{
						return;
					}
					Attributes bg = (Attributes)(Curses.Native.getbkgd(handle));
					short pair = (short)(((int)(bg & ~Attributes.A_COLOR))
												>> 8);
					Color f, g;
					Curses.DrawingContext.PairContent(pair, out f, out g);
					pair = DrawingContext.AllocColorPair((Color)f, value);
					Attributes newbg = bg & ~(Attributes.A_COLOR);
					newbg |= (Attributes)(((int)pair) << 8);
					if(bg != newbg)
					{
						Curses.Native.wbkgd
							(handle, ((uint)newbg) | ((uint)0x20));
					}
				}
			}

	/// <summary>
	/// <para>Close this window and free the resources
	/// associated with it.</para>
	/// </summary>
	///
	/// <param name="refresh">
	/// <para>A flag that indicates if the parent window should be
	/// refreshed after closing the window.</para>
	/// </param>
	///
	/// <remarks>
	/// <para>This method will have no effect if it is invoked on the
	/// screen window, or on a window that has already been closed.</para>
	/// </remarks>
	public virtual void Close(bool refresh)
			{
				if(handle != (IntPtr)0 &&
				   handle != Curses.Native.CursesHelpGetStdScr())
				{
					// Close child windows.
					if(children != null)
					{
						foreach(Window child in children)
						{
							child.Close(false);
						}
					}

					// Delete the underlying window handle.
					Curses.Native.delwin(handle);
					handle = (IntPtr)0;

					// Refresh the parent window.
					if(refresh && parent != null && parent.handle != (IntPtr)0)
					{
						parent.Refresh();
					}
				}
			}

	/// <summary>
	/// <para>Dispose this window.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This method implements the <see cref="T:System.IDisposable"/>
	/// interface.</para>
	/// </remarks>
	public void Dispose()
			{
				Close(false);
			}

	/// <summary>
	/// <para>Enumerate the children of this window.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This method implements the <see cref="T:System.IEnumerable"/>
	/// interface.</para>
	/// </remarks>
	public IEnumerator GetEnumerator()
			{
				return children.GetEnumerator();
			}

	/// <summary>
	/// <para>Search for a child of this window by name.</para>
	/// </summary>
	///
	/// <param name="name">
	/// <para>The name of the window to look for.</para>
	/// </param>
	///
	/// <returns>
	/// <para>Returns the child called <paramref name="name"/>, or
	/// <see langword="null"/> if there is no such child.</para>
	/// </returns>
	public Window GetChildByName(String name)
			{
				if(children != null)
				{
					foreach(Window child in children)
					{
						if(child.Name == name)
						{
							return child;
						}
					}
				}
				return null;
			}

	/// <summary>
	/// <para>Search for a descendent of this window by name.</para>
	/// </summary>
	///
	/// <param name="name">
	/// <para>The name of the window to look for.</para>
	/// </param>
	///
	/// <returns>
	/// <para>Returns the descendent called <paramref name="name"/>, or
	/// <see langword="null"/> if there is no such descendent.</para>
	/// </returns>
	///
	/// <remarks>
	/// <para>This method will recursively search child windows
	/// for the name.</para>
	/// </remarks>
	public Window GetWindowByName(String name)
			{
				Window temp;
				if(children != null)
				{
					foreach(Window child in children)
					{
						if(child.Name == name)
						{
							return child;
						}
						temp = child.GetWindowByName(name);
						if(temp != null)
						{
							return temp;
						}
					}
				}
				return null;
			}

	/// <summary>
	/// <para>Refresh the contents of this window.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This method must be called to update the actual screen
	/// with any changes that have been made to the window's contents.</para>
	/// </remarks>
	public virtual void Refresh()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wrefresh(handle);
				}
			}

	/// <summary>
	/// <para>Mark a number of lines within this window as
	/// needing a full update at the next refresh.</para>
	/// </summary>
	///
	/// <param name="y1">
	/// <para>The first line to be touched.</para>
	/// </param>
	///
	/// <param name="y2">
	/// <para>The last line to be touched.</para>
	/// </param>
	public virtual void Touch(int y1, int y2)
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wtouchln(handle, y1, (y2 - y1 + 1), 1);
				}
			}

	/// <summary>
	/// <para>Mark this entire window as needing a full update
	/// at the next refresh.</para>
	/// </summary>
	public virtual void Touch()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wtouchln(handle, 0, height, 1);
				}
			}

	/// <summary>
	/// <para>Draw a border around the edges of this window.</para>
	/// </summary>
	public virtual void Border()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wborder(handle, 0, 0, 0, 0, 0, 0, 0, 0);
				}
			}

	/// <summary>
	/// <para>Clear the entire contents of this window.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This is normally followed by a call to <c>Refresh</c>.</para>
	/// </remarks>
	public virtual void Clear()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.wclear(handle);
				}
			}

	/// <summary>
	/// <para>Erase the entire contents of this window.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This does not mark the contents of the window for update,
	/// as <c>Clear</c> does.</para>
	/// </remarks>
	public virtual void Erase()
			{
				if(handle != (IntPtr)0)
				{
					Curses.Native.werase(handle);
				}
			}

	/// <summary>
	/// <para>Get a drawing context that can be used to draw into
	/// this window.</para>
	/// </summary>
	public Curses.DrawingContext DrawingContext
			{
				get
				{
					return new Curses.DrawingContext(this, handle);
				}
			}

	/// <summary>
	/// <para>Resize this window object and all of its children
	/// to match the underlying curses window structures.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This is typically used after a screen resize has
	/// been detected.</para>
	/// </remarks>
	internal void ResizeToMatch()
			{
				if(handle != (IntPtr)0)
				{
				   	if(handle == Curses.Native.CursesHelpGetStdScr())
					{
						width = Curses.Native.CursesHelpGetCols();
						height = Curses.Native.CursesHelpGetLines();
					}
					else
					{
						x = Curses.Native.CursesHelpGetX(handle);
						y = Curses.Native.CursesHelpGetY(handle);
						width = Curses.Native.CursesHelpGetWidth(handle);
						height = Curses.Native.CursesHelpGetHeight(handle);
					}
				}
				if(children != null)
				{
					foreach(Window w in children)
					{
						w.ResizeToMatch();
					}
				}
			}

} // class Window

} // namespace Curses
