/*
 * Misc.cs - Miscellaneous curses routines.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Curses
{

using System;

public sealed class Misc
{
	// Internal state.
	internal static bool dockedCursor = false;
	internal static bool soundEnabled = true;
	internal static bool reportTimeouts = false;

	/// <summary>
	/// <para>Ring the terminal's audible alarm bell.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>If the terminal does not have an audible alarm, this
	/// may flash the screen instead.  If screen flashing is also
	/// not present, then this method will do nothing.</para>
	/// </remarks>
	public static void Beep()
			{
				if(soundEnabled && Setup.Initialize())
				{
					Curses.Native.beep();
				}
			}

	/// <summary>
	/// <para>Flash the screen.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>If the terminal does not support screen flashing, this
	/// may ring the audible alarm beel instead.  If the audible alarm is
	/// not present, then this method will do nothing.</para>
	/// </remarks>
	public static void Flash()
			{
				if(soundEnabled && Setup.Initialize())
				{
					Curses.Native.flash();
				}
			}

	/// <summary>
	/// <para>Dock the cursor somewhere out of the way where it cannot
	/// be seen, or is unobtrusive.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>It is very difficult to make the cursor disappear curses in
	/// applications that don't require it.  e.g., games.  This
	/// method can be used to either make the cursor invisible, or
	/// to move it where it is not interferring with the main display.</para>
	/// </remarks>
	public static void DockCursor()
			{
				if(Setup.Initialize())
				{
					IntPtr stdscr = Curses.Native.CursesHelpGetStdScr();
					Curses.Native.wmove(stdscr,
										Curses.Native.CursesHelpGetLines() - 1,
										Curses.Native.CursesHelpGetCols() - 1);
				}
			}

	/// <summary>
	/// <para>Get or set the docking state for the cursor.</para>
	/// </summary>
	///
	/// <value>
	/// <para><see langword="true"/> if the cursor will be docked
	/// automatically when input is requested.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>When this property is set to <see langword="true"/>,
	/// the cursor will be automatically docked every time
	/// <c>Input.GetNextChar</c> is called.  This will typically
	/// move the cursor out of the way after screen updates,
	/// and just before the automatic screen refresh.</para>
	/// </remarks>
	public static bool DockedCursor
			{
				get
				{
					return dockedCursor;
				}
				set
				{
					dockedCursor = value;
				}
			}

	/// <summary>
	/// <para>Get or set the "sound enabled" state.</para>
	/// </summary>
	///
	/// <value>
	/// <para>Returns <see langword="true"/> if sound has been enabled.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>If this property is set to <see langword="false"/>, then
	/// the <c>Beep</c> and <c>Flash</c> methods will do nothing.
	/// This is a convenient way to disable sound effects in games.</para>
	/// </remarks>
	public static bool SoundEnabled
			{
				get
				{
					return soundEnabled;
				}
				set
				{
					soundEnabled = value;
				}
			}

	/// <summary>
	/// <para>Get or set the "report timeout" state.</para>
	/// </summary>
	///
	/// <value>
	/// <para>Returns <see langword="true"/> if timeouts will be
	/// reported via <c>Input.GetNextChar</c>.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>If this property is set to <see langword="true"/>, then
	/// the <c>Input.GetNextChar</c> method will return -1 whenever a
	/// timeout occurs.  The default value is <see langword="false"/>.</para>
	///
	/// <para>Some applications may need to set this property if events
	/// happen on a timeout that may affect the application's main character
	/// input loop.</para>
	/// </remarks>
	public static bool ReportTimeouts
			{
				get
				{
					return reportTimeouts;
				}
				set
				{
					reportTimeouts = value;
				}
			}

} // class Misc

} // namespace Curses
