//
// System.Web.UI.WebControls.ImageMap.cs
//
// Authors:
//	Lluis Sanchez Gual (lluis@novell.com)
//
// (C) 2005 Novell, Inc (http://www.novell.com)
//

//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

#if NET_2_0

using System.ComponentModel;
using System.Security.Permissions;

namespace System.Web.UI.WebControls
{
	[ParseChildren (true, "HotSpots")]
	[DefaultProperty ("HotSpots")]
	[DefaultEvent ("Click")]
	[AspNetHostingPermissionAttribute (SecurityAction.LinkDemand, Level = AspNetHostingPermissionLevel.Minimal)]
	[AspNetHostingPermissionAttribute (SecurityAction.InheritanceDemand, Level = AspNetHostingPermissionLevel.Minimal)]
	public class ImageMap: Image, IPostBackEventHandler
	{
		HotSpotCollection spots;
		
		private static readonly object ClickEvent = new object();
		
		public event ImageMapEventHandler Click
		{
			add { Events.AddHandler (ClickEvent, value); }
			remove { Events.RemoveHandler (ClickEvent, value); }
		}
		
		protected virtual void OnClick (ImageMapEventArgs e)
		{
			if (Events != null) {
				ImageMapEventHandler eh = (ImageMapEventHandler) Events [ClickEvent];
				if (eh!= null) eh (this, e);
			}
		}

	    [DefaultValueAttribute (HotSpotMode.NotSet)]
		public virtual HotSpotMode HotSpotMode {
			get {
				object o = ViewState ["HotSpotMode"];
				return o != null ? (HotSpotMode) o : HotSpotMode.NotSet;
			}
			set {
				ViewState ["HotSpotMode"] = value;
			}
		}
		
	    [DefaultValueAttribute ("")]
		public virtual string Target {
			get {
				object o = ViewState ["Target"];
				return o != null ? (string) o : "";
			}
			set {
				ViewState ["Target"] = value;
			}
		}

	    [NotifyParentPropertyAttribute (true)]
	    [PersistenceModeAttribute (PersistenceMode.InnerDefaultProperty)]
	    [DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Content)]
		public HotSpotCollection HotSpots {
			get {
				if (spots == null) {
					spots = new HotSpotCollection ();
					if (IsTrackingViewState)
						((IStateManager)spots).TrackViewState ();
				}
				return spots;
			}
		}
		
		protected override void TrackViewState ()
		{
			base.TrackViewState ();
			if (spots != null) ((IStateManager)spots).TrackViewState ();
		}
		
		protected override object SaveViewState ()
		{
			object ob1 = base.SaveViewState ();
			object ob2 = spots != null ? ((IStateManager)spots).SaveViewState () : null;
			
			if (ob1 != null || ob2 != null)
				return new Pair (ob1, ob2);
			else
				return null;
		}
		
		protected override void LoadViewState (object savedState)
		{
			if (savedState == null) {
				base.LoadViewState (null);
				return;
			}
			
			Pair pair = (Pair) savedState;
			base.LoadViewState (pair.First);
			((IStateManager)HotSpots).LoadViewState (pair.Second);
		}
		
		public void RaisePostBackEvent (string eventArgument)
		{
			HotSpot spot = HotSpots [int.Parse (eventArgument)];
			OnClick (new ImageMapEventArgs (spot.PostBackValue));
		}

		protected override void AddAttributesToRender (HtmlTextWriter writer)
		{
			base.AddAttributesToRender (writer);
			writer.AddAttribute (HtmlTextWriterAttribute.Usemap, "#ImageMap" + ClientID);
		}
		
		protected override void Render (HtmlTextWriter writer)
		{
			base.Render (writer);
			writer.AddAttribute (HtmlTextWriterAttribute.Name, "ImageMap" + ClientID);
			writer.RenderBeginTag (HtmlTextWriterTag.Map);
			
			if (spots != null) {
				for (int n=0; n<spots.Count; n++) {
					HotSpot spot = spots [n];
					writer.AddAttribute (HtmlTextWriterAttribute.Shape, spot.MarkupName);
					writer.AddAttribute (HtmlTextWriterAttribute.Coords, spot.GetCoordinates ());
					writer.AddAttribute (HtmlTextWriterAttribute.Title, spot.AlternateText);
					writer.AddAttribute (HtmlTextWriterAttribute.Alt, spot.AlternateText);
					if (spot.AccessKey.Length > 0)
						writer.AddAttribute (HtmlTextWriterAttribute.Accesskey, spot.AccessKey);
					if (spot.TabIndex != 0)
						writer.AddAttribute (HtmlTextWriterAttribute.Tabindex, spot.TabIndex.ToString ());
					
					HotSpotMode mode = spot.HotSpotMode != HotSpotMode.NotSet ? spot.HotSpotMode : HotSpotMode;
					switch (mode) {
						case HotSpotMode.Inactive:
							writer.AddAttribute ("nohref", "true");
							break;
						case HotSpotMode.Navigate:
							string target = spot.Target.Length > 0 ? spot.Target : Target; 
							writer.AddAttribute (HtmlTextWriterAttribute.Href, spot.NavigateUrl);
							writer.AddAttribute (HtmlTextWriterAttribute.Target, target);
							break;
						case HotSpotMode.PostBack:
							writer.AddAttribute (HtmlTextWriterAttribute.Href, Page.ClientScript.GetPostBackClientHyperlink (this, n.ToString()));
							break;
					}
						
					writer.RenderBeginTag (HtmlTextWriterTag.Area);
					writer.RenderEndTag ();
				}
			} 
			
			writer.RenderEndTag ();
		}
	}
}

#endif
