//
// assembly:	System_test
// namespace:	MonoTests.System.Text.RegularExpressions
// file:	PerlTrials.cs
//
// author:	Dan Lewis (dlewis@gmx.co.uk)
// 		(c) 2002

using System.Text.RegularExpressions;

namespace MonoTests.System.Text.RegularExpressions {

	class PerlTrials {
		// ported from perl-5.6.1/t/op/re_tests

		public static RegexTrial[] trials = new RegexTrial[] {
			new RegexTrial (@"abc", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"abc", RegexOptions.None, "xbc", "Fail."),
			new RegexTrial (@"abc", RegexOptions.None, "axc", "Fail."),
			new RegexTrial (@"abc", RegexOptions.None, "abx", "Fail."),
			new RegexTrial (@"abc", RegexOptions.None, "xabcy", "Pass. Group[0]=(1,3)"),
			new RegexTrial (@"abc", RegexOptions.None, "ababc", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"ab*c", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab*bc", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab*bc", RegexOptions.None, "abbc", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"ab*bc", RegexOptions.None, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@".{1}", RegexOptions.None, "abbbbc", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@".{3,4}", RegexOptions.None, "abbbbc", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"ab{0,}bc", RegexOptions.None, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab+bc", RegexOptions.None, "abbc", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"ab+bc", RegexOptions.None, "abc", "Fail."),
			new RegexTrial (@"ab+bc", RegexOptions.None, "abq", "Fail."),
			new RegexTrial (@"ab{1,}bc", RegexOptions.None, "abq", "Fail."),
			new RegexTrial (@"ab+bc", RegexOptions.None, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{1,}bc", RegexOptions.None, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{1,3}bc", RegexOptions.None, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{3,4}bc", RegexOptions.None, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{4,5}bc", RegexOptions.None, "abbbbc", "Fail."),
			new RegexTrial (@"ab?bc", RegexOptions.None, "abbc", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"ab?bc", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab{0,1}bc", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab?bc", RegexOptions.None, "abbbbc", "Fail."),
			new RegexTrial (@"ab?c", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab{0,1}c", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^abc$", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^abc$", RegexOptions.None, "abcc", "Fail."),
			new RegexTrial (@"^abc", RegexOptions.None, "abcc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^abc$", RegexOptions.None, "aabc", "Fail."),
			new RegexTrial (@"abc$", RegexOptions.None, "aabc", "Pass. Group[0]=(1,3)"),
			new RegexTrial (@"abc$", RegexOptions.None, "aabcd", "Fail."),
			new RegexTrial (@"^", RegexOptions.None, "abc", "Pass. Group[0]=(0,0)"),
			new RegexTrial (@"$", RegexOptions.None, "abc", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"a.c", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a.c", RegexOptions.None, "axc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a.*c", RegexOptions.None, "axyzc", "Pass. Group[0]=(0,5)"),
			new RegexTrial (@"a.*c", RegexOptions.None, "axyzd", "Fail."),
			new RegexTrial (@"a[bc]d", RegexOptions.None, "abc", "Fail."),
			new RegexTrial (@"a[bc]d", RegexOptions.None, "abd", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[b-d]e", RegexOptions.None, "abd", "Fail."),
			new RegexTrial (@"a[b-d]e", RegexOptions.None, "ace", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[b-d]", RegexOptions.None, "aac", "Pass. Group[0]=(1,2)"),
			new RegexTrial (@"a[-b]", RegexOptions.None, "a-", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a[b-]", RegexOptions.None, "a-", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a[b-a]", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"a[]b", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"a[", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"a]", RegexOptions.None, "a]", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a[]]b", RegexOptions.None, "a]b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[^bc]d", RegexOptions.None, "aed", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[^bc]d", RegexOptions.None, "abd", "Fail."),
			new RegexTrial (@"a[^-b]c", RegexOptions.None, "adc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[^-b]c", RegexOptions.None, "a-c", "Fail."),
			new RegexTrial (@"a[^]b]c", RegexOptions.None, "a]c", "Fail."),
			new RegexTrial (@"a[^]b]c", RegexOptions.None, "adc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"\ba\b", RegexOptions.None, "a-", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"\ba\b", RegexOptions.None, "-a", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"\ba\b", RegexOptions.None, "-a-", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"\by\b", RegexOptions.None, "xy", "Fail."),
			new RegexTrial (@"\by\b", RegexOptions.None, "yz", "Fail."),
			new RegexTrial (@"\by\b", RegexOptions.None, "xyz", "Fail."),
			new RegexTrial (@"\Ba\B", RegexOptions.None, "a-", "Fail."),
			new RegexTrial (@"\Ba\B", RegexOptions.None, "-a", "Fail."),
			new RegexTrial (@"\Ba\B", RegexOptions.None, "-a-", "Fail."),
			new RegexTrial (@"\By\b", RegexOptions.None, "xy", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"\by\B", RegexOptions.None, "yz", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"\By\B", RegexOptions.None, "xyz", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"\w", RegexOptions.None, "a", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"\w", RegexOptions.None, "-", "Fail."),
			new RegexTrial (@"\W", RegexOptions.None, "a", "Fail."),
			new RegexTrial (@"\W", RegexOptions.None, "-", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"a\sb", RegexOptions.None, "a b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a\sb", RegexOptions.None, "a-b", "Fail."),
			new RegexTrial (@"a\Sb", RegexOptions.None, "a b", "Fail."),
			new RegexTrial (@"a\Sb", RegexOptions.None, "a-b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"\d", RegexOptions.None, "1", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"\d", RegexOptions.None, "-", "Fail."),
			new RegexTrial (@"\D", RegexOptions.None, "1", "Fail."),
			new RegexTrial (@"\D", RegexOptions.None, "-", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"[\w]", RegexOptions.None, "a", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"[\w]", RegexOptions.None, "-", "Fail."),
			new RegexTrial (@"[\W]", RegexOptions.None, "a", "Fail."),
			new RegexTrial (@"[\W]", RegexOptions.None, "-", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"a[\s]b", RegexOptions.None, "a b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[\s]b", RegexOptions.None, "a-b", "Fail."),
			new RegexTrial (@"a[\S]b", RegexOptions.None, "a b", "Fail."),
			new RegexTrial (@"a[\S]b", RegexOptions.None, "a-b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"[\d]", RegexOptions.None, "1", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"[\d]", RegexOptions.None, "-", "Fail."),
			new RegexTrial (@"[\D]", RegexOptions.None, "1", "Fail."),
			new RegexTrial (@"[\D]", RegexOptions.None, "-", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"ab|cd", RegexOptions.None, "abc", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"ab|cd", RegexOptions.None, "abcd", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"()ef", RegexOptions.None, "def", "Pass. Group[0]=(1,2) Group[1]=(1,0)"),
			new RegexTrial (@"*a", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"(*)b", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"$b", RegexOptions.None, "b", "Fail."),
			new RegexTrial (@"a\", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"a\(b", RegexOptions.None, "a(b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a\(*b", RegexOptions.None, "ab", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a\(*b", RegexOptions.None, "a((b", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"a\\b", RegexOptions.None, "a\\b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"abc)", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"(abc", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"((a))", RegexOptions.None, "abc", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1)"),
			new RegexTrial (@"(a)b(c)", RegexOptions.None, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(2,1)"),
			new RegexTrial (@"a+b+c", RegexOptions.None, "aabbabc", "Pass. Group[0]=(4,3)"),
			new RegexTrial (@"a{1,}b{1,}c", RegexOptions.None, "aabbabc", "Pass. Group[0]=(4,3)"),
			new RegexTrial (@"a**", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"a.+?c", RegexOptions.None, "abcabc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"(a+|b)*", RegexOptions.None, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)"),
			new RegexTrial (@"(a+|b){0,}", RegexOptions.None, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)"),
			new RegexTrial (@"(a+|b)+", RegexOptions.None, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)"),
			new RegexTrial (@"(a+|b){1,}", RegexOptions.None, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)"),
			new RegexTrial (@"(a+|b)?", RegexOptions.None, "ab", "Pass. Group[0]=(0,1) Group[1]=(0,1)"),
			new RegexTrial (@"(a+|b){0,1}", RegexOptions.None, "ab", "Pass. Group[0]=(0,1) Group[1]=(0,1)"),
			new RegexTrial (@")(", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"[^ab]*", RegexOptions.None, "cde", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"abc", RegexOptions.None, "", "Fail."),
			new RegexTrial (@"a*", RegexOptions.None, "", "Pass. Group[0]=(0,0)"),
			new RegexTrial (@"([abc])*d", RegexOptions.None, "abbbcd", "Pass. Group[0]=(0,6) Group[1]=(0,1)(1,1)(2,1)(3,1)(4,1)"),
			new RegexTrial (@"([abc])*bcd", RegexOptions.None, "abcd", "Pass. Group[0]=(0,4) Group[1]=(0,1)"),
			new RegexTrial (@"a|b|c|d|e", RegexOptions.None, "e", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"(a|b|c|d|e)f", RegexOptions.None, "ef", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"abcd*efg", RegexOptions.None, "abcdefg", "Pass. Group[0]=(0,7)"),
			new RegexTrial (@"ab*", RegexOptions.None, "xabyabbbz", "Pass. Group[0]=(1,2)"),
			new RegexTrial (@"ab*", RegexOptions.None, "xayabbbz", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"(ab|cd)e", RegexOptions.None, "abcde", "Pass. Group[0]=(2,3) Group[1]=(2,2)"),
			new RegexTrial (@"[abhgefdc]ij", RegexOptions.None, "hij", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^(ab|cd)e", RegexOptions.None, "abcde", "Fail."),
			new RegexTrial (@"(abc|)ef", RegexOptions.None, "abcdef", "Pass. Group[0]=(4,2) Group[1]=(4,0)"),
			new RegexTrial (@"(a|b)c*d", RegexOptions.None, "abcd", "Pass. Group[0]=(1,3) Group[1]=(1,1)"),
			new RegexTrial (@"(ab|ab*)bc", RegexOptions.None, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,1)"),
			new RegexTrial (@"a([bc]*)c*", RegexOptions.None, "abc", "Pass. Group[0]=(0,3) Group[1]=(1,2)"),
			new RegexTrial (@"a([bc]*)(c*d)", RegexOptions.None, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,2) Group[2]=(3,1)"),
			new RegexTrial (@"a([bc]+)(c*d)", RegexOptions.None, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,2) Group[2]=(3,1)"),
			new RegexTrial (@"a([bc]*)(c+d)", RegexOptions.None, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,1) Group[2]=(2,2)"),
			new RegexTrial (@"a[bcd]*dcdcde", RegexOptions.None, "adcdcde", "Pass. Group[0]=(0,7)"),
			new RegexTrial (@"a[bcd]+dcdcde", RegexOptions.None, "adcdcde", "Fail."),
			new RegexTrial (@"(ab|a)b*c", RegexOptions.None, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,2)"),
			new RegexTrial (@"((a)(b)c)(d)", RegexOptions.None, "abcd", "Pass. Group[0]=(0,4) Group[1]=(0,3) Group[2]=(0,1) Group[3]=(1,1) Group[4]=(3,1)"),
			new RegexTrial (@"[a-zA-Z_][a-zA-Z0-9_]*", RegexOptions.None, "alpha", "Pass. Group[0]=(0,5)"),
			new RegexTrial (@"^a(bc+|b[eh])g|.h$", RegexOptions.None, "abh", "Pass. Group[0]=(1,2) Group[1]="),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.None, "effgz", "Pass. Group[0]=(0,5) Group[1]=(0,5) Group[2]="),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.None, "ij", "Pass. Group[0]=(0,2) Group[1]=(0,2) Group[2]=(1,1)"),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.None, "effg", "Fail."),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.None, "bcdd", "Fail."),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.None, "reffgz", "Pass. Group[0]=(1,5) Group[1]=(1,5) Group[2]="),
			new RegexTrial (@"((((((((((a))))))))))", RegexOptions.None, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)"),
			new RegexTrial (@"((((((((((a))))))))))\10", RegexOptions.None, "aa", "Pass. Group[0]=(0,2) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)"),
			new RegexTrial (@"((((((((((a))))))))))!", RegexOptions.None, "aa", "Fail."),
			new RegexTrial (@"((((((((((a))))))))))!", RegexOptions.None, "a!", "Pass. Group[0]=(0,2) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)"),
			new RegexTrial (@"(((((((((a)))))))))", RegexOptions.None, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1)"),
			new RegexTrial (@"multiple words of text", RegexOptions.None, "uh-uh", "Fail."),
			new RegexTrial (@"multiple words", RegexOptions.None, "multiple words, yeah", "Pass. Group[0]=(0,14)"),
			new RegexTrial (@"(.*)c(.*)", RegexOptions.None, "abcde", "Pass. Group[0]=(0,5) Group[1]=(0,2) Group[2]=(3,2)"),
			new RegexTrial (@"\((.*), (.*)\)", RegexOptions.None, "(a, b)", "Pass. Group[0]=(0,6) Group[1]=(1,1) Group[2]=(4,1)"),
			new RegexTrial (@"[k]", RegexOptions.None, "ab", "Fail."),
			new RegexTrial (@"abcd", RegexOptions.None, "abcd", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"a(bc)d", RegexOptions.None, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,2)"),
			new RegexTrial (@"a[-]?c", RegexOptions.None, "ac", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"(abc)\1", RegexOptions.None, "abcabc", "Pass. Group[0]=(0,6) Group[1]=(0,3)"),
			new RegexTrial (@"([a-c]*)\1", RegexOptions.None, "abcabc", "Pass. Group[0]=(0,6) Group[1]=(0,3)"),
			new RegexTrial (@"\1", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"\2", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"(a)|\1", RegexOptions.None, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1)"),
			new RegexTrial (@"(a)|\1", RegexOptions.None, "x", "Fail."),
			new RegexTrial (@"(a)|\2", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"(([a-c])b*?\2)*", RegexOptions.None, "ababbbcbc", "Pass. Group[0]=(0,5) Group[1]=(0,3)(3,2) Group[2]=(0,1)(3,1)"),
			new RegexTrial (@"(([a-c])b*?\2){3}", RegexOptions.None, "ababbbcbc", "Pass. Group[0]=(0,9) Group[1]=(0,3)(3,3)(6,3) Group[2]=(0,1)(3,1)(6,1)"),
			new RegexTrial (@"((\3|b)\2(a)x)+", RegexOptions.None, "aaxabxbaxbbx", "Fail."),
			new RegexTrial (@"((\3|b)\2(a)x)+", RegexOptions.None, "aaaxabaxbaaxbbax", "Pass. Group[0]=(12,4) Group[1]=(12,4) Group[2]=(12,1) Group[3]=(14,1)"),
			new RegexTrial (@"((\3|b)\2(a)){2,}", RegexOptions.None, "bbaababbabaaaaabbaaaabba", "Pass. Group[0]=(15,9) Group[1]=(15,3)(18,3)(21,3) Group[2]=(15,1)(18,1)(21,1) Group[3]=(17,1)(20,1)(23,1)"),
			new RegexTrial (@"abc", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"abc", RegexOptions.IgnoreCase, "XBC", "Fail."),
			new RegexTrial (@"abc", RegexOptions.IgnoreCase, "AXC", "Fail."),
			new RegexTrial (@"abc", RegexOptions.IgnoreCase, "ABX", "Fail."),
			new RegexTrial (@"abc", RegexOptions.IgnoreCase, "XABCY", "Pass. Group[0]=(1,3)"),
			new RegexTrial (@"abc", RegexOptions.IgnoreCase, "ABABC", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"ab*c", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab*bc", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab*bc", RegexOptions.IgnoreCase, "ABBC", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"ab*?bc", RegexOptions.IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{0,}?bc", RegexOptions.IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab+?bc", RegexOptions.IgnoreCase, "ABBC", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"ab+bc", RegexOptions.IgnoreCase, "ABC", "Fail."),
			new RegexTrial (@"ab+bc", RegexOptions.IgnoreCase, "ABQ", "Fail."),
			new RegexTrial (@"ab{1,}bc", RegexOptions.IgnoreCase, "ABQ", "Fail."),
			new RegexTrial (@"ab+bc", RegexOptions.IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{1,}?bc", RegexOptions.IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{1,3}?bc", RegexOptions.IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{3,4}?bc", RegexOptions.IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{4,5}?bc", RegexOptions.IgnoreCase, "ABBBBC", "Fail."),
			new RegexTrial (@"ab??bc", RegexOptions.IgnoreCase, "ABBC", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"ab??bc", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab{0,1}?bc", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab??bc", RegexOptions.IgnoreCase, "ABBBBC", "Fail."),
			new RegexTrial (@"ab??c", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab{0,1}?c", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^abc$", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^abc$", RegexOptions.IgnoreCase, "ABCC", "Fail."),
			new RegexTrial (@"^abc", RegexOptions.IgnoreCase, "ABCC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^abc$", RegexOptions.IgnoreCase, "AABC", "Fail."),
			new RegexTrial (@"abc$", RegexOptions.IgnoreCase, "AABC", "Pass. Group[0]=(1,3)"),
			new RegexTrial (@"^", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,0)"),
			new RegexTrial (@"$", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"a.c", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a.c", RegexOptions.IgnoreCase, "AXC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a.*?c", RegexOptions.IgnoreCase, "AXYZC", "Pass. Group[0]=(0,5)"),
			new RegexTrial (@"a.*c", RegexOptions.IgnoreCase, "AXYZD", "Fail."),
			new RegexTrial (@"a[bc]d", RegexOptions.IgnoreCase, "ABC", "Fail."),
			new RegexTrial (@"a[bc]d", RegexOptions.IgnoreCase, "ABD", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[b-d]e", RegexOptions.IgnoreCase, "ABD", "Fail."),
			new RegexTrial (@"a[b-d]e", RegexOptions.IgnoreCase, "ACE", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[b-d]", RegexOptions.IgnoreCase, "AAC", "Pass. Group[0]=(1,2)"),
			new RegexTrial (@"a[-b]", RegexOptions.IgnoreCase, "A-", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a[b-]", RegexOptions.IgnoreCase, "A-", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a[b-a]", RegexOptions.IgnoreCase, "-", "Error."),
			new RegexTrial (@"a[]b", RegexOptions.IgnoreCase, "-", "Error."),
			new RegexTrial (@"a[", RegexOptions.IgnoreCase, "-", "Error."),
			new RegexTrial (@"a]", RegexOptions.IgnoreCase, "A]", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a[]]b", RegexOptions.IgnoreCase, "A]B", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[^bc]d", RegexOptions.IgnoreCase, "AED", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[^bc]d", RegexOptions.IgnoreCase, "ABD", "Fail."),
			new RegexTrial (@"a[^-b]c", RegexOptions.IgnoreCase, "ADC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[^-b]c", RegexOptions.IgnoreCase, "A-C", "Fail."),
			new RegexTrial (@"a[^]b]c", RegexOptions.IgnoreCase, "A]C", "Fail."),
			new RegexTrial (@"a[^]b]c", RegexOptions.IgnoreCase, "ADC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab|cd", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"ab|cd", RegexOptions.IgnoreCase, "ABCD", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"()ef", RegexOptions.IgnoreCase, "DEF", "Pass. Group[0]=(1,2) Group[1]=(1,0)"),
			new RegexTrial (@"*a", RegexOptions.IgnoreCase, "-", "Error."),
			new RegexTrial (@"(*)b", RegexOptions.IgnoreCase, "-", "Error."),
			new RegexTrial (@"$b", RegexOptions.IgnoreCase, "B", "Fail."),
			new RegexTrial (@"a\", RegexOptions.IgnoreCase, "-", "Error."),
			new RegexTrial (@"a\(b", RegexOptions.IgnoreCase, "A(B", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a\(*b", RegexOptions.IgnoreCase, "AB", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a\(*b", RegexOptions.IgnoreCase, "A((B", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"a\\b", RegexOptions.IgnoreCase, "A\\B", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"abc)", RegexOptions.IgnoreCase, "-", "Error."),
			new RegexTrial (@"(abc", RegexOptions.IgnoreCase, "-", "Error."),
			new RegexTrial (@"((a))", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1)"),
			new RegexTrial (@"(a)b(c)", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(2,1)"),
			new RegexTrial (@"a+b+c", RegexOptions.IgnoreCase, "AABBABC", "Pass. Group[0]=(4,3)"),
			new RegexTrial (@"a{1,}b{1,}c", RegexOptions.IgnoreCase, "AABBABC", "Pass. Group[0]=(4,3)"),
			new RegexTrial (@"a**", RegexOptions.IgnoreCase, "-", "Error."),
			new RegexTrial (@"a.+?c", RegexOptions.IgnoreCase, "ABCABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a.*?c", RegexOptions.IgnoreCase, "ABCABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a.{0,5}?c", RegexOptions.IgnoreCase, "ABCABC", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"(a+|b)*", RegexOptions.IgnoreCase, "AB", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)"),
			new RegexTrial (@"(a+|b){0,}", RegexOptions.IgnoreCase, "AB", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)"),
			new RegexTrial (@"(a+|b)+", RegexOptions.IgnoreCase, "AB", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)"),
			new RegexTrial (@"(a+|b){1,}", RegexOptions.IgnoreCase, "AB", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)"),
			new RegexTrial (@"(a+|b)?", RegexOptions.IgnoreCase, "AB", "Pass. Group[0]=(0,1) Group[1]=(0,1)"),
			new RegexTrial (@"(a+|b){0,1}", RegexOptions.IgnoreCase, "AB", "Pass. Group[0]=(0,1) Group[1]=(0,1)"),
			new RegexTrial (@"(a+|b){0,1}?", RegexOptions.IgnoreCase, "AB", "Pass. Group[0]=(0,0) Group[1]="),
			new RegexTrial (@")(", RegexOptions.IgnoreCase, "-", "Error."),
			new RegexTrial (@"[^ab]*", RegexOptions.IgnoreCase, "CDE", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"abc", RegexOptions.IgnoreCase, "", "Fail."),
			new RegexTrial (@"a*", RegexOptions.IgnoreCase, "", "Pass. Group[0]=(0,0)"),
			new RegexTrial (@"([abc])*d", RegexOptions.IgnoreCase, "ABBBCD", "Pass. Group[0]=(0,6) Group[1]=(0,1)(1,1)(2,1)(3,1)(4,1)"),
			new RegexTrial (@"([abc])*bcd", RegexOptions.IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(0,1)"),
			new RegexTrial (@"a|b|c|d|e", RegexOptions.IgnoreCase, "E", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"(a|b|c|d|e)f", RegexOptions.IgnoreCase, "EF", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"abcd*efg", RegexOptions.IgnoreCase, "ABCDEFG", "Pass. Group[0]=(0,7)"),
			new RegexTrial (@"ab*", RegexOptions.IgnoreCase, "XABYABBBZ", "Pass. Group[0]=(1,2)"),
			new RegexTrial (@"ab*", RegexOptions.IgnoreCase, "XAYABBBZ", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"(ab|cd)e", RegexOptions.IgnoreCase, "ABCDE", "Pass. Group[0]=(2,3) Group[1]=(2,2)"),
			new RegexTrial (@"[abhgefdc]ij", RegexOptions.IgnoreCase, "HIJ", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^(ab|cd)e", RegexOptions.IgnoreCase, "ABCDE", "Fail."),
			new RegexTrial (@"(abc|)ef", RegexOptions.IgnoreCase, "ABCDEF", "Pass. Group[0]=(4,2) Group[1]=(4,0)"),
			new RegexTrial (@"(a|b)c*d", RegexOptions.IgnoreCase, "ABCD", "Pass. Group[0]=(1,3) Group[1]=(1,1)"),
			new RegexTrial (@"(ab|ab*)bc", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3) Group[1]=(0,1)"),
			new RegexTrial (@"a([bc]*)c*", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3) Group[1]=(1,2)"),
			new RegexTrial (@"a([bc]*)(c*d)", RegexOptions.IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(1,2) Group[2]=(3,1)"),
			new RegexTrial (@"a([bc]+)(c*d)", RegexOptions.IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(1,2) Group[2]=(3,1)"),
			new RegexTrial (@"a([bc]*)(c+d)", RegexOptions.IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(1,1) Group[2]=(2,2)"),
			new RegexTrial (@"a[bcd]*dcdcde", RegexOptions.IgnoreCase, "ADCDCDE", "Pass. Group[0]=(0,7)"),
			new RegexTrial (@"a[bcd]+dcdcde", RegexOptions.IgnoreCase, "ADCDCDE", "Fail."),
			new RegexTrial (@"(ab|a)b*c", RegexOptions.IgnoreCase, "ABC", "Pass. Group[0]=(0,3) Group[1]=(0,2)"),
			new RegexTrial (@"((a)(b)c)(d)", RegexOptions.IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(0,3) Group[2]=(0,1) Group[3]=(1,1) Group[4]=(3,1)"),
			new RegexTrial (@"[a-zA-Z_][a-zA-Z0-9_]*", RegexOptions.IgnoreCase, "ALPHA", "Pass. Group[0]=(0,5)"),
			new RegexTrial (@"^a(bc+|b[eh])g|.h$", RegexOptions.IgnoreCase, "ABH", "Pass. Group[0]=(1,2) Group[1]="),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "EFFGZ", "Pass. Group[0]=(0,5) Group[1]=(0,5) Group[2]="),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "IJ", "Pass. Group[0]=(0,2) Group[1]=(0,2) Group[2]=(1,1)"),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "EFFG", "Fail."),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "BCDD", "Fail."),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "REFFGZ", "Pass. Group[0]=(1,5) Group[1]=(1,5) Group[2]="),
			new RegexTrial (@"((((((((((a))))))))))", RegexOptions.IgnoreCase, "A", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)"),
			new RegexTrial (@"((((((((((a))))))))))\10", RegexOptions.IgnoreCase, "AA", "Pass. Group[0]=(0,2) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)"),
			new RegexTrial (@"((((((((((a))))))))))!", RegexOptions.IgnoreCase, "AA", "Fail."),
			new RegexTrial (@"((((((((((a))))))))))!", RegexOptions.IgnoreCase, "A!", "Pass. Group[0]=(0,2) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)"),
			new RegexTrial (@"(((((((((a)))))))))", RegexOptions.IgnoreCase, "A", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1)"),
			new RegexTrial (@"(?:(?:(?:(?:(?:(?:(?:(?:(?:(a))))))))))", RegexOptions.IgnoreCase, "A", "Pass. Group[0]=(0,1) Group[1]=(0,1)"),
			new RegexTrial (@"(?:(?:(?:(?:(?:(?:(?:(?:(?:(a|b|c))))))))))", RegexOptions.IgnoreCase, "C", "Pass. Group[0]=(0,1) Group[1]=(0,1)"),
			new RegexTrial (@"multiple words of text", RegexOptions.IgnoreCase, "UH-UH", "Fail."),
			new RegexTrial (@"multiple words", RegexOptions.IgnoreCase, "MULTIPLE WORDS, YEAH", "Pass. Group[0]=(0,14)"),
			new RegexTrial (@"(.*)c(.*)", RegexOptions.IgnoreCase, "ABCDE", "Pass. Group[0]=(0,5) Group[1]=(0,2) Group[2]=(3,2)"),
			new RegexTrial (@"\((.*), (.*)\)", RegexOptions.IgnoreCase, "(A, B)", "Pass. Group[0]=(0,6) Group[1]=(1,1) Group[2]=(4,1)"),
			new RegexTrial (@"[k]", RegexOptions.IgnoreCase, "AB", "Fail."),
			new RegexTrial (@"abcd", RegexOptions.IgnoreCase, "ABCD", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"a(bc)d", RegexOptions.IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(1,2)"),
			new RegexTrial (@"a[-]?c", RegexOptions.IgnoreCase, "AC", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"(abc)\1", RegexOptions.IgnoreCase, "ABCABC", "Pass. Group[0]=(0,6) Group[1]=(0,3)"),
			new RegexTrial (@"([a-c]*)\1", RegexOptions.IgnoreCase, "ABCABC", "Pass. Group[0]=(0,6) Group[1]=(0,3)"),
			new RegexTrial (@"a(?!b).", RegexOptions.None, "abad", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"a(?=d).", RegexOptions.None, "abad", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"a(?=c|d).", RegexOptions.None, "abad", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"a(?:b|c|d)(.)", RegexOptions.None, "ace", "Pass. Group[0]=(0,3) Group[1]=(2,1)"),
			new RegexTrial (@"a(?:b|c|d)*(.)", RegexOptions.None, "ace", "Pass. Group[0]=(0,3) Group[1]=(2,1)"),
			new RegexTrial (@"a(?:b|c|d)+?(.)", RegexOptions.None, "ace", "Pass. Group[0]=(0,3) Group[1]=(2,1)"),
			new RegexTrial (@"a(?:b|c|d)+?(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,3) Group[1]=(2,1)"),
			new RegexTrial (@"a(?:b|c|d)+(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,8) Group[1]=(7,1)"),
			new RegexTrial (@"a(?:b|c|d){2}(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,4) Group[1]=(3,1)"),
			new RegexTrial (@"a(?:b|c|d){4,5}(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,7) Group[1]=(6,1)"),
			new RegexTrial (@"a(?:b|c|d){4,5}?(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,6) Group[1]=(5,1)"),
			new RegexTrial (@"((foo)|(bar))*", RegexOptions.None, "foobar", "Pass. Group[0]=(0,6) Group[1]=(0,3)(3,3) Group[2]=(0,3) Group[3]=(3,3)"),
			new RegexTrial (@":(?:", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"a(?:b|c|d){6,7}(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,8) Group[1]=(7,1)"),
			new RegexTrial (@"a(?:b|c|d){6,7}?(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,8) Group[1]=(7,1)"),
			new RegexTrial (@"a(?:b|c|d){5,6}(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,8) Group[1]=(7,1)"),
			new RegexTrial (@"a(?:b|c|d){5,6}?(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,7) Group[1]=(6,1)"),
			new RegexTrial (@"a(?:b|c|d){5,7}(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,8) Group[1]=(7,1)"),
			new RegexTrial (@"a(?:b|c|d){5,7}?(.)", RegexOptions.None, "acdbcdbe", "Pass. Group[0]=(0,7) Group[1]=(6,1)"),
			new RegexTrial (@"a(?:b|(c|e){1,2}?|d)+?(.)", RegexOptions.None, "ace", "Pass. Group[0]=(0,3) Group[1]=(1,1) Group[2]=(2,1)"),
			new RegexTrial (@"^(.+)?B", RegexOptions.None, "AB", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"^([^a-z])|(\^)$", RegexOptions.None, ".", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]="),
			new RegexTrial (@"^[<>]&", RegexOptions.None, "<&OUT", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"^(a\1?){4}$", RegexOptions.None, "aaaaaaaaaa", "Pass. Group[0]=(0,10) Group[1]=(0,1)(1,2)(3,3)(6,4)"),
			new RegexTrial (@"^(a\1?){4}$", RegexOptions.None, "aaaaaaaaa", "Fail."),
			new RegexTrial (@"^(a\1?){4}$", RegexOptions.None, "aaaaaaaaaaa", "Fail."),
			new RegexTrial (@"^(a(?(1)\1)){4}$", RegexOptions.None, "aaaaaaaaaa", "Pass. Group[0]=(0,10) Group[1]=(0,1)(1,2)(3,3)(6,4)"),
			new RegexTrial (@"^(a(?(1)\1)){4}$", RegexOptions.None, "aaaaaaaaa", "Fail."),
			new RegexTrial (@"^(a(?(1)\1)){4}$", RegexOptions.None, "aaaaaaaaaaa", "Fail."),
			new RegexTrial (@"((a{4})+)", RegexOptions.None, "aaaaaaaaa", "Pass. Group[0]=(0,8) Group[1]=(0,8) Group[2]=(0,4)(4,4)"),
			new RegexTrial (@"(((aa){2})+)", RegexOptions.None, "aaaaaaaaaa", "Pass. Group[0]=(0,8) Group[1]=(0,8) Group[2]=(0,4)(4,4) Group[3]=(0,2)(2,2)(4,2)(6,2)"),
			new RegexTrial (@"(((a{2}){2})+)", RegexOptions.None, "aaaaaaaaaa", "Pass. Group[0]=(0,8) Group[1]=(0,8) Group[2]=(0,4)(4,4) Group[3]=(0,2)(2,2)(4,2)(6,2)"),
			new RegexTrial (@"(?:(f)(o)(o)|(b)(a)(r))*", RegexOptions.None, "foobar", "Pass. Group[0]=(0,6) Group[1]=(0,1) Group[2]=(1,1) Group[3]=(2,1) Group[4]=(3,1) Group[5]=(4,1) Group[6]=(5,1)"),
			new RegexTrial (@"(?<=a)b", RegexOptions.None, "ab", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"(?<=a)b", RegexOptions.None, "cb", "Fail."),
			new RegexTrial (@"(?<=a)b", RegexOptions.None, "b", "Fail."),
			new RegexTrial (@"(?<!c)b", RegexOptions.None, "ab", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"(?<!c)b", RegexOptions.None, "cb", "Fail."),
			new RegexTrial (@"(?<!c)b", RegexOptions.None, "b", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"(?<!c)b", RegexOptions.None, "b", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"(?<%)b", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"(?:..)*a", RegexOptions.None, "aba", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"(?:..)*?a", RegexOptions.None, "aba", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"^(?:b|a(?=(.)))*\1", RegexOptions.None, "abc", "Pass. Group[0]=(0,2) Group[1]=(1,1)"),
			new RegexTrial (@"^(){3,5}", RegexOptions.None, "abc", "Pass. Group[0]=(0,0) Group[1]=(0,0)(0,0)(0,0)"),
			new RegexTrial (@"^(a+)*ax", RegexOptions.None, "aax", "Pass. Group[0]=(0,3) Group[1]=(0,1)"),
			new RegexTrial (@"^((a|b)+)*ax", RegexOptions.None, "aax", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(0,1)"),
			new RegexTrial (@"^((a|bc)+)*ax", RegexOptions.None, "aax", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(0,1)"),
			new RegexTrial (@"(a|x)*ab", RegexOptions.None, "cab", "Pass. Group[0]=(1,2) Group[1]="),
			new RegexTrial (@"(a)*ab", RegexOptions.None, "cab", "Pass. Group[0]=(1,2) Group[1]="),
			new RegexTrial (@"(?:(?i)a)b", RegexOptions.None, "ab", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"((?i)a)b", RegexOptions.None, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"(?:(?i)a)b", RegexOptions.None, "Ab", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"((?i)a)b", RegexOptions.None, "Ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"(?:(?i)a)b", RegexOptions.None, "aB", "Fail."),
			new RegexTrial (@"((?i)a)b", RegexOptions.None, "aB", "Fail."),
			new RegexTrial (@"(?i:a)b", RegexOptions.None, "ab", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"((?i:a))b", RegexOptions.None, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"(?i:a)b", RegexOptions.None, "Ab", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"((?i:a))b", RegexOptions.None, "Ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"(?i:a)b", RegexOptions.None, "aB", "Fail."),
			new RegexTrial (@"((?i:a))b", RegexOptions.None, "aB", "Fail."),
			new RegexTrial (@"(?:(?-i)a)b", RegexOptions.IgnoreCase, "ab", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"((?-i)a)b", RegexOptions.IgnoreCase, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"(?:(?-i)a)b", RegexOptions.IgnoreCase, "aB", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"((?-i)a)b", RegexOptions.IgnoreCase, "aB", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"(?:(?-i)a)b", RegexOptions.IgnoreCase, "Ab", "Fail."),
			new RegexTrial (@"((?-i)a)b", RegexOptions.IgnoreCase, "Ab", "Fail."),
			new RegexTrial (@"(?:(?-i)a)b", RegexOptions.IgnoreCase, "aB", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"((?-i)a)b", RegexOptions.IgnoreCase, "aB", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"(?:(?-i)a)b", RegexOptions.IgnoreCase, "AB", "Fail."),
			new RegexTrial (@"((?-i)a)b", RegexOptions.IgnoreCase, "AB", "Fail."),
			new RegexTrial (@"(?-i:a)b", RegexOptions.IgnoreCase, "ab", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"((?-i:a))b", RegexOptions.IgnoreCase, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"(?-i:a)b", RegexOptions.IgnoreCase, "aB", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"((?-i:a))b", RegexOptions.IgnoreCase, "aB", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"(?-i:a)b", RegexOptions.IgnoreCase, "Ab", "Fail."),
			new RegexTrial (@"((?-i:a))b", RegexOptions.IgnoreCase, "Ab", "Fail."),
			new RegexTrial (@"(?-i:a)b", RegexOptions.IgnoreCase, "aB", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"((?-i:a))b", RegexOptions.IgnoreCase, "aB", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"(?-i:a)b", RegexOptions.IgnoreCase, "AB", "Fail."),
			new RegexTrial (@"((?-i:a))b", RegexOptions.IgnoreCase, "AB", "Fail."),
			new RegexTrial (@"((?-i:a.))b", RegexOptions.IgnoreCase, "a\nB", "Fail."),
			new RegexTrial (@"((?s-i:a.))b", RegexOptions.IgnoreCase, "a\nB", "Pass. Group[0]=(0,3) Group[1]=(0,2)"),
			new RegexTrial (@"((?s-i:a.))b", RegexOptions.IgnoreCase, "B\nB", "Fail."),
			new RegexTrial (@"(?:c|d)(?:)(?:a(?:)(?:b)(?:b(?:))(?:b(?:)(?:b)))", RegexOptions.None, "cabbbb", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"(?:c|d)(?:)(?:aaaaaaaa(?:)(?:bbbbbbbb)(?:bbbbbbbb(?:))(?:bbbbbbbb(?:)(?:bbbbbbbb)))", RegexOptions.None, "caaaaaaaabbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb", "Pass. Group[0]=(0,41)"),
			new RegexTrial (@"(ab)\d\1", RegexOptions.IgnoreCase, "Ab4ab", "Pass. Group[0]=(0,5) Group[1]=(0,2)"),
			new RegexTrial (@"(ab)\d\1", RegexOptions.IgnoreCase, "ab4Ab", "Pass. Group[0]=(0,5) Group[1]=(0,2)"),
			new RegexTrial (@"foo\w*\d{4}baz", RegexOptions.None, "foobar1234baz", "Pass. Group[0]=(0,13)"),
			new RegexTrial (@"x(~~)*(?:(?:F)?)?", RegexOptions.None, "x~~", "Pass. Group[0]=(0,3) Group[1]=(1,2)"),
			new RegexTrial (@"^a(?#xxx){3}c", RegexOptions.None, "aaac", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"(?<![cd])b", RegexOptions.None, "dbcb", "Fail."),
			new RegexTrial (@"(?<![cd])[ab]", RegexOptions.None, "dbaacb", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"(?<!(c|d))b", RegexOptions.None, "dbcb", "Fail."),
			new RegexTrial (@"(?<!(c|d))[ab]", RegexOptions.None, "dbaacb", "Pass. Group[0]=(2,1) Group[1]="),
			new RegexTrial (@"(?<!cd)[ab]", RegexOptions.None, "cdaccb", "Pass. Group[0]=(5,1)"),
			new RegexTrial (@"^(?:a?b?)*$", RegexOptions.None, "a--", "Fail."),
			new RegexTrial (@"((?s)^a(.))((?m)^b$)", RegexOptions.None, "a\nb\nc\n", "Pass. Group[0]=(0,3) Group[1]=(0,2) Group[2]=(1,1) Group[3]=(2,1)"),
			new RegexTrial (@"((?m)^b$)", RegexOptions.None, "a\nb\nc\n", "Pass. Group[0]=(2,1) Group[1]=(2,1)"),
			new RegexTrial (@"(?m)^b", RegexOptions.None, "a\nb\n", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"(?m)^(b)", RegexOptions.None, "a\nb\n", "Pass. Group[0]=(2,1) Group[1]=(2,1)"),
			new RegexTrial (@"((?m)^b)", RegexOptions.None, "a\nb\n", "Pass. Group[0]=(2,1) Group[1]=(2,1)"),
			new RegexTrial (@"\n((?m)^b)", RegexOptions.None, "a\nb\n", "Pass. Group[0]=(1,2) Group[1]=(2,1)"),
			new RegexTrial (@"((?s).)c(?!.)", RegexOptions.None, "a\nb\nc\n", "Pass. Group[0]=(3,2) Group[1]=(3,1)"),
			new RegexTrial (@"((?s).)c(?!.)", RegexOptions.None, "a\nb\nc\n", "Pass. Group[0]=(3,2) Group[1]=(3,1)"),
			new RegexTrial (@"((?s)b.)c(?!.)", RegexOptions.None, "a\nb\nc\n", "Pass. Group[0]=(2,3) Group[1]=(2,2)"),
			new RegexTrial (@"((?s)b.)c(?!.)", RegexOptions.None, "a\nb\nc\n", "Pass. Group[0]=(2,3) Group[1]=(2,2)"),
			new RegexTrial (@"^b", RegexOptions.None, "a\nb\nc\n", "Fail."),
			new RegexTrial (@"()^b", RegexOptions.None, "a\nb\nc\n", "Fail."),
			new RegexTrial (@"((?m)^b)", RegexOptions.None, "a\nb\nc\n", "Pass. Group[0]=(2,1) Group[1]=(2,1)"),
			new RegexTrial (@"(x)?(?(1)a|b)", RegexOptions.None, "a", "Fail."),
			new RegexTrial (@"(x)?(?(1)b|a)", RegexOptions.None, "a", "Pass. Group[0]=(0,1) Group[1]="),
			new RegexTrial (@"()?(?(1)b|a)", RegexOptions.None, "a", "Pass. Group[0]=(0,1) Group[1]="),
			new RegexTrial (@"()(?(1)b|a)", RegexOptions.None, "a", "Fail."),
			new RegexTrial (@"()?(?(1)a|b)", RegexOptions.None, "a", "Pass. Group[0]=(0,1) Group[1]=(0,0)"),
			new RegexTrial (@"^(\()?blah(?(1)(\)))$", RegexOptions.None, "(blah)", "Pass. Group[0]=(0,6) Group[1]=(0,1) Group[2]=(5,1)"),
			new RegexTrial (@"^(\()?blah(?(1)(\)))$", RegexOptions.None, "blah", "Pass. Group[0]=(0,4) Group[1]= Group[2]="),
			new RegexTrial (@"^(\()?blah(?(1)(\)))$", RegexOptions.None, "blah)", "Fail."),
			new RegexTrial (@"^(\()?blah(?(1)(\)))$", RegexOptions.None, "(blah", "Fail."),
			new RegexTrial (@"^(\(+)?blah(?(1)(\)))$", RegexOptions.None, "(blah)", "Pass. Group[0]=(0,6) Group[1]=(0,1) Group[2]=(5,1)"),
			new RegexTrial (@"^(\(+)?blah(?(1)(\)))$", RegexOptions.None, "blah", "Pass. Group[0]=(0,4) Group[1]= Group[2]="),
			new RegexTrial (@"^(\(+)?blah(?(1)(\)))$", RegexOptions.None, "blah)", "Fail."),
			new RegexTrial (@"^(\(+)?blah(?(1)(\)))$", RegexOptions.None, "(blah", "Fail."),
			new RegexTrial (@"(?(1)a|b|c)", RegexOptions.None, "a", "Error."),
			new RegexTrial (@"(?(?!a)a|b)", RegexOptions.None, "a", "Fail."),
			new RegexTrial (@"(?(?!a)b|a)", RegexOptions.None, "a", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"(?(?=a)b|a)", RegexOptions.None, "a", "Fail."),
			new RegexTrial (@"(?(?=a)a|b)", RegexOptions.None, "a", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"(?=(a+?))(\1ab)", RegexOptions.None, "aaab", "Pass. Group[0]=(1,3) Group[1]=(1,1) Group[2]=(1,3)"),
			new RegexTrial (@"^(?=(a+?))\1ab", RegexOptions.None, "aaab", "Fail."),
			new RegexTrial (@"(\w+:)+", RegexOptions.None, "one:", "Pass. Group[0]=(0,4) Group[1]=(0,4)"),
			new RegexTrial (@"$(?<=^(a))", RegexOptions.None, "a", "Pass. Group[0]=(1,0) Group[1]=(0,1)"),
			new RegexTrial (@"(?=(a+?))(\1ab)", RegexOptions.None, "aaab", "Pass. Group[0]=(1,3) Group[1]=(1,1) Group[2]=(1,3)"),
			new RegexTrial (@"^(?=(a+?))\1ab", RegexOptions.None, "aaab", "Fail."),
			new RegexTrial (@"([\w:]+::)?(\w+)$", RegexOptions.None, "abcd:", "Fail."),
			new RegexTrial (@"([\w:]+::)?(\w+)$", RegexOptions.None, "abcd", "Pass. Group[0]=(0,4) Group[1]= Group[2]=(0,4)"),
			new RegexTrial (@"([\w:]+::)?(\w+)$", RegexOptions.None, "xy:z:::abcd", "Pass. Group[0]=(0,11) Group[1]=(0,7) Group[2]=(7,4)"),
			new RegexTrial (@"^[^bcd]*(c+)", RegexOptions.None, "aexycd", "Pass. Group[0]=(0,5) Group[1]=(4,1)"),
			new RegexTrial (@"(a*)b+", RegexOptions.None, "caab", "Pass. Group[0]=(1,3) Group[1]=(1,2)"),
			new RegexTrial (@"([\w:]+::)?(\w+)$", RegexOptions.None, "abcd:", "Fail."),
			new RegexTrial (@"([\w:]+::)?(\w+)$", RegexOptions.None, "abcd", "Pass. Group[0]=(0,4) Group[1]= Group[2]=(0,4)"),
			new RegexTrial (@"([\w:]+::)?(\w+)$", RegexOptions.None, "xy:z:::abcd", "Pass. Group[0]=(0,11) Group[1]=(0,7) Group[2]=(7,4)"),
			new RegexTrial (@"^[^bcd]*(c+)", RegexOptions.None, "aexycd", "Pass. Group[0]=(0,5) Group[1]=(4,1)"),
			new RegexTrial (@"(>a+)ab", RegexOptions.None, "aaab", "Fail."),
			new RegexTrial (@"(?>a+)b", RegexOptions.None, "aaab", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"([[:]+)", RegexOptions.None, "a:[b]:", "Pass. Group[0]=(1,2) Group[1]=(1,2)"),
			new RegexTrial (@"([[=]+)", RegexOptions.None, "a=[b]=", "Pass. Group[0]=(1,2) Group[1]=(1,2)"),
			new RegexTrial (@"([[.]+)", RegexOptions.None, "a.[b].", "Pass. Group[0]=(1,2) Group[1]=(1,2)"),
			new RegexTrial (@"[a[:]b[:c]", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"[a[:]b[:c]", RegexOptions.None, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"((?>a+)b)", RegexOptions.None, "aaab", "Pass. Group[0]=(0,4) Group[1]=(0,4)"),
			new RegexTrial (@"(?>(a+))b", RegexOptions.None, "aaab", "Pass. Group[0]=(0,4) Group[1]=(0,3)"),
			new RegexTrial (@"((?>[^()]+)|\([^()]*\))+", RegexOptions.None, "((abc(ade)ufh()()x", "Pass. Group[0]=(2,16) Group[1]=(2,3)(5,5)(10,3)(13,2)(15,2)(17,1)"),
			new RegexTrial (@"(?<=x+)", RegexOptions.None, "xxxxy", "Pass. Group[0]=(1,0)"),
			new RegexTrial (@"a{37,17}", RegexOptions.None, "-", "Error."),
			new RegexTrial (@"\Z", RegexOptions.None, "a\nb\n", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"\z", RegexOptions.None, "a\nb\n", "Pass. Group[0]=(4,0)"),
			new RegexTrial (@"$", RegexOptions.None, "a\nb\n", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"\Z", RegexOptions.None, "b\na\n", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"\z", RegexOptions.None, "b\na\n", "Pass. Group[0]=(4,0)"),
			new RegexTrial (@"$", RegexOptions.None, "b\na\n", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"\Z", RegexOptions.None, "b\na", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"\z", RegexOptions.None, "b\na", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"$", RegexOptions.None, "b\na", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"\Z", RegexOptions.Multiline, "a\nb\n", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"\z", RegexOptions.Multiline, "a\nb\n", "Pass. Group[0]=(4,0)"),
			new RegexTrial (@"$", RegexOptions.Multiline, "a\nb\n", "Pass. Group[0]=(1,0)"),
			new RegexTrial (@"\Z", RegexOptions.Multiline, "b\na\n", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"\z", RegexOptions.Multiline, "b\na\n", "Pass. Group[0]=(4,0)"),
			new RegexTrial (@"$", RegexOptions.Multiline, "b\na\n", "Pass. Group[0]=(1,0)"),
			new RegexTrial (@"\Z", RegexOptions.Multiline, "b\na", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"\z", RegexOptions.Multiline, "b\na", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"$", RegexOptions.Multiline, "b\na", "Pass. Group[0]=(1,0)"),
			new RegexTrial (@"a\Z", RegexOptions.None, "a\nb\n", "Fail."),
			new RegexTrial (@"a\z", RegexOptions.None, "a\nb\n", "Fail."),
			new RegexTrial (@"a$", RegexOptions.None, "a\nb\n", "Fail."),
			new RegexTrial (@"a\Z", RegexOptions.None, "b\na\n", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"a\z", RegexOptions.None, "b\na\n", "Fail."),
			new RegexTrial (@"a$", RegexOptions.None, "b\na\n", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"a\Z", RegexOptions.None, "b\na", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"a\z", RegexOptions.None, "b\na", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"a$", RegexOptions.None, "b\na", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"a\z", RegexOptions.Multiline, "a\nb\n", "Fail."),
			new RegexTrial (@"a$", RegexOptions.Multiline, "a\nb\n", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"a\Z", RegexOptions.Multiline, "b\na\n", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"a\z", RegexOptions.Multiline, "b\na\n", "Fail."),
			new RegexTrial (@"a$", RegexOptions.Multiline, "b\na\n", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"a\Z", RegexOptions.Multiline, "b\na", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"a\z", RegexOptions.Multiline, "b\na", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"a$", RegexOptions.Multiline, "b\na", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"aa\Z", RegexOptions.None, "aa\nb\n", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.None, "aa\nb\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.None, "aa\nb\n", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.None, "b\naa\n", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"aa\z", RegexOptions.None, "b\naa\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.None, "b\naa\n", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"aa\Z", RegexOptions.None, "b\naa", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"aa\z", RegexOptions.None, "b\naa", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"aa$", RegexOptions.None, "b\naa", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"aa\z", RegexOptions.Multiline, "aa\nb\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.Multiline, "aa\nb\n", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"aa\Z", RegexOptions.Multiline, "b\naa\n", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"aa\z", RegexOptions.Multiline, "b\naa\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.Multiline, "b\naa\n", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"aa\Z", RegexOptions.Multiline, "b\naa", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"aa\z", RegexOptions.Multiline, "b\naa", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"aa$", RegexOptions.Multiline, "b\naa", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"aa\Z", RegexOptions.None, "ac\nb\n", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.None, "ac\nb\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.None, "ac\nb\n", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.None, "b\nac\n", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.None, "b\nac\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.None, "b\nac\n", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.None, "b\nac", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.None, "b\nac", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.None, "b\nac", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.Multiline, "ac\nb\n", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.Multiline, "ac\nb\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.Multiline, "ac\nb\n", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.Multiline, "b\nac\n", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.Multiline, "b\nac\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.Multiline, "b\nac\n", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.Multiline, "b\nac", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.Multiline, "b\nac", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.Multiline, "b\nac", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.None, "ca\nb\n", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.None, "ca\nb\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.None, "ca\nb\n", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.None, "b\nca\n", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.None, "b\nca\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.None, "b\nca\n", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.None, "b\nca", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.None, "b\nca", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.None, "b\nca", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.Multiline, "ca\nb\n", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.Multiline, "ca\nb\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.Multiline, "ca\nb\n", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.Multiline, "b\nca\n", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.Multiline, "b\nca\n", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.Multiline, "b\nca\n", "Fail."),
			new RegexTrial (@"aa\Z", RegexOptions.Multiline, "b\nca", "Fail."),
			new RegexTrial (@"aa\z", RegexOptions.Multiline, "b\nca", "Fail."),
			new RegexTrial (@"aa$", RegexOptions.Multiline, "b\nca", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.None, "ab\nb\n", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.None, "ab\nb\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.None, "ab\nb\n", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.None, "b\nab\n", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"ab\z", RegexOptions.None, "b\nab\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.None, "b\nab\n", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"ab\Z", RegexOptions.None, "b\nab", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"ab\z", RegexOptions.None, "b\nab", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"ab$", RegexOptions.None, "b\nab", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"ab\z", RegexOptions.Multiline, "ab\nb\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.Multiline, "ab\nb\n", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"ab\Z", RegexOptions.Multiline, "b\nab\n", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"ab\z", RegexOptions.Multiline, "b\nab\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.Multiline, "b\nab\n", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"ab\Z", RegexOptions.Multiline, "b\nab", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"ab\z", RegexOptions.Multiline, "b\nab", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"ab$", RegexOptions.Multiline, "b\nab", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"ab\Z", RegexOptions.None, "ac\nb\n", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.None, "ac\nb\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.None, "ac\nb\n", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.None, "b\nac\n", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.None, "b\nac\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.None, "b\nac\n", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.None, "b\nac", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.None, "b\nac", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.None, "b\nac", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.Multiline, "ac\nb\n", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.Multiline, "ac\nb\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.Multiline, "ac\nb\n", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.Multiline, "b\nac\n", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.Multiline, "b\nac\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.Multiline, "b\nac\n", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.Multiline, "b\nac", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.Multiline, "b\nac", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.Multiline, "b\nac", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.None, "ca\nb\n", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.None, "ca\nb\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.None, "ca\nb\n", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.None, "b\nca\n", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.None, "b\nca\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.None, "b\nca\n", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.None, "b\nca", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.None, "b\nca", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.None, "b\nca", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.Multiline, "ca\nb\n", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.Multiline, "ca\nb\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.Multiline, "ca\nb\n", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.Multiline, "b\nca\n", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.Multiline, "b\nca\n", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.Multiline, "b\nca\n", "Fail."),
			new RegexTrial (@"ab\Z", RegexOptions.Multiline, "b\nca", "Fail."),
			new RegexTrial (@"ab\z", RegexOptions.Multiline, "b\nca", "Fail."),
			new RegexTrial (@"ab$", RegexOptions.Multiline, "b\nca", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.None, "abb\nb\n", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.None, "abb\nb\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.None, "abb\nb\n", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.None, "b\nabb\n", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"abb\z", RegexOptions.None, "b\nabb\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.None, "b\nabb\n", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"abb\Z", RegexOptions.None, "b\nabb", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"abb\z", RegexOptions.None, "b\nabb", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"abb$", RegexOptions.None, "b\nabb", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"abb\z", RegexOptions.Multiline, "abb\nb\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.Multiline, "abb\nb\n", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"abb\Z", RegexOptions.Multiline, "b\nabb\n", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"abb\z", RegexOptions.Multiline, "b\nabb\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.Multiline, "b\nabb\n", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"abb\Z", RegexOptions.Multiline, "b\nabb", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"abb\z", RegexOptions.Multiline, "b\nabb", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"abb$", RegexOptions.Multiline, "b\nabb", "Pass. Group[0]=(2,3)"),
			new RegexTrial (@"abb\Z", RegexOptions.None, "ac\nb\n", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.None, "ac\nb\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.None, "ac\nb\n", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.None, "b\nac\n", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.None, "b\nac\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.None, "b\nac\n", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.None, "b\nac", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.None, "b\nac", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.None, "b\nac", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.Multiline, "ac\nb\n", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.Multiline, "ac\nb\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.Multiline, "ac\nb\n", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.Multiline, "b\nac\n", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.Multiline, "b\nac\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.Multiline, "b\nac\n", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.Multiline, "b\nac", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.Multiline, "b\nac", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.Multiline, "b\nac", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.None, "ca\nb\n", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.None, "ca\nb\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.None, "ca\nb\n", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.None, "b\nca\n", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.None, "b\nca\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.None, "b\nca\n", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.None, "b\nca", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.None, "b\nca", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.None, "b\nca", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.Multiline, "ca\nb\n", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.Multiline, "ca\nb\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.Multiline, "ca\nb\n", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.Multiline, "b\nca\n", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.Multiline, "b\nca\n", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.Multiline, "b\nca\n", "Fail."),
			new RegexTrial (@"abb\Z", RegexOptions.Multiline, "b\nca", "Fail."),
			new RegexTrial (@"abb\z", RegexOptions.Multiline, "b\nca", "Fail."),
			new RegexTrial (@"abb$", RegexOptions.Multiline, "b\nca", "Fail."),
			new RegexTrial (@"(^|x)(c)", RegexOptions.None, "ca", "Pass. Group[0]=(0,1) Group[1]=(0,0) Group[2]=(0,1)"),
			new RegexTrial (@"a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "x", "Fail."),
			new RegexTrial (@"round\(((?>[^()]+))\)", RegexOptions.None, "_I(round(xs * sz),1)", "Pass. Group[0]=(3,14) Group[1]=(9,7)"),
			new RegexTrial (@"foo.bart", RegexOptions.None, "foo.bart", "Pass. Group[0]=(0,8)"),
			new RegexTrial (@"^d[x][x][x]", RegexOptions.Multiline, "abcd\ndxxx", "Pass. Group[0]=(5,4)"),
			new RegexTrial (@".X(.+)+X", RegexOptions.None, "bbbbXcXaaaaaaaa", "Pass. Group[0]=(3,4) Group[1]=(5,1)"),
			new RegexTrial (@".X(.+)+XX", RegexOptions.None, "bbbbXcXXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(5,1)"),
			new RegexTrial (@".XX(.+)+X", RegexOptions.None, "bbbbXXcXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(6,1)"),
			new RegexTrial (@".X(.+)+X", RegexOptions.None, "bbbbXXaaaaaaaaa", "Fail."),
			new RegexTrial (@".X(.+)+XX", RegexOptions.None, "bbbbXXXaaaaaaaaa", "Fail."),
			new RegexTrial (@".XX(.+)+X", RegexOptions.None, "bbbbXXXaaaaaaaaa", "Fail."),
			new RegexTrial (@".X(.+)+[X]", RegexOptions.None, "bbbbXcXaaaaaaaa", "Pass. Group[0]=(3,4) Group[1]=(5,1)"),
			new RegexTrial (@".X(.+)+[X][X]", RegexOptions.None, "bbbbXcXXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(5,1)"),
			new RegexTrial (@".XX(.+)+[X]", RegexOptions.None, "bbbbXXcXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(6,1)"),
			new RegexTrial (@".X(.+)+[X]", RegexOptions.None, "bbbbXXaaaaaaaaa", "Fail."),
			new RegexTrial (@".X(.+)+[X][X]", RegexOptions.None, "bbbbXXXaaaaaaaaa", "Fail."),
			new RegexTrial (@".XX(.+)+[X]", RegexOptions.None, "bbbbXXXaaaaaaaaa", "Fail."),
			new RegexTrial (@".[X](.+)+[X]", RegexOptions.None, "bbbbXcXaaaaaaaa", "Pass. Group[0]=(3,4) Group[1]=(5,1)"),
			new RegexTrial (@".[X](.+)+[X][X]", RegexOptions.None, "bbbbXcXXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(5,1)"),
			new RegexTrial (@".[X][X](.+)+[X]", RegexOptions.None, "bbbbXXcXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(6,1)"),
			new RegexTrial (@".[X](.+)+[X]", RegexOptions.None, "bbbbXXaaaaaaaaa", "Fail."),
			new RegexTrial (@".[X](.+)+[X][X]", RegexOptions.None, "bbbbXXXaaaaaaaaa", "Fail."),
			new RegexTrial (@".[X][X](.+)+[X]", RegexOptions.None, "bbbbXXXaaaaaaaaa", "Fail."),
			new RegexTrial (@"tt+$", RegexOptions.None, "xxxtt", "Pass. Group[0]=(3,2)"),
			new RegexTrial (@"([\d-z]+)", RegexOptions.None, "a0-za", "Pass. Group[0]=(1,3) Group[1]=(1,3)"),
			new RegexTrial (@"([\d-\s]+)", RegexOptions.None, "a0- z", "Pass. Group[0]=(1,3) Group[1]=(1,3)"),
			new RegexTrial (@"\GX.*X", RegexOptions.None, "aaaXbX", "Fail."),
			new RegexTrial (@"(\d+\.\d+)", RegexOptions.None, "3.1415926", "Pass. Group[0]=(0,9) Group[1]=(0,9)"),
			new RegexTrial (@"(\ba.{0,10}br)", RegexOptions.None, "have a web browser", "Pass. Group[0]=(5,8) Group[1]=(5,8)"),
			new RegexTrial (@"\.c(pp|xx|c)?$", RegexOptions.IgnoreCase, "Changes", "Fail."),
			new RegexTrial (@"\.c(pp|xx|c)?$", RegexOptions.IgnoreCase, "IO.c", "Pass. Group[0]=(2,2) Group[1]="),
			new RegexTrial (@"(\.c(pp|xx|c)?$)", RegexOptions.IgnoreCase, "IO.c", "Pass. Group[0]=(2,2) Group[1]=(2,2) Group[2]="),
			new RegexTrial (@"^([a-z]:)", RegexOptions.None, "C:/", "Fail."),
			new RegexTrial (@"^\S\s+aa$", RegexOptions.Multiline, "\nx aa", "Pass. Group[0]=(1,4)"),
			new RegexTrial (@"(^|a)b", RegexOptions.None, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"^([ab]*?)(b)?(c)$", RegexOptions.None, "abac", "Pass. Group[0]=(0,4) Group[1]=(0,3) Group[2]= Group[3]=(3,1)"),
			new RegexTrial (@"(\w)?(abc)\1b", RegexOptions.None, "abcab", "Fail."),
			new RegexTrial (@"^(?:.,){2}c", RegexOptions.None, "a,b,c", "Pass. Group[0]=(0,5)"),
			new RegexTrial (@"^(.,){2}c", RegexOptions.None, "a,b,c", "Pass. Group[0]=(0,5) Group[1]=(0,2)(2,2)"),
			new RegexTrial (@"^(?:[^,]*,){2}c", RegexOptions.None, "a,b,c", "Pass. Group[0]=(0,5)"),
			new RegexTrial (@"^([^,]*,){2}c", RegexOptions.None, "a,b,c", "Pass. Group[0]=(0,5) Group[1]=(0,2)(2,2)"),
			new RegexTrial (@"^([^,]*,){3}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]*,){3,}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]*,){0,3}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]{1,3},){3}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]{1,3},){3,}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]{1,3},){0,3}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]{1,},){3}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]{1,},){3,}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]{1,},){0,3}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]{0,3},){3}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]{0,3},){3,}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"^([^,]{0,3},){0,3}d", RegexOptions.None, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)"),
			new RegexTrial (@"(?i)", RegexOptions.None, "", "Pass. Group[0]=(0,0)"),
			new RegexTrial (@"(?!\A)x", RegexOptions.Multiline, "a\nxb\n", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"^(a(b)?)+$", RegexOptions.None, "aba", "Pass. Group[0]=(0,3) Group[1]=(0,2)(2,1) Group[2]=(1,1)"),
			new RegexTrial (@"^(aa(bb)?)+$", RegexOptions.None, "aabbaa", "Pass. Group[0]=(0,6) Group[1]=(0,4)(4,2) Group[2]=(2,2)"),
			new RegexTrial (@"^.{9}abc.*\n", RegexOptions.Multiline, "123\nabcabcabcabc\n", "Pass. Group[0]=(4,13)"),
			new RegexTrial (@"^(a)?a$", RegexOptions.None, "a", "Pass. Group[0]=(0,1) Group[1]="),
			new RegexTrial (@"^(a)?(?(1)a|b)+$", RegexOptions.None, "a", "Fail."),
			new RegexTrial (@"^(a\1?)(a\1?)(a\2?)(a\3?)$", RegexOptions.None, "aaaaaa", "Pass. Group[0]=(0,6) Group[1]=(0,1) Group[2]=(1,2) Group[3]=(3,1) Group[4]=(4,2)"),
			new RegexTrial (@"^(a\1?){4}$", RegexOptions.None, "aaaaaa", "Pass. Group[0]=(0,6) Group[1]=(0,1)(1,2)(3,1)(4,2)"),
			new RegexTrial (@"^(0+)?(?:x(1))?", RegexOptions.None, "x1", "Pass. Group[0]=(0,2) Group[1]= Group[2]=(1,1)"),
			new RegexTrial (@"^([0-9a-fA-F]+)(?:x([0-9a-fA-F]+)?)(?:x([0-9a-fA-F]+))?", RegexOptions.None, "012cxx0190", "Pass. Group[0]=(0,10) Group[1]=(0,4) Group[2]= Group[3]=(6,4)"),
			new RegexTrial (@"^(b+?|a){1,2}c", RegexOptions.None, "bbbac", "Pass. Group[0]=(0,5) Group[1]=(0,3)(3,1)"),
			new RegexTrial (@"^(b+?|a){1,2}c", RegexOptions.None, "bbbbac", "Pass. Group[0]=(0,6) Group[1]=(0,4)(4,1)"),
			new RegexTrial (@"\((\w\. \w+)\)", RegexOptions.None, "cd. (A. Tw)", "Pass. Group[0]=(4,7) Group[1]=(5,5)"),
			new RegexTrial (@"((?:aaaa|bbbb)cccc)?", RegexOptions.None, "aaaacccc", "Pass. Group[0]=(0,8) Group[1]=(0,8)"),
			new RegexTrial (@"((?:aaaa|bbbb)cccc)?", RegexOptions.None, "bbbbcccc", "Pass. Group[0]=(0,8) Group[1]=(0,8)"),
			
			new RegexTrial (@"^(foo)|(bar)$", RegexOptions.None, "foobar", "Pass. Group[0]=(0,3) Group[1]=(0,3) Group[2]="),
			new RegexTrial (@"^(foo)|(bar)$", RegexOptions.RightToLeft, "foobar", "Pass. Group[0]=(3,3) Group[1]= Group[2]=(3,3)"),

			new RegexTrial (@"b", RegexOptions.RightToLeft, "babaaa", "Pass. Group[0]=(2,1)"),
			new RegexTrial (@"bab", RegexOptions.RightToLeft, "babababaa", "Pass. Group[0]=(4,3)"),
			new RegexTrial (@"abb", RegexOptions.RightToLeft , "abb", "Pass. Group[0]=(0,3)"),
												
            new RegexTrial (@"b$", RegexOptions.RightToLeft | RegexOptions.Multiline, "aab\naab", "Pass. Group[0]=(6,1)"),
            new RegexTrial (@"^a", RegexOptions.RightToLeft | RegexOptions.Multiline, "aab\naab", "Pass. Group[0]=(4,1)"),
            new RegexTrial (@"^aaab", RegexOptions.RightToLeft | RegexOptions.Multiline, "aaab\naab", "Pass. Group[0]=(0,4)"),
            new RegexTrial (@"abb{2}", RegexOptions.RightToLeft , "abbb", "Pass. Group[0]=(0,4)"),
            new RegexTrial (@"abb{1,2}", RegexOptions.RightToLeft , "abbb", "Pass. Group[0]=(0,4)"),
            
            new RegexTrial (@"abb{1,2}", RegexOptions.RightToLeft , "abbbbbaaaa", "Pass. Group[0]=(0,4)"),
            new RegexTrial (@"\Ab", RegexOptions.RightToLeft, "bab\naaa", "Pass. Group[0]=(0,1)"),
            new RegexTrial (@"\Abab$", RegexOptions.RightToLeft, "bab", "Pass. Group[0]=(0,3)"),
            new RegexTrial (@"b\Z", RegexOptions.RightToLeft, "bab\naaa", "Fail."),
            new RegexTrial (@"b\Z", RegexOptions.RightToLeft, "babaaab", "Pass. Group[0]=(6,1)"),
            new RegexTrial (@"b\z", RegexOptions.RightToLeft, "babaaa", "Fail."),
            new RegexTrial (@"b\z", RegexOptions.RightToLeft, "babaaab", "Pass. Group[0]=(6,1)"),
			new RegexTrial (@"a\G", RegexOptions.RightToLeft, "babaaa", "Pass. Group[0]=(5,1)"),
			new RegexTrial (@"\Abaaa\G", RegexOptions.RightToLeft, "baaa", "Pass. Group[0]=(0,4)"),
//			new RegexTrial (@"b", RegexOptions.RightToLeft, "babaaa", "Pass. Group[0]=(2,1)"),
//			new RegexTrial (@"b", RegexOptions.RightToLeft, "babaaa", "Pass. Group[0]=(2,1)"),
//			new RegexTrial (@"b", RegexOptions.RightToLeft, "babaaa", "Pass. Group[0]=(2,1)"),
//			new RegexTrial (@"b", RegexOptions.RightToLeft, "babaaa", "Pass. Group[0]=(2,1)"),

			new RegexTrial (@"\bc", RegexOptions.RightToLeft, "aaa c aaa c a", "Pass. Group[0]=(10,1)"),
			new RegexTrial (@"\bc", RegexOptions.RightToLeft, "c aaa c", "Pass. Group[0]=(6,1)"),
			new RegexTrial (@"\bc", RegexOptions.RightToLeft, "aaa ac", "Fail."),
			new RegexTrial (@"\bc", RegexOptions.RightToLeft, "c aaa", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"\bc", RegexOptions.RightToLeft, "aaacaaa", "Fail."),
			new RegexTrial (@"\bc", RegexOptions.RightToLeft, "aaac aaa", "Fail."),
			new RegexTrial (@"\bc", RegexOptions.RightToLeft, "aaa ca caaa", "Pass. Group[0]=(7,1)"),
			
			new RegexTrial (@"\Bc", RegexOptions.RightToLeft, "ac aaa ac", "Pass. Group[0]=(8,1)"),
			new RegexTrial (@"\Bc", RegexOptions.RightToLeft, "aaa c", "Fail."),
			new RegexTrial (@"\Bc", RegexOptions.RightToLeft, "ca aaa", "Fail."),
			new RegexTrial (@"\Bc", RegexOptions.RightToLeft, "aaa c aaa", "Fail."),
			new RegexTrial (@"\Bc", RegexOptions.RightToLeft, " acaca ", "Pass. Group[0]=(4,1)"),
			new RegexTrial (@"\Bc", RegexOptions.RightToLeft, "aaac aaac", "Pass. Group[0]=(8,1)"),
			new RegexTrial (@"\Bc", RegexOptions.RightToLeft, "aaa caaa", "Fail."),
				
			new RegexTrial (@"b(a?)b", RegexOptions.RightToLeft, "aabababbaaababa", "Pass. Group[0]=(11,3) Group[1]=(12,1)"),
			new RegexTrial (@"b{4}", RegexOptions.RightToLeft, "abbbbaabbbbaabbb", "Pass. Group[0]=(7,4)"),
			new RegexTrial (@"b\1aa(.)", RegexOptions.RightToLeft, "bBaaB", "Pass. Group[0]=(0,5) Group[1]=(4,1)"),
			new RegexTrial (@"b(.)aa\1", RegexOptions.RightToLeft, "bBaaB", "Fail."),
			
			new RegexTrial (@"^(a\1?){4}$", RegexOptions.RightToLeft, "aaaaaa", "Pass. Group[0]=(0,6) Group[1]=(5,1)(3,2)(2,1)(0,2)"),
			new RegexTrial (@"^([0-9a-fA-F]+)(?:x([0-9a-fA-F]+)?)(?:x([0-9a-fA-F]+))?", RegexOptions.RightToLeft, "012cxx0190", "Pass. Group[0]=(0,10) Group[1]=(0,4) Group[2]= Group[3]=(6,4)"),
			new RegexTrial (@"^(b+?|a){1,2}c", RegexOptions.RightToLeft, "bbbac", "Pass. Group[0]=(0,5) Group[1]=(3,1)(0,3)"),			
			new RegexTrial (@"\((\w\. \w+)\)", RegexOptions.RightToLeft, "cd. (A. Tw)", "Pass. Group[0]=(4,7) Group[1]=(5,5)"),
			new RegexTrial (@"((?:aaaa|bbbb)cccc)?", RegexOptions.RightToLeft, "aaaacccc", "Pass. Group[0]=(0,8) Group[1]=(0,8)"),
			new RegexTrial (@"((?:aaaa|bbbb)cccc)?", RegexOptions.RightToLeft, "bbbbcccc", "Pass. Group[0]=(0,8) Group[1]=(0,8)"),
			
			new RegexTrial (@"(?<=a)b", RegexOptions.RightToLeft, "ab", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"(?<=a)b", RegexOptions.RightToLeft, "cb", "Fail."),
			new RegexTrial (@"(?<=a)b", RegexOptions.RightToLeft, "b", "Fail."),
			new RegexTrial (@"(?<!c)b", RegexOptions.RightToLeft, "ab", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"(?<!c)b", RegexOptions.RightToLeft, "cb", "Fail."),
			new RegexTrial (@"(?<!c)b", RegexOptions.RightToLeft, "b", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"(?<!c)b", RegexOptions.RightToLeft, "b", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"a(?=d).", RegexOptions.RightToLeft, "adabad", "Pass. Group[0]=(4,2)"),
			new RegexTrial (@"a(?=c|d).", RegexOptions.RightToLeft, "adabad", "Pass. Group[0]=(4,2)"),
			
			new RegexTrial (@"ab*c", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab*bc", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab*bc", RegexOptions.RightToLeft, "abbc", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"ab*bc", RegexOptions.RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@".{1}", RegexOptions.RightToLeft, "abbbbc", "Pass. Group[0]=(5,1)"),
			new RegexTrial (@".{3,4}", RegexOptions.RightToLeft, "abbbbc", "Pass. Group[0]=(2,4)"),
			new RegexTrial (@"ab{0,}bc", RegexOptions.RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab+bc", RegexOptions.RightToLeft, "abbc", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"ab+bc", RegexOptions.RightToLeft, "abc", "Fail."),
			new RegexTrial (@"ab+bc", RegexOptions.RightToLeft, "abq", "Fail."),
			new RegexTrial (@"ab{1,}bc", RegexOptions.RightToLeft, "abq", "Fail."),
			new RegexTrial (@"ab+bc", RegexOptions.RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{1,}bc", RegexOptions.RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{1,3}bc", RegexOptions.RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{3,4}bc", RegexOptions.RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)"),
			new RegexTrial (@"ab{4,5}bc", RegexOptions.RightToLeft, "abbbbc", "Fail."),
			new RegexTrial (@"ab?bc", RegexOptions.RightToLeft, "abbc", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"ab?bc", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab{0,1}bc", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab?bc", RegexOptions.RightToLeft, "abbbbc", "Fail."),
			new RegexTrial (@"ab?c", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"ab{0,1}c", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^abc$", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^abc$", RegexOptions.RightToLeft, "abcc", "Fail."),
			new RegexTrial (@"^abc", RegexOptions.RightToLeft, "abcc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^abc$", RegexOptions.RightToLeft, "aabc", "Fail."),
			new RegexTrial (@"abc$", RegexOptions.RightToLeft, "aabc", "Pass. Group[0]=(1,3)"),
			new RegexTrial (@"abc$", RegexOptions.RightToLeft, "aabcd", "Fail."),
			new RegexTrial (@"^", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,0)"),
			new RegexTrial (@"$", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(3,0)"),
			new RegexTrial (@"a.c", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a.c", RegexOptions.RightToLeft, "axc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a.*c", RegexOptions.RightToLeft, "axyzc", "Pass. Group[0]=(0,5)"),
			new RegexTrial (@"a.*c", RegexOptions.RightToLeft, "axyzd", "Fail."),
			new RegexTrial (@"a[bc]d", RegexOptions.RightToLeft, "abc", "Fail."),
			new RegexTrial (@"a[bc]d", RegexOptions.RightToLeft, "abd", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[b-d]e", RegexOptions.RightToLeft, "abd", "Fail."),
			new RegexTrial (@"a[b-d]e", RegexOptions.RightToLeft, "ace", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[b-d]", RegexOptions.RightToLeft, "aac", "Pass. Group[0]=(1,2)"),
			new RegexTrial (@"a[-b]", RegexOptions.RightToLeft, "a-", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a[b-]", RegexOptions.RightToLeft, "a-", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a[b-a]", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"a[]b", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"a[", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"a]", RegexOptions.RightToLeft, "a]", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a[]]b", RegexOptions.RightToLeft, "a]b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[^bc]d", RegexOptions.RightToLeft, "aed", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[^bc]d", RegexOptions.RightToLeft, "abd", "Fail."),
			new RegexTrial (@"a[^-b]c", RegexOptions.RightToLeft, "adc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[^-b]c", RegexOptions.RightToLeft, "a-c", "Fail."),
			new RegexTrial (@"a[^]b]c", RegexOptions.RightToLeft, "a]c", "Fail."),
			new RegexTrial (@"a[^]b]c", RegexOptions.RightToLeft, "adc", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"\ba\b", RegexOptions.RightToLeft, "a-", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"\ba\b", RegexOptions.RightToLeft, "-a", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"\ba\b", RegexOptions.RightToLeft, "-a-", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"\by\b", RegexOptions.RightToLeft, "xy", "Fail."),
			new RegexTrial (@"\by\b", RegexOptions.RightToLeft, "yz", "Fail."),
			new RegexTrial (@"\by\b", RegexOptions.RightToLeft, "xyz", "Fail."),
			new RegexTrial (@"\Ba\B", RegexOptions.RightToLeft, "a-", "Fail."),
			new RegexTrial (@"\Ba\B", RegexOptions.RightToLeft, "-a", "Fail."),
			new RegexTrial (@"\Ba\B", RegexOptions.RightToLeft, "-a-", "Fail."),
			new RegexTrial (@"\By\b", RegexOptions.RightToLeft, "xy", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"\by\B", RegexOptions.RightToLeft, "yz", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"\By\B", RegexOptions.RightToLeft, "xyz", "Pass. Group[0]=(1,1)"),
			new RegexTrial (@"\w", RegexOptions.RightToLeft, "a", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"\w", RegexOptions.RightToLeft, "-", "Fail."),
			new RegexTrial (@"\W", RegexOptions.RightToLeft, "a", "Fail."),
			new RegexTrial (@"\W", RegexOptions.RightToLeft, "-", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"a\sb", RegexOptions.RightToLeft, "a b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a\sb", RegexOptions.RightToLeft, "a-b", "Fail."),
			new RegexTrial (@"a\Sb", RegexOptions.RightToLeft, "a b", "Fail."),
			new RegexTrial (@"a\Sb", RegexOptions.RightToLeft, "a-b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"\d", RegexOptions.RightToLeft, "1", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"\d", RegexOptions.RightToLeft, "-", "Fail."),
			new RegexTrial (@"\D", RegexOptions.RightToLeft, "1", "Fail."),
			new RegexTrial (@"\D", RegexOptions.RightToLeft, "-", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"[\w]", RegexOptions.RightToLeft, "a", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"[\w]", RegexOptions.RightToLeft, "-", "Fail."),
			new RegexTrial (@"[\W]", RegexOptions.RightToLeft, "a", "Fail."),
			new RegexTrial (@"[\W]", RegexOptions.RightToLeft, "-", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"a[\s]b", RegexOptions.RightToLeft, "a b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a[\s]b", RegexOptions.RightToLeft, "a-b", "Fail."),
			new RegexTrial (@"a[\S]b", RegexOptions.RightToLeft, "a b", "Fail."),
			new RegexTrial (@"a[\S]b", RegexOptions.RightToLeft, "a-b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"[\d]", RegexOptions.RightToLeft, "1", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"[\d]", RegexOptions.RightToLeft, "-", "Fail."),
			new RegexTrial (@"[\D]", RegexOptions.RightToLeft, "1", "Fail."),
			new RegexTrial (@"[\D]", RegexOptions.RightToLeft, "-", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"ab|cd", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"ab|cd", RegexOptions.RightToLeft, "abcd", "Pass. Group[0]=(2,2)"),
			new RegexTrial (@"()ef", RegexOptions.RightToLeft, "def", "Pass. Group[0]=(1,2) Group[1]=(1,0)"),
			new RegexTrial (@"*a", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"(*)b", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"$b", RegexOptions.RightToLeft, "b", "Fail."),
			new RegexTrial (@"a\", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"a\(b", RegexOptions.RightToLeft, "a(b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"a\(*b", RegexOptions.RightToLeft, "ab", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"a\(*b", RegexOptions.RightToLeft, "a((b", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"a\\b", RegexOptions.RightToLeft, "a\\b", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"abc)", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"(abc", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"((a))", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1)"),
			new RegexTrial (@"(a)b(c)", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(2,1)"),
			new RegexTrial (@"a+b+c", RegexOptions.RightToLeft, "aabbabc", "Pass. Group[0]=(4,3)"),
			new RegexTrial (@"a{1,}b{1,}c", RegexOptions.RightToLeft, "aabbabc", "Pass. Group[0]=(4,3)"),
			new RegexTrial (@"a**", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"a.+?c", RegexOptions.RightToLeft, "abcabc", "Pass. Group[0]=(3,3)"),
			new RegexTrial (@"(a+|b)*", RegexOptions.RightToLeft, "ab", "Pass. Group[0]=(0,2) Group[1]=(1,1)(0,1)"),
			new RegexTrial (@"(a+|b){0,}", RegexOptions.RightToLeft, "ab", "Pass. Group[0]=(0,2) Group[1]=(1,1)(0,1)"),
			new RegexTrial (@"(a+|b)+", RegexOptions.RightToLeft, "ab", "Pass. Group[0]=(0,2) Group[1]=(1,1)(0,1)"),
			new RegexTrial (@"(a+|b){1,}", RegexOptions.RightToLeft, "ab", "Pass. Group[0]=(0,2) Group[1]=(1,1)(0,1)"),
			new RegexTrial (@"(a+|b)?", RegexOptions.RightToLeft, "ab", "Pass. Group[0]=(1,1) Group[1]=(1,1)"),
			new RegexTrial (@"(a+|b){0,1}", RegexOptions.RightToLeft, "ab", "Pass. Group[0]=(1,1) Group[1]=(1,1)"),
			new RegexTrial (@")(", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"[^ab]*", RegexOptions.RightToLeft, "cde", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"abc", RegexOptions.RightToLeft, "", "Fail."),
			new RegexTrial (@"a*", RegexOptions.RightToLeft, "", "Pass. Group[0]=(0,0)"),
			new RegexTrial (@"([abc])*d", RegexOptions.RightToLeft, "abbbcd", "Pass. Group[0]=(0,6) Group[1]=(4,1)(3,1)(2,1)(1,1)(0,1)"),
			new RegexTrial (@"([abc])*bcd", RegexOptions.RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(0,1)"),
			new RegexTrial (@"a|b|c|d|e", RegexOptions.RightToLeft, "e", "Pass. Group[0]=(0,1)"),
			new RegexTrial (@"(a|b|c|d|e)f", RegexOptions.RightToLeft, "ef", "Pass. Group[0]=(0,2) Group[1]=(0,1)"),
			new RegexTrial (@"abcd*efg", RegexOptions.RightToLeft, "abcdefg", "Pass. Group[0]=(0,7)"),
			new RegexTrial (@"ab*", RegexOptions.RightToLeft, "xabyabbbz", "Pass. Group[0]=(4,4)"),
			new RegexTrial (@"ab*", RegexOptions.RightToLeft, "xayabbbz", "Pass. Group[0]=(3,4)"),
			new RegexTrial (@"(ab|cd)e", RegexOptions.RightToLeft, "abcde", "Pass. Group[0]=(2,3) Group[1]=(2,2)"),
			new RegexTrial (@"[abhgefdc]ij", RegexOptions.RightToLeft, "hij", "Pass. Group[0]=(0,3)"),
			new RegexTrial (@"^(ab|cd)e", RegexOptions.RightToLeft, "abcde", "Fail."),
			new RegexTrial (@"(abc|)ef", RegexOptions.RightToLeft, "abcdef", "Pass. Group[0]=(4,2) Group[1]=(4,0)"),
			new RegexTrial (@"(a|b)c*d", RegexOptions.RightToLeft, "abcd", "Pass. Group[0]=(1,3) Group[1]=(1,1)"),
			new RegexTrial (@"(ab|ab*)bc", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,1)"),
			new RegexTrial (@"a([bc]*)c*", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3) Group[1]=(1,1)"),
			new RegexTrial (@"a([bc]*)(c*d)", RegexOptions.RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,1) Group[2]=(2,2)"),
			new RegexTrial (@"a([bc]+)(c*d)", RegexOptions.RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,1) Group[2]=(2,2)"),
			new RegexTrial (@"a([bc]*)(c+d)", RegexOptions.RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,1) Group[2]=(2,2)"),
			new RegexTrial (@"a[bcd]*dcdcde", RegexOptions.RightToLeft, "adcdcde", "Pass. Group[0]=(0,7)"),
			new RegexTrial (@"a[bcd]+dcdcde", RegexOptions.RightToLeft, "adcdcde", "Fail."),
			new RegexTrial (@"(ab|a)b*c", RegexOptions.RightToLeft, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,1)"),
			new RegexTrial (@"((a)(b)c)(d)", RegexOptions.RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(0,3) Group[2]=(0,1) Group[3]=(1,1) Group[4]=(3,1)"),
			new RegexTrial (@"[a-zA-Z_][a-zA-Z0-9_]*", RegexOptions.RightToLeft, "alpha", "Pass. Group[0]=(0,5)"),
			new RegexTrial (@"^a(bc+|b[eh])g|.h$", RegexOptions.RightToLeft, "abh", "Pass. Group[0]=(1,2) Group[1]="),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.RightToLeft, "effgz", "Pass. Group[0]=(0,5) Group[1]=(0,5) Group[2]="),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.RightToLeft, "ij", "Pass. Group[0]=(0,2) Group[1]=(0,2) Group[2]=(1,1)"),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.RightToLeft, "effg", "Fail."),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.RightToLeft, "bcdd", "Fail."),
			new RegexTrial (@"(bc+d$|ef*g.|h?i(j|k))", RegexOptions.RightToLeft, "reffgz", "Pass. Group[0]=(1,5) Group[1]=(1,5) Group[2]="),
			new RegexTrial (@"((((((((((a))))))))))", RegexOptions.RightToLeft, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)"),
			new RegexTrial (@"((((((((((a))))))))))\10", RegexOptions.RightToLeft, "aa", "Fail."),
			new RegexTrial (@"\10((((((((((a))))))))))", RegexOptions.RightToLeft, "aa", "Pass. Group[0]=(0,2) Group[1]=(1,1) Group[2]=(1,1) Group[3]=(1,1) Group[4]=(1,1) Group[5]=(1,1) Group[6]=(1,1) Group[7]=(1,1) Group[8]=(1,1) Group[9]=(1,1) Group[10]=(1,1)"),
			new RegexTrial (@"((((((((((a))))))))))!", RegexOptions.RightToLeft, "aa", "Fail."),
			new RegexTrial (@"((((((((((a))))))))))!", RegexOptions.RightToLeft, "a!", "Pass. Group[0]=(0,2) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)"),
			new RegexTrial (@"(((((((((a)))))))))", RegexOptions.RightToLeft, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1)"),
			new RegexTrial (@"multiple words of text", RegexOptions.RightToLeft, "uh-uh", "Fail."),
			new RegexTrial (@"multiple words", RegexOptions.RightToLeft, "multiple words, yeah", "Pass. Group[0]=(0,14)"),
			new RegexTrial (@"(.*)c(.*)", RegexOptions.RightToLeft, "abcde", "Pass. Group[0]=(0,5) Group[1]=(0,2) Group[2]=(3,2)"),
			new RegexTrial (@"\((.*), (.*)\)", RegexOptions.RightToLeft, "(a, b)", "Pass. Group[0]=(0,6) Group[1]=(1,1) Group[2]=(4,1)"),
			new RegexTrial (@"[k]", RegexOptions.RightToLeft, "ab", "Fail."),
			new RegexTrial (@"abcd", RegexOptions.RightToLeft, "abcd", "Pass. Group[0]=(0,4)"),
			new RegexTrial (@"a(bc)d", RegexOptions.RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,2)"),
			new RegexTrial (@"a[-]?c", RegexOptions.RightToLeft, "ac", "Pass. Group[0]=(0,2)"),
			new RegexTrial (@"(abc)\1", RegexOptions.RightToLeft, "abcabc", "Fail."),
			new RegexTrial (@"\1(abc)", RegexOptions.RightToLeft, "abcabc", "Pass. Group[0]=(0,6) Group[1]=(3,3)"),
			new RegexTrial (@"([a-c]*)\1", RegexOptions.RightToLeft, "abcabc", "Fail."),
			new RegexTrial (@"\1([a-c]*)", RegexOptions.RightToLeft, "abcabc", "Pass. Group[0]=(0,6) Group[1]=(3,3)"),
			new RegexTrial (@"\1", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"\2", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"(a)|\1", RegexOptions.RightToLeft, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1)"),
			new RegexTrial (@"(a)|\1", RegexOptions.RightToLeft, "x", "Fail."),
			new RegexTrial (@"(a)|\2", RegexOptions.RightToLeft, "-", "Error."),
			new RegexTrial (@"(([a-c])b*?\2)*", RegexOptions.RightToLeft, "ababbbcbc", "Pass. Group[0]=(9,0) Group[1]= Group[2]="),
			new RegexTrial (@"(([a-c])b*?\2){3}", RegexOptions.RightToLeft, "ababbbcbc", "Fail."),
			new RegexTrial (@"((\3|b)\2(a)x)+", RegexOptions.RightToLeft, "aaxabxbaxbbx", "Fail."),
			new RegexTrial (@"((\3|b)\2(a)x)+", RegexOptions.RightToLeft, "aaaxabaxbaaxbbax", "Fail."),
			new RegexTrial (@"((\3|b)\2(a)){2,}", RegexOptions.RightToLeft, "bbaababbabaaaaabbaaaabba", "Fail."),
		    
			              
                        new RegexTrial (@"\((?>[^()]+|\((?<depth>)|\)(?<-depth>))*(?(depth)(?!))\)", RegexOptions.None, "((a(b))c)", 
                            "Pass. Group[0]=(0,9) Group[1]="),
                        new RegexTrial (@"^\((?>[^()]+|\((?<depth>)|\)(?<-depth>))*(?(depth)(?!))\)$", RegexOptions.None, "((a(b))c)", 
                            "Pass. Group[0]=(0,9) Group[1]="),
                        new RegexTrial (@"^\((?>[^()]+|\((?<depth>)|\)(?<-depth>))*(?(depth)(?!))\)$", RegexOptions.None, "((a(b))c", "Fail."), 

                        new RegexTrial (@"^\((?>[^()]+|\((?<depth>)|\)(?<-depth>))*(?(depth)(?!))\)$", RegexOptions.None, "())", "Fail."), 


                        new RegexTrial (@"(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))", RegexOptions.None, "((a(b))c)", 
                            "Pass. Group[0]=(0,9) Group[1]=(0,9) Group[2]=(0,1)(1,2)(3,2) Group[3]=(5,1)(6,2)(8,1) Group[4]= Group[5]=(4,1)(2,4)(1,7)"),
                        new RegexTrial (@"^(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))$", RegexOptions.None, "((a(b))c)", 
                            "Pass. Group[0]=(0,9) Group[1]=(0,9) Group[2]=(0,1)(1,2)(3,2) Group[3]=(5,1)(6,2)(8,1) Group[4]= Group[5]=(4,1)(2,4)(1,7)"),
                        new RegexTrial (@"(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))", RegexOptions.None, "x(a((b)))b)x", 
                            "Pass. Group[0]=(1,9) Group[1]=(1,9) Group[2]=(1,2)(3,1)(4,2) Group[3]=(6,1)(7,1)(8,2) Group[4]= Group[5]=(5,1)(4,3)(2,6)"),
                        new RegexTrial (@"(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))", RegexOptions.None, "x((a((b)))x", 
                            "Pass. Group[0]=(2,9) Group[1]=(2,9) Group[2]=(2,2)(4,1)(5,2) Group[3]=(7,1)(8,1)(9,2) Group[4]= Group[5]=(6,1)(5,3)(3,6)"),
                        new RegexTrial (@"^(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))$", RegexOptions.None, "((a(b))c","Fail."),
                        new RegexTrial (@"^(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))$", RegexOptions.None, "((a(b))c))","Fail."),
                        new RegexTrial (@"^(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))$", RegexOptions.None, ")(","Fail."),
                        new RegexTrial (@"^(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))$", RegexOptions.None, "((a((b))c)","Fail."),

    			new RegexTrial (@"b", RegexOptions.RightToLeft, "babaaa", "Pass. Group[0]=(2,1)")
               };
	}
}
