// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <cc++/slog.h>
#include <cc++/process.h>
#include "server.h"

using namespace server;
using namespace ost;
using namespace std;

#ifdef	WIN32

static bool test = true;

static SERVICE_STATUS_HANDLE hStatus = 0;
static SERVICE_STATUS sStatus;
static HANDLE hDown;

static BOOL WINAPI stop(DWORD code)
{
        if(code == CTRL_LOGOFF_EVENT)
                return TRUE;

	BayonneService::stop();
	BayonneDriver::stop();
	PersistProperty::save();

	if(code)
		Bayonne::errlog("failed", "server exiting; reason=%d", code);
	else
		Bayonne::errlog("notice", "server exiting; normal termination");

	return TRUE;
}

#else
static int mainpid;

static RETSIGTYPE stop(int signo)
{
	if(signo > 0 && getpid() != mainpid)
	{
		kill(mainpid, signo);
		return;
	}

        signal(SIGINT, SIG_IGN);
        signal(SIGABRT, SIG_IGN);
        signal(SIGTERM, SIG_IGN);
        signal(SIGQUIT, SIG_IGN);

	BayonneService::stop();
	BayonneDriver::stop();
	PersistProperty::save();

        if(signo == -2)
                Bayonne::errlog("failed", "server exiting; no timeslots allocated");
        else if(signo)
                Bayonne::errlog("failed", "server exiting; reason=%d", signo);
        else
                Bayonne::errlog("notice", "server exiting; normal termination");

#ifdef	HAVE_LIBEXEC
	BayonneSysexec::cleanup();
#endif

	Thread::sleep(100);

	::exit(signo);
}
#endif

static void version(void)
{
        cout << VERSION << endl;
        exit(0);
}

static void loadStack(const char *id)
{
	BayonneDriver *d = BayonneDriver::loadDriver(keypaths.getLast("drivers"), id);

	if(!d)
	{
		cerr << "bayonne cannot load " << id << " stack" << endl;
		stop(-1);
	}
	d->start();
}	

static void loading(void)
{
	char buffer[256];
	char lang[32];
	char ** keys;
	const char *cp;
	char *sp;
	const char *dname = keyserver.getLast("driver");
	unsigned count = keyoptions.getCount("modules");
	BayonneDriver *driver;
	BayonneTranslator *translator = NULL;
	static char vlib[64];

	if(!Process::getEnv("SERVER_SYSEXEC"))
		Process::setEnv("SERVER_SYSEXEC", keypaths.getLast("scripts"), true);

	Process::setEnv("SERVER_PREFIX", keypaths.getLast("datafiles"), true);
        Process::setEnv("SERVER_PROMPTS", keypaths.getLast("prompts"), true);
        Process::setEnv("SERVER_SCRIPTS", keypaths.getLast("scripts"), true);
        Process::setEnv("SERVER_LIBEXEC", keypaths.getLast("libexec"), true);     
        Process::setEnv("SERVER_SOFTWARE", "bayonne", true);
        Process::setEnv("SERVER_VERSION", VERSION, true);
        Process::setEnv("SERVER_TOKEN", " ", true);

#ifdef	HAVE_LIBEXEC
	size_t bs = 0;
	int pri = 0;

	cp = keyengine.getLast("gateways");
	if(cp)
		pri = atoi(cp);

	cp = keyengine.getLast("buffers");
	if(cp)
		bs = atoi(cp) * 1024;

	getcwd(buffer, sizeof(buffer));

	BayonneSysexec::allocate(keypaths.getLast("libexec"), bs, pri,
		keypaths.getLast("modexec"));
#endif

	PersistProperty::load();

        cp = keyserver.getLast("language");
        if(cp && (strlen(cp) == 2 || cp[2] == '_'))  
		BayonneTranslator::loadTranslator(keypaths.getLast("phrases"), cp);  

        cp = keyserver.getLast("voice");
        if(cp && strchr(cp, '/'))
        {
                setString(lang, sizeof(lang), cp);
                sp = strchr(lang, '/');
                if(sp)
                        *sp = 0;
                if(strlen(lang) == 2 || sp[2] == '_')
                        translator = BayonneTranslator::loadTranslator(keypaths.getLast("phrases"), lang);

	}

	if(cp && translator)
	{
		snprintf(buffer, sizeof(buffer), "%s/%s", keypaths.getLast("prompts"), cp);
		if(isDir(buffer))
		{
			slog.debug("using %s as default voice library", cp);
			Bayonne::init_translator = translator;
			Bayonne::init_voicelib = cp;
		}
		else if(cp[2] == '_')
		{
			vlib[0] = cp[0];
			vlib[1] = cp[1];
			cp = strchr(cp, '/');
			if(!cp)
				cp = "/default";
			snprintf(vlib + 2, sizeof(vlib) - 2, "%s", cp);
			snprintf(buffer, sizeof(buffer), "%s/%s",
				keypaths.getLast("prompts"), vlib);
			if(isDir(buffer))
			{
                        	slog.debug("using %s as default voice library", cp);
                        	Bayonne::init_translator = translator;
                        	Bayonne::init_voicelib = vlib; 
			}
		}
        } 

	driver = BayonneDriver::loadDriver(keypaths.getLast("drivers"), dname);
	if(!driver)
		stop(-1);

	if(!count)
		goto start;

	keys = (char **)keyoptions.getList("modules");

	while(*keys)
		Bayonne::loadPlugin(keypaths.getLast("modules"), *keys++);

start:
	if(!driver)
		stop(-1);

	driver->start();
	count = Bayonne::getTimeslotsUsed();
	if(!count)
		stop(-2);

	Bayonne::start_driver = driver;

	cp = keyoptions.getLast("sip.stack");
	if(cp && atoi(cp) > 0)
		loadStack("sip");

	cp = keyoptions.getLast("h323.stack");
	if(cp && atoi(cp) > 0)
		loadStack("h323");

	count = Bayonne::getTimeslotsUsed();

#ifdef	HAVE_LIBEXEC
	BayonneSysexec::startup();
#endif

	Bayonne::reload();
	if(!Bayonne::compile_count)
	{
		Bayonne::errlog("critical", "no applications defined");
		stop(-1);
	}
	else
		Bayonne::errlog("notice", "%d applications compiled", Bayonne::compile_count); 
	BayonneService::start();
	Bayonne::errlog("notice", "%s driver started; %d timeslot(s) used", dname, count);
#ifndef	WIN32
        Process::setPosixSignal(SIGPIPE, SIG_IGN);
        Process::setPosixSignal(SIGINT, &stop);
        Process::setPosixSignal(SIGHUP, &stop);
        Process::setPosixSignal(SIGTERM, &stop);
        Process::setPosixSignal(SIGABRT, &stop);
#endif
}

static void logging(void)
{
	slog.open("bayonne", Slog::classDaemon);
	const char *level = keyserver.getLast("logging");

	if(!stricmp(level, "notice"))
		slog.level(Slog::levelNotice);
	else if(!stricmp(level, "info"))
		slog.level(Slog::levelInfo);
	else if(!stricmp(level, "error"))
		slog.level(Slog::levelError);
	else if(!stricmp(level, "debug"))
		slog.level(Slog::levelDebug);
}

static void starting(char **argv, bool trace)
{
	loadConfig();
	if(trace)
		keyserver.setValue("logging", "debug");

	parseConfig(argv);
        Process::setScheduler(keyengine.getLast("scheduler"));
	Process::setPriority(atoi(keyengine.getLast("priority")));
#ifndef	WIN32
	if(!getuid())
	{
		if(!Process::setUser(keyserver.getLast("user")))
		{
			Bayonne::errlog("fatal", "%s: cannot set user",
				keyserver.getLast("user"));
			stop(-1);
		}
	}
	if(!trace)
		Process::detach();

	mainpid = getpid();
#endif
	
	logging();
        Bayonne::errlog("notice", "starting %s on %s %s; driver=%s, timeslots=%s",           
		VERSION,
                keyserver.getLast("cpu"), keyserver.getLast("platform"),
		keyserver.getLast("driver"), keyserver.getLast("timeslots"));

	loading();
	Runtime::process();
}

#ifdef	WIN32

static void control(DWORD request)
{
	switch(request)
	{
	case SERVICE_CONTROL_SHUTDOWN:
	case SERVICE_CONTROL_STOP:
		BayonneService::stop();
		BayonneDriver::stop();
		Bayonne::errlog("notice", "server exiting; service shutdown");
		sStatus.dwWin32ExitCode = 0;
		sStatus.dwCurrentState = SERVICE_STOPPED;
		SetServiceStatus(hStatus, &sStatus);
		SetEvent(hDown);
		return;
	}
	SetServiceStatus(hStatus, &sStatus);
}

static VOID service(DWORD argc, LPTSTR *argv)
{
	hStatus = RegisterServiceCtrlHandler("Bayonne", (LPHANDLER_FUNCTION)control);
	hDown = CreateEvent(0, TRUE, FALSE, 0);

	if(!hStatus || !hDown)
		return;

	sStatus.dwServiceType = SERVICE_WIN32;
	sStatus.dwCurrentState = SERVICE_START_PENDING;
	sStatus.dwControlsAccepted = 0;
	sStatus.dwWin32ExitCode = 0;
	sStatus.dwServiceSpecificExitCode = 0;
	sStatus.dwCheckPoint = 0;
	sStatus.dwWaitHint = 500;

	SetServiceStatus(hStatus, &sStatus);

	starting(argv, false);

	sStatus.dwCurrentState = SERVICE_RUNNING;
	sStatus.dwControlsAccepted = SERVICE_ACCEPT_STOP | SERVICE_ACCEPT_SHUTDOWN;
	sStatus.dwWin32ExitCode = 0;
	sStatus.dwWaitHint = 0;
	SetServiceStatus(hStatus, &sStatus);

	WaitForSingleObject(hDown, INFINITE);
	return;
};

static SERVICE_TABLE_ENTRY services[] =
{
	{"Bayonne", (LPSERVICE_MAIN_FUNCTION)service},
	{NULL, NULL}
};

#endif

static void testing(char **argv)
{
	testConfig(*argv);		
	if(!parseConfig(argv))
		Script::exec_extensions = "";

        Process::setScheduler(keyengine.getLast("scheduler"));
        Process::setPriority(atoi(keyengine.getLast("priority"))); 

	logging();

	Bayonne::errlog("notice", "testing %s on %s %s; driver=%s, timeslots=%s",
		VERSION, 
		keyserver.getLast("cpu"), keyserver.getLast("platform"),
		keyserver.getLast("driver"), keyserver.getLast("timeslots"));

	loading();
	Runtime::process();
}


static void banner(void)
{
	cout << "SERVER VERSION " << VERSION << "; ";
	cout << keyserver.getLast("node") << " ";
	cout << keyserver.getLast("platform") << " ";
	cout << keyserver.getLast("cpu") << " ";
	cout << endl;
}

int main(int argc, char **argv)
{
	char *cp;

        if(argc < 2)
        {
                cerr << "use: bayonne --option [args...]" << endl;
                exit(-1);
        }
#ifdef	HAVE_LIBEXEC
	BayonneSysexec::setArgv0(argv[0]);
#endif
        cp = argv[1];
        if(!strncmp(cp, "--", 2))
                ++cp;

#ifdef	WIN32
	SetConsoleTitle("Bayonne");
        SetConsoleCtrlHandler((PHANDLER_ROUTINE)stop, TRUE);
#else
	mainpid = getpid();
	Process::setPosixSignal(SIGPIPE, SIG_IGN);
        Process::setPosixSignal(SIGINT, &stop);
        Process::setPosixSignal(SIGHUP, &stop);
        Process::setPosixSignal(SIGTERM, &stop);
        Process::setPosixSignal(SIGABRT, &stop);

	if(!getuid())
		Bayonne::provision_system = true;
	else if(Process::getEnv("HOME"))
		Bayonne::provision_user = true;

#endif

	if(!stricmp(cp, "-version"))
		version();
	else if(!stricmp(cp, "-debug"))
	{
		Mutex::setDebug(true);
		testing(argv);
	}
	else if(!stricmp(cp, "-test"))
		testing(argv);
	else if(!stricmp(cp, "-trace") || !stricmp(cp, "-load"))
		starting(argv, true);
	else if(!stricmp(cp, "-start"))
	{
#ifdef	WIN32
		if(StartServiceCtrlDispatcher(services) == FALSE)
		{
			Bayonne::errlog("fatel", "service failed to start");
			stop(-1);
		}		
#else
		starting(argv, false);
		return 0;
#endif
	}
	else if(!stricmp(cp, "-dump-install"))
	{
		loadConfig();
		banner();
		dumpConfig(keypaths);
		exit(0);
	}
	else if(!stricmp(cp, "-dump-testing"))
	{
		testConfig(*argv);
		banner();
		dumpConfig(keypaths);
		exit(0);
	}
#ifdef	WIN32
	else if(!stricmp(cp, "-dump-registry"))
	{
		// we assume these objects are registry initialized
		// and dump them without further processing.
		cout << "[paths]" << endl;
		dumpConfig(keypaths);
		cout << "[server]" << endl;
		dumpConfig(keyserver);
                cout << "[engine]" << endl;
                dumpConfig(keyengine);
		exit(0);
	}
#endif

	cerr << "bayonne: " << *(argv + 1) << ": unknown option" << endl;
	exit(-1);
	return -1;
}
