/*
 *  Tables of Text Functions for AutoGen
 *
 *  copyright 1992-2001 Bruce Korb
 *
 *  AutoGen is free software.
 *  
 *  You may redistribute it and/or modify it under the terms of the
 *  GNU General Public License, as published by the Free Software
 *  Foundation; either version 2, or (at your option) any later version.
 *  
 *  AutoGen is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with AutoGen.  See the file "COPYING".  If not,
 *  write to:  The Free Software Foundation, Inc.,
 *             59 Temple Place - Suite 330,
 *             Boston,  MA  02111-1307, USA.
 *
 *  The 20 AutoGen functions are tagged with special attributes:
 *
 *  A - is invoked via an alias
 *
 *  U - is unnamed.  May *not* be explicitly invoked.  May not have
 *      have an alias.  These three are used by AutoGen for its purposes.
 *
 *  L - has a special load procedure defined
 *
 *  S - situational.  May be explicitly invoked in certain situations.
 *      For example, "ELSE" may only be specified within an "IF" block.
 *      Their load procedures are enabled by the block macro (e.g. IF),
 *      and disabled by the block ending macro (e.g. ENDIF).
 *      While disabled, the load procedure is the "Bogus" method.
 *
 *      If a function is neither has a special load procedure nor is
 *      situational, then the "Unknown" load method is applied.
 *
 *  H - has a handler procedure defined.  Only these procedures should
 *      be encountered by the dispatcher during processing.
 *
 *  U L H - BOGUS       Out-of-context or unknown functions are bogus.
 *    L H - CASE        Select one of several template blocks
 *  A L   - COMMENT     A block of comment to be ignored
 *    L H - DEFINE      Define a user AutoGen macro
 *    S   - ELIF        Alternate Conditional Template Block
 *    S   - ELSE        Alternate Template Block
 *    S   - ENDDEF      Ends a macro definition.
 *    S   - ENDFOR      Terminates the @code{FOR} function template block
 *    S   - ENDIF       Terminate the @code{IF} Template Block
 *    S   - ENDWHILE    Terminate the @code{WHILE} Template Block
 *    S   - ESAC        Terminate the @code{CASE} Template Block
 *  A L H - EXPR        Evaluate and emit an Expression
 *    L H - FOR         Emit a template block multiple times
 *    L H - IF          Conditionally Emit a Template Block
 *    L H - INCLUDE     Read in and emit a template block
 *      H - INVOKE      Invoke a User Defined Macro
 *  A S   - SELECT      Selection block for CASE function
 *  U   H - TEXT        A block of text to be emitted.
 *  U L H - UNKNOWN     Either a user macro or a value name.
 *    L H - WHILE       Conditionally loop over a Template Block
 */

#ifndef AUTOGEN_FUNCTIONS_H
#define AUTOGEN_FUNCTIONS_H

#define FUNC_CT    20

extern const char*  apzFuncNames[ FUNC_CT ];

/*
 *  Enumerate all the function types, whether they have
 *  implementation functions or not.
 */
typedef enum {
    FTYP_BOGUS,        /* Out-of-context or unknown functions are bogus. */
    FTYP_CASE,         /* Select one of several template blocks */
    FTYP_COMMENT,      /* A block of comment to be ignored */
    FTYP_DEFINE,       /* Define a user AutoGen macro */
    FTYP_ELIF,         /* Alternate Conditional Template Block */
    FTYP_ELSE,         /* Alternate Template Block */
    FTYP_ENDDEF,       /* Ends a macro definition. */
    FTYP_ENDFOR,       /* Terminates the @code{FOR} function template block */
    FTYP_ENDIF,        /* Terminate the @code{IF} Template Block */
    FTYP_ENDWHILE,     /* Terminate the @code{WHILE} Template Block */
    FTYP_ESAC,         /* Terminate the @code{CASE} Template Block */
    FTYP_EXPR,         /* Evaluate and emit an Expression */
    FTYP_FOR,          /* Emit a template block multiple times */
    FTYP_IF,           /* Conditionally Emit a Template Block */
    FTYP_INCLUDE,      /* Read in and emit a template block */
    FTYP_INVOKE,       /* Invoke a User Defined Macro */
    FTYP_SELECT,       /* Selection block for CASE function */
    FTYP_TEXT,         /* A block of text to be emitted. */
    FTYP_UNKNOWN,      /* Either a user macro or a value name. */
    FTYP_WHILE,        /* Conditionally loop over a Template Block */

    FTYP_SELECT_COMPARE_FULL          = 0x8000,  /* *==* */
    FTYP_SELECT_COMPARE_SKP_START     = 0x8001,  /* *==  */
    FTYP_SELECT_COMPARE_SKP_END       = 0x8002,  /*  ==* */
    FTYP_SELECT_COMPARE               = 0x8003,  /*  ==  */

    FTYP_SELECT_EQUIVALENT_FULL       = 0x8004,  /* *=*  */
    FTYP_SELECT_EQUIVALENT_SKP_START  = 0x8005,  /* *=   */
    FTYP_SELECT_EQUIVALENT_SKP_END    = 0x8006,  /*  =*  */
    FTYP_SELECT_EQUIVALENT            = 0x8007,  /*  =   */

    FTYP_SELECT_MATCH_FULL            = 0x8008,  /* *~~* */
    FTYP_SELECT_MATCH_SKP_START       = 0x8009,  /* *~~  */
    FTYP_SELECT_MATCH_SKP_END         = 0x800A,  /*  ~~* */
    FTYP_SELECT_MATCH                 = 0x800B,  /*  ~~  */

    FTYP_SELECT_EQV_MATCH_FULL        = 0x800C,  /* *~*  */
    FTYP_SELECT_EQV_MATCH_SKP_START   = 0x800D,  /* *~   */
    FTYP_SELECT_EQV_MATCH_SKP_END     = 0x800E,  /*  ~*  */
    FTYP_SELECT_EQV_MATCH             = 0x800F,  /*  ~   */

    FTYP_SELECT_MATCH_ANYTHING        = 0x801C   /*  *   */
} teFuncType;

/*
 *  The function processing procedures.
 */
tHdlrProc mFunc_Bogus;
tHdlrProc mFunc_Case;
tHdlrProc mFunc_Define;
tHdlrProc mFunc_Expr;
tHdlrProc mFunc_For;
tHdlrProc mFunc_If;
tHdlrProc mFunc_Include;
tHdlrProc mFunc_Invoke;
tHdlrProc mFunc_Text;
tHdlrProc mFunc_Unknown;
tHdlrProc mFunc_While;

/*
 *  Template Loading Functions
 */
tLoadProc mLoad_Ending; /* generic block loading ending */
tLoadProc mLoad_Bogus;
tLoadProc mLoad_Case;
tLoadProc mLoad_Comment;
tLoadProc mLoad_Define;
tLoadProc mLoad_Expr;
tLoadProc mLoad_For;
tLoadProc mLoad_If;
tLoadProc mLoad_Unknown;
tLoadProc mLoad_While;

extern tpLoadProc   apLoadProc[ 20 ];
extern tpLoadProc*  papLoadProc;
#endif /* AUTOGEN_FUNCTIONS_H */

#ifdef LOAD_FUNCTIONS  /* tpParse.c use only * * * * * * * * * * * * * * * *
 *
 *  Pointers for the load processing (template scanning phase)
 */
tpLoadProc apLoadProc[ 20 ] = {
    /* BOGUS     */ &mLoad_Bogus,
    /* CASE      */ &mLoad_Case,
    /* COMMENT   */ &mLoad_Comment,
    /* DEFINE    */ &mLoad_Define,
    /* ELIF      */ &mLoad_Bogus   /*dynamic*/,
    /* ELSE      */ &mLoad_Bogus   /*dynamic*/,
    /* ENDDEF    */ &mLoad_Bogus   /*dynamic*/,
    /* ENDFOR    */ &mLoad_Bogus   /*dynamic*/,
    /* ENDIF     */ &mLoad_Bogus   /*dynamic*/,
    /* ENDWHILE  */ &mLoad_Bogus   /*dynamic*/,
    /* ESAC      */ &mLoad_Bogus   /*dynamic*/,
    /* EXPR      */ &mLoad_Expr,
    /* FOR       */ &mLoad_For,
    /* IF        */ &mLoad_If,
    /* INCLUDE   */ &mLoad_Expr,
    /* INVOKE    */ &mLoad_Unknown /*default*/,
    /* SELECT    */ &mLoad_Bogus   /*dynamic*/,
    /* TEXT      */ &mLoad_Unknown /*default*/,
    /* UNKNOWN   */ &mLoad_Unknown,
    /* WHILE     */ &mLoad_While
};
tpLoadProc* papLoadProc = apLoadProc;

/*
 *  name-to-function type mapping table.
 *  This table must be sorted alphabetically by the content
 *  of the naming string.
 */
typedef struct name_type tNameType;
struct name_type {
    size_t      cmpLen;  /* compare length (sans NUL) */
    tCC*        pName;   /* ptr to name */
    teFuncType  fType;   /* function type enum */
    int         unused;
};

/*
 *  Define all the strings that are used to determine
 *  the function enumeration number.  Sort by string content.
 */
tSCC zFn_Case[]      = "CASE";
tSCC zFn_Comment_0[] = "#";
tSCC zFn_Define[]    = "DEFINE";
tSCC zFn_Elif[]      = "ELIF";
tSCC zFn_Else[]      = "ELSE";
tSCC zFn_Enddef[]    = "ENDDEF";
tSCC zFn_Endfor[]    = "ENDFOR";
tSCC zFn_Endif[]     = "ENDIF";
tSCC zFn_Endwhile[]  = "ENDWHILE";
tSCC zFn_Esac[]      = "ESAC";
tSCC zFn_Expr_0[]    = "-";
tSCC zFn_Expr_1[]    = "?";
tSCC zFn_Expr_2[]    = "%";
tSCC zFn_Expr_3[]    = ";";
tSCC zFn_Expr_4[]    = "(";
tSCC zFn_Expr_5[]    = "`";
tSCC zFn_Expr_6[]    = "\"";
tSCC zFn_Expr_7[]    = "'";
tSCC zFn_For[]       = "FOR";
tSCC zFn_If[]        = "IF";
tSCC zFn_Include[]   = "INCLUDE";
tSCC zFn_Invoke[]    = "INVOKE";
tSCC zFn_Select_0[]  = "~";
tSCC zFn_Select_1[]  = "=";
tSCC zFn_Select_2[]  = "*";
tSCC zFn_While[]     = "WHILE";


/*
 *  The number of names by which the macros go.
 *  Some have multiple names (e.g. selection clauses).
 */
#define FUNC_ALIAS_LOW_INDEX    0
#define FUNC_ALIAS_HIGH_INDEX   11
#define FUNC_NAMES_LOW_INDEX    12
#define FUNC_NAMES_HIGH_INDEX   25
#define FUNCTION_NAME_CT        26

/*
 *  And now, the table sorted by string content
 */
tNameType nameTypeTable[ FUNCTION_NAME_CT ] = {
    {  1, zFn_Expr_6,      FTYP_EXPR },
    {  1, zFn_Comment_0,   FTYP_COMMENT },
    {  1, zFn_Expr_2,      FTYP_EXPR },
    {  1, zFn_Expr_7,      FTYP_EXPR },
    {  1, zFn_Expr_4,      FTYP_EXPR },
    {  1, zFn_Select_2,    FTYP_SELECT },
    {  1, zFn_Expr_0,      FTYP_EXPR },
    {  1, zFn_Expr_3,      FTYP_EXPR },
    {  1, zFn_Select_1,    FTYP_SELECT },
    {  1, zFn_Expr_1,      FTYP_EXPR },
    {  1, zFn_Expr_5,      FTYP_EXPR },
    {  1, zFn_Select_0,    FTYP_SELECT },
    {  4, zFn_Case,        FTYP_CASE },
    {  6, zFn_Define,      FTYP_DEFINE },
    {  4, zFn_Elif,        FTYP_ELIF },
    {  4, zFn_Else,        FTYP_ELSE },
    {  6, zFn_Enddef,      FTYP_ENDDEF },
    {  6, zFn_Endfor,      FTYP_ENDFOR },
    {  5, zFn_Endif,       FTYP_ENDIF },
    {  8, zFn_Endwhile,    FTYP_ENDWHILE },
    {  4, zFn_Esac,        FTYP_ESAC },
    {  3, zFn_For,         FTYP_FOR },
    {  2, zFn_If,          FTYP_IF },
    {  7, zFn_Include,     FTYP_INCLUDE },
    {  6, zFn_Invoke,      FTYP_INVOKE },
    {  5, zFn_While,       FTYP_WHILE } };

const char*  apzFuncNames[ FUNC_CT ] = {
    "Bogus",    "CASE",     "COMMENT",  "DEFINE",   "ELIF",     "ELSE",
    "ENDDEF",   "ENDFOR",   "ENDIF",    "ENDWHILE", "ESAC",     "EXPR",
    "FOR",      "IF",       "INCLUDE",  "INVOKE",   "SELECT",   "Text",
    "Unknown",  "WHILE"
};
#endif /* LOAD_FUNCTIONS */

#define FUNCTION_CKSUM 7851

#ifdef DEFINE_LOAD_FUNCTIONS
/*
 *  Source in each file that contains load functions
 *  This is used by the binary template library builder.
 */
#include "funcCase.c"
#include "funcDef.c"
#include "funcEval.c"
#include "funcFor.c"
#include "funcIf.c"
#include "functions.c"
#endif /* DEFINE_LOAD_FUNCTIONS */

#ifdef HANDLE_FUNCTIONS /* tpProcess.c use only * * * * * * * * * * * * * *

 *  Template Processing Function Table
 *
 *  Pointers to the procedure to call when the function code
 *  is encountered.
 */
static tpHdlrProc  apHdlrProc[ FUNC_CT ] = {
    /* BOGUS     */ &mFunc_Bogus,
    /* CASE      */ &mFunc_Case,
    /* COMMENT   */ &mFunc_Bogus,
    /* DEFINE    */ &mFunc_Define,
    /* ELIF      */ &mFunc_Bogus,
    /* ELSE      */ &mFunc_Bogus,
    /* ENDDEF    */ &mFunc_Bogus,
    /* ENDFOR    */ &mFunc_Bogus,
    /* ENDIF     */ &mFunc_Bogus,
    /* ENDWHILE  */ &mFunc_Bogus,
    /* ESAC      */ &mFunc_Bogus,
    /* EXPR      */ &mFunc_Expr,
    /* FOR       */ &mFunc_For,
    /* IF        */ &mFunc_If,
    /* INCLUDE   */ &mFunc_Include,
    /* INVOKE    */ &mFunc_Invoke,
    /* SELECT    */ &mFunc_Bogus,
    /* TEXT      */ &mFunc_Text,
    /* UNKNOWN   */ &mFunc_Unknown,
    /* WHILE     */ &mFunc_While
};

#endif /* HANDLE_FUNCTIONS */
