/*	ctgamma.c
 *
 *	Complex tgamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * #include <complex.h>
 * double complex x, y, cgamma();
 *
 * y = ctgamma( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns complex-valued gamma function of the complex argument.
 *
 * Arguments |Re(x)| < 18 are increased by recurrence.
 * Large arguments are handled by Stirling's formula. Large negative
 * arguments are made positive using the reflection formula.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -20,20      500000      2.0e-14     2.7e-15
 *    IEEE     -100,100     100000      1.4e-13     1.5e-14
 *
 * Error for arguments outside the test range will be larger
 * owing to error amplification by the exponential function.
 *
 */
/*
Cephes Math Library Release 2.7:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004


#include "complex.h"
#include <float.h>
#include <math.h>
#include <stdio.h>
/* Stirling's formula for the gamma function */
#define NSTIR 7
static double STIR[NSTIR] = {
#if 0
    7.20489541602001055909E-5,
    8.39498720672087279993E-4,
    -5.17179090826059219337E-5,
#endif
    -5.92166437353693882865E-4,
    6.97281375836585777429E-5,
    7.84039221720066627474E-4,
    -2.29472093621399176955E-4,
    -2.68132716049382716049E-3,
    3.47222222222222222222E-3,
    8.33333333333333333333E-2
};
#define MAXSTIR 143.01608

/* square root of 2*pi	*/
#define	SQTM_PI	2.50662827463100050242E0

/* Gamma function computed by Stirling's formula.  */

static double complex cstirf(double complex);

static double complex
cstirf(double complex x)
{
    double complex y, w;
    int     i;

    w = 1.0 / x;

    y = STIR[0];
    for (i = 1; i < NSTIR; i++)
    {
	y = y * w + STIR[i];
    }

    w = 1.0 + w * y;
#if 1
    y = cpow(x, x - 0.5) * cexp(-x);
#else
    y = (x - 0.5) * clog(x) - x;
    y = cexp(y);
#endif
    y = SQTM_PI * y * w;
    return (y);
}


double complex
ctgamma(x)
double complex x;
{
    double  p, q;
    double complex c, u;

    if (fabs(creal(x)) > 18.0)
    {
	if (creal(x) < 0.0)
	{
	    q = creal(x);
	    p = floor(q);
	    if ((p == q) && cimag(u) == 0.0)
	    {
		// mtherr( "cgamma", OVERFLOW );
		return (DBL_MAX + I * DBL_MAX);
	    }
	    /* Compute complex sin(pi x)  */
	    c = csin(M_PI * x);
	    /* Reflection formula.  */
	    c = M_PI / (c * ctgamma(1.0 - x));
	}
	else
	{
	    c = cstirf(x);
	}
	return (c);
    }

    c = 1.0;
    p = 0.0;
    u = x;
    while (creal(u) < 18.0)
    {
	if ((fabs(creal(u)) < 1.e-9) && (fabs(cimag(u)) < 1.e-9))
	    goto small;
	c *= u;
	p += 1.0;
	u = x + p;
    }
    u = cstirf(u);
    return (u / c);


  small:
    if ((x) == 0.0)
    {
	//mtherr( "cgamma", SING );
	return (DBL_MAX + DBL_MAX * I);
    }
    else
	return (1.0 / (((1.0 + 0.5772156649015329 * u) * u) * c));
}
