/*	csqrt.c
 *
 *	Complex square root
 *
 *
 *
 * SYNOPSIS:
 *
 * double complex csqrt();
 * double complex z, w;
 *
 * w = csqrt (z);
 *
 *
 *
 * DESCRIPTION:
 *
 *
 * If z = x + iy,  r = |z|, then
 *
 *                       1/2
 * Re w  =  [ (r + x)/2 ]   ,
 *
 *                       1/2
 * Im w  =  [ (r - x)/2 ]   .
 *
 * Cancellation error in r-x or r+x is avoided by using the
 * identity  2 Re w Im w  =  y.
 *
 * Note that -w is also a square root of z.  The root chosen
 * is always in the right half plane and Im w has the same sign as y.
 *
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC       -10,+10     25000       3.2e-17     9.6e-18
 *    IEEE      -10,+10   1,000,000     2.9e-16     6.1e-17
 *
 */
/*
Cephes Math Library Release 2.1:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004


#include "complex.h"
#include <float.h>
#include <math.h>

double  complex
(csqrt)(z)
double complex z;
{
    double complex w;
    double  x, y, r, t;

    x = creal(z);
    y = cimag(z);

    if (y == 0.0)
    {
	if (x == 0.0)
	{
	    w = 0.0 + y * I;
	}
	else
	{
	    r = fabs(x);
	    r = sqrt(r);
	    if (x < 0.0)
	    {
		w = 0.0 + r * I;
	    }
	    else
	    {
		w = r + y * I;
	    }
	}
	//return (w);
    }
    if (x == 0.0)
    {
	r = fabs(y);
	r = sqrt(0.5 * r);
	if (y > 0)
	    w = r + r * I;
	else
	    w = r - r * I;
	//return (w);
    }
    else
    {
    	int	ea, es;
	int	ex, ey;

	// Extract exponents of real (x) and imaginary (y) parts

	frexp(x, &ex);
	frexp(y, &ey);

	// Calculate scale & rescale factors

	es = ((ex + ey) >> 2);		// rescale

	ea = es << 1;			// scale		

	// Scale x and y

	x = ldexp(x, -ea);
	y = ldexp(y, -ea);

	w = x + y * I;
	r = cabs(w);

        if (x > 0)
        {
	    t = sqrt(0.5 * r + 0.5 * x);
	    r = ldexp(fabs((0.5 * y) / t), es);
	    t = ldexp(t, es);
        }
        else
        {
	    r = sqrt(0.5 * r - 0.5 * x);
	    t = ldexp(fabs((0.5 * y) / r), es);
	    r = ldexp(r, es);
        }
    
	if (y < 0)
	    w = t - r * I;
	else
	    w = t + r * I;
    }

    return (w);
}
