/*	clgammal.c
 *
 *	Natural logarithm of complex gamma function, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * #include <complex.h>
 * long double complex x, y, clgaml();
 *
 * y = clgammal( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the base e (2.718...) logarithm of the complex gamma
 * function of the complex argument.
 *
 * The logarithm of the gamma function is approximated by the
 * logarithmic version of Stirling's asymptotic formula.
 * Arguments of real part less than 16 are increased by recurrence.
 * The cosecant reflection formula is employed for arguments
 * having real part less than -16.5.
 *
 * Arguments greater than MAXLGM return MAXNUM and an error
 * message.  MAXLGM = 1.048e+4928L.
 *
 *
 *
 * ACCURACY:
 *
 *
 * arithmetic   domain     # trials      peak         rms
 * 80-bit long double:
 *    IEEE      -10,10       30000     7.9e-18      5.2e-19
 *    IEEE      -50,50       20000                  1.1e-19
 *    IEEE     -100,100      20000                  7.4e-20
 * 128-bit long double:
 *    IEEE      -10,10       21000     4.4e-32      3.6e-33
 *    IEEE     -100,100      23000                  4.4e-34
 * The error criterion was relative when the function magnitude
 * was greater than one but absolute when it was less than one.
 */

/*
Cephes Math Library Release 2.7:  April, 1998
Copyright 1998 Stephen L. Moshier
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004

#ifdef LD128BITS
#define NGITER 50.0L
#define NLGITER 50.0L
#define LGMXINT 44.4L
#define GSMALL 1.e-17L
#else
#define NGITER 20.0L
#define NLGITER 16.0L
#define LGMXINT 78.3L
#define GSMALL 1.e-9L
#endif

/* log pi */
static long double LOGPIL = 1.1447298858494001741434273513530587116473L;

#include "complex.h"
#include <float.h>
#include <math.h>

/* Asymptotic expansion of log gamma  */
#define NUMA 9
static long double A[NUMA] = {
#if 0
    1.3402864044168391994478951000690131124914E1L,
    -1.3924322169059011164274322169059011164274E0L,
#endif
    1.7964437236883057316493849001588939669435E-1L,	// 43867./244188.
    -2.9550653594771241830065359477124183006536E-2L,	// 3617./122400.
    6.4102564102564102564102564102564102564103E-3L,	// 1./156.
    -1.9175269175269175269175269175269175269175E-3L,	// 691./360360.
    8.4175084175084175084175084175084175084175E-4L,     // 1./1188.
    -5.9523809523809523809523809523809523809524E-4L,    // 1./1680.
    7.9365079365079365079365079365079365079365E-4L,     // 1./12600.
    -2.7777777777777777777777777777777777777778E-3L,    // 1./360.
    8.3333333333333333333333333333333333333333E-2L      // 1./12.
};
/* log( sqrt( 2*pi ) ) */
static long double LS2PIL = 0.918938533204672741780329736405617639861397L;
#define MAXLGML 1.04848146839019521116e+4928L



/* Logarithm of gamma function */

long double complex
clgammal(x)
long double complex x;
{
    long double complex c, w, u, v;
    long double p, q, a;
    int     i, cj;

    cj = 0;
    if (cimagl(x) < 0.0L)
    {
	cj = 1;
	x = conjl(x);
    }

/* -z gamma(-z) gamma(z) = pi / sin(pi z) */
/* log gamma(z) = log pi - log sin(pi z) - log(-z) - log gamma(-z) */
    if (creall(x) < -NLGITER)
    {
	q = creall(x);
	p = floorl(q);
	if ((p == q) && cimagl(x) == 0.0L)
	    goto loverf;
	if (fabsl(cimagl(x)) > LGMXINT)
	{
	    /* sin z grows exponentially with Im(z).  Find ln sin(pi z)
	       from |sin z| = sqrt( sin^2 x + sinh^2 y),
	       arg sin z = arctan(tanh y / tan x).  */
	    
	    c = M_PIl * cimagl(x) - M_LN2l +
		I * M_PIl * (0.5L - q);
	    c = LOGPIL - c - clgammal(1.0L - x);
	}
	else
	{
	    /* Reduce sine arg mod pi.  */
	    u = csinl(M_PIl * (x - p));
	    if (u == 0.0L)
		goto loverf;
	    w = clgammal(1.0L - x);
	    c = LOGPIL - clogl(u) - w;
	    /* Adjust for reduced sine arg.  */
	    // cimagl(c) += M_PIl * p;
	    c += I * (M_PIl * p);
	}
	goto ldone;
    }
    w = 0.0L;
    if (creall(x) < NLGITER)
    {
	/* To satisfy Im {clgam(z)} = arg cgamma(z), accumulate
	   arg u during the recurrence.  */
	a = 0.0L;
	w = 1.0L;
	p = 0.0L;
	u = x;
	while (creall(u) < NLGITER)
	{
	    if (u == 0.0L)
		goto loverf;
	    w *= u;
	    a += cargl(u);
	    p += 1.0L;
	    u = x + p;
	}
	x = u;
	/*      w = -logl(cabsl(w)) - I * a; */
	p = creall(w);
	q = cimagl(w);
	w = -0.5 * logl(p * p + q * q) - I * a;
    }

    if (creall(x) > MAXLGML)
    {
      loverf:
	//mtherr("clgaml", OVERFLOW);
	c = LDBL_MAX + LDBL_MAX * I;
	goto ldone;
    }

    c = (x - 0.5L) * clogl(x) - x + LS2PIL + w;

    if (cabsl(x) > 1.0e10L)
	goto ldone;

    v = 1.0L / (x * x);
    u = A[0];
    for (i = 1; i < NUMA; i++)
    {
	u = u * v + A[i];
    }
    c = c + u / x;

  ldone:
    if (cj)
	c = conjl(c);
    return (c);
}
