/*							clgammaf
 *
 *	Natural logarithm of complex gamma function, single precision
 *
 *
 *
 * SYNOPSIS:
 *
 * #include <complex.h>
 * float complex x, y, clgammaf();
 *
 * y = clgammaf( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the base e (2.718...) logarithm of the complex gamma
 * function of the argument.
 *
 * The logarithm of the gamma function is approximated by the
 * logarithmic version of Stirling's asymptotic formula.
 * Arguments of real part less than 8 are increased by recurrence.
 * The cosecant reflection formula is employed for arguments
 * having real part less than -8.
 *
 * Arguments greater than MAXLGM return MAXNUM and an error
 * message.  MAXLGM = 2.035093e36.
 *
 *
 *
 * ACCURACY:
 *
 *
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -20,20     9 000 000    2.6e-6       1.1e-7
 *    IEEE     -100,100      500 000                 7.5e-8
 * The error criterion was relative when the function magnitude
 * was greater than one but absolute when it was less than one.
 */
/*
Cephes Math Library Release 2.7:  March, 1998
Copyright 1984, 1998 Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004


#include "complex.h"
#include <float.h>
#include <math.h>


/* Asymptotic expansion of log gamma  */
static float A[] = {
#if 0
    -1.3924322169059011164274322169059011164274E0,
    1.7964437236883057316493849001588939669435E-1,
    -2.9550653594771241830065359477124183006536E-2,
    6.4102564102564102564102564102564102564103E-3,
    -1.9175269175269175269175269175269175269175E-3,
    8.4175084175084175084175084175084175084175E-4,
#endif
    -5.9523809523809523809523809523809523809524E-4,	// 1./1680.
    7.9365079365079365079365079365079365079365E-4,	// 1./12600.
    -2.7777777777777777777777777777777777777778E-3,	// 1./360.
    8.3333333333333333333333333333333333333333E-2	// 1./12.
};
/* log( sqrt( 2*pi ) ) */
static float LS2PI = 0.91893853320467274178f;
#define MAXLGM 2.035093e36f
static float LOGPI = 1.14472988584940017414f;

/* Logarithm of gamma function */

float   complex
clgammaf(x)
float complex x;
{
    float complex c, w, u, v;
    float   p, q, a;
    int     i, cj;

    cj = 0;
    if (cimagf(x) < 0)
    {
	cj = 1;
	x = conjf(x);
    }

/* Reflection formula -z gamma(-z) gamma(z) = pi / sin(pi z)  */
    if (crealf(x) < -12.0f)
    {
	q = crealf(x);
	p = floorf(q);
	if ((p == q) && (cimagf(x) == 0.0f))
	    goto loverf;
	if (fabsf(cimag(x)) > 18.4f)
	{
	    /* sin z grows exponentially with Im(z).  Find ln sin(pi z)
	       from |sin z| = sqrt( sin^2 x + sinh^2 y),
	       arg sin z = arctan(tanh y / tan x).  */
	    c = M_PI * cimagf(x) - 0.6931471805599453094f
		+ I * M_PI * (0.5f - q);
	    c = LOGPI - c - clgammaf(1.0f - x);
	}
	else
	{
	    /* Reduce sine arg mod pi.  */
	    u = csinf(M_PI * (x - p));
	    if (u == 0.0f)
		goto loverf;
	    w = clgammaf(1.0f - x);
	    c = LOGPI - clogf(u) - w;
	    /* Adjust for reduced sine arg.  */
	    // cimagf(c) += M_PI * p;
	    c += I * (M_PI * p);
	}
	goto ldone;
    }
    w = 0.0f;
    if (crealf(x) < 7.0f)
    {
	/* To satisfy Im {clgam(z)} = arg cgamma(z), accumulate
	   arg u during the recurrence.  */
	a = 0.0f;
	w = 1.0f;
	p = 0.0f;
	u = x;
	while (crealf(u) < 7.0f)
	{
	    if (u == 0.0f)
		goto loverf;
	    w *= u;
	    a += cargf(u);
	    p += 1.0f;
	    u = x + p;
	}
	x = u;
	w = -logf(cabsf(w)) - I * a;
    }

    if (crealf(x) > MAXLGM)
    {
      loverf:
	// mtherr("clgammaf", OVERFLOW);
	c = FLT_MAX + FLT_MAX * I;
	goto ldone;
    }

    c = (x - 0.5f) * clogf(x) - x + LS2PI + w;

    if (cabsf(x) > 1.0e8f)
	goto ldone;

    v = 1.0f / (x * x);
    u = A[0];
    for (i = 1; i < 4; i++)
    {
	u = u * v + A[i];
    }
    c = c + u / x;

  ldone:
    if (cj)
	c = conjf(c);
    return (c);
}
