/*	tgammal.c
 *
 *	Gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, tgammal();
 * extern int signgam;
 *
 * y = tgammal( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns gamma function of the argument.  The result is
 * correctly signed, and the sign (+1 or -1) is also
 * returned in a global (extern) variable named signgam.
 * This variable is also filled in by the logarithmic gamma
 * function lgam().
 *
 * Arguments |x| <= 13 are reduced by recurrence and the function
 * approximated by a rational function of degree 7/8 in the
 * interval (2,3).  Large arguments are handled by Stirling's
 * formula. Large negative arguments are made positive using
 * a reflection formula.
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE     -40,+40	    10000	3.6e-19	    7.9e-20
 *    IEEE    -1755,+1755   10000	4.8e-18	    6.5e-19
 *
 * Accuracy for large arguments is dominated by error in powl().
 *
 */

/*	--------------	*/
/*	gamma function	*/
/*	--------------	*/
/*
 * Return Specifications from ISO C99:
 *
 * -- tgamma(+Inf) returns +Inf.
 *
 * -- tgamma(x) returns a NaN and raises the invalid
 *    exception if x is a negative integer or zero.
 *
 * -- tgamma(-Inf) returns a NaN and raises the invalid
 *    exception.
 */
/*
Copyright 1994 by Stephen L. Moshier
Modified for DJGPP/GNU by KB Williams,
Kbwms@aol.com, December 2001
*/

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

/* *INDENT-OFF* */
/*
gamma(x+2)  = gamma(x+2) P(x)/Q(x)
0 <= x <= 1
Relative error
n=7, d=8
Peak error =  1.83e-20
Relative error spread =	 8.4e-23
*/

static long double P[8] =
{
 4.212760487471622013093E-5L,
 4.542931960608009155600E-4L,
 4.092666828394035500949E-3L,
 2.385363243461108252554E-2L,
 1.113062816019361559013E-1L,
 3.629515436640239168939E-1L,
 8.378004301573126728826E-1L,
 1.000000000000000000009E0L,
};
static long double Q[9] =
{
-1.397148517476170440917E-5L,
 2.346584059160635244282E-4L,
-1.237799246653152231188E-3L,
-7.955933682494738320586E-4L,
 2.773706565840072979165E-2L,
-4.633887671244534213831E-2L,
-2.243510905670329164562E-1L,
 4.150160950588455434583E-1L,
 9.999999999999999999908E-1L,
};
#
#define MAXGAML 1755.54834290446L

/* Stirling's formula for the gamma function
gamma(x) = sqrt(2 pi) x^(x-.5) exp(-x) (1 + 1/x P(1/x))
z(x) = x
13 <= x <= 1024
Relative error
n=8, d=0
Peak error =  9.44e-21
Relative error spread =	 8.8e-4
*/
#
static long double STIR[9] =
{
 7.147391378143610789273E-4L,
-2.363848809501759061727E-5L,
-5.950237554056330156018E-4L,
 6.989332260623193171870E-5L,
 7.840334842744753003862E-4L,
-2.294719747873185405699E-4L,
-2.681327161876304418288E-3L,
 3.472222222230075327854E-3L,
 8.333333333333331800504E-2L,
};
#
#define MAXSTIR 1024.0L

/* sqrt(2 * PI) */
static const long double SQTPI = 2.5066282746310005024157652848110E0L;
/* 1/gamma(x) = z P(z)
 * z(x) = 1/x
 * 0 < x < 0.03125
 * Peak relative error 4.2e-23
 */
#
static long double S[9] =
{
-1.193945051381510095614E-3L,
 7.220599478036909672331E-3L,
-9.622023360406271645744E-3L,
-4.219773360705915470089E-2L,
 1.665386113720805206758E-1L,
-4.200263503403344054473E-2L,
-6.558780715202540684668E-1L,
 5.772156649015328608253E-1L,
 1.000000000000000000000E0L,
};
#
/* 1/gamma(-x) = z P(z)
 * z(x) = 1/x
 * 0 < x < 0.03125
 * Peak relative error 5.16e-23
 * Relative error spread =  2.5e-24
 */
#
static long double SN[9] =
{
 1.133374167243894382010E-3L,
 7.220837261893170325704E-3L,
 9.621911155035976733706E-3L,
-4.219773343731191721664E-2L,
-1.665386113944413519335E-1L,
-4.200263503402112910504E-2L,
 6.558780715202536547116E-1L,
 5.772156649015328608727E-1L,
-1.000000000000000000000E0L,
};
#
/* *INDENT-ON* */
#define RETVAL printf("%s, line %3d, Retval set to %LG\n",\
		__FILE__, __LINE__, Retval);


//int	signgam = 0;
extern int signgam;

static long double polevll(long double, long double *, int);
static long double polevll(long double x, long double *p, int n)
{
    long double y;
    long double *A = p;

    y = *A++;
    do
    {
	y = y * x + *A++;
    }
    while (--n);
    return (y);
}
/* *INDENT-ON* */

/* Gamma function computed by Stirling's formula.
 */
static long double stirf(long double);
static long double stirf(long double x)
{
    long double y, w, v;

    w = 1.0L / x;
/* For large x, use rational coefficients from the analytical expansion.  */
    if (x > 1024.0L)
    {
/* *INDENT-OFF* */
	w =  (((((6.97281375836585777429E-5L  * w
		+ 7.84039221720066627474E-4L) * w
		- 2.29472093621399176955E-4L) * w
		- 2.68132716049382716049E-3L) * w
		+ 3.47222222222222222222E-3L) * w
		+ 8.33333333333333333333E-2L) * w
		+ 1.0L;
/* *INDENT-ON* */
    }
    else
    {
	w = 1.0L + w * polevll(w, STIR, 8);
    }
    y = expl(x);
    if (x > MAXSTIR)
    {					/* Avoid overflow in pow() */
	v = powl(x, 0.5L * x - 0.25L);
	y = v * (v / y);
    }
    else
    {
	y = powl(x, x - 0.5L) / y;
    }
    y = SQTPI * y * w;
    return (y);
}

/* ------------------------------------------------- */
/* tgammal - returns gamma of long double argument x */
/* ------------------------------------------------- */
# if defined __STDC__
long double tgammal(long double x)
# else
long double tgammal(x)
long double x;
# endif
{
    long double p, q, Retval, z;

    q = fabsl(x), p = floorl(q);

    if (!isfinitel(x))
    {
	if ((isinfl(x) > 0) || isnanl(x))
	{
	    Retval = x;
	}
	else				// -Inf
	{
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	    Retval = NAN;
	}
    }
    else if ((p == q) && x <= 0)	// x is negative integer or zero
    {
    	if (x)				// x is negative integer
	{
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	    Retval = NAN;
	}
	else        			// x is +-zero
	{
	    __math_set_errno(ERANGE);
	    __fp_raise_except(FE_DIVBYZERO);
	    Retval = copysignl(HUGE_VALL, x);
	}
    }
    else if ((q > MAXGAML) || (q <= 1.0L/LDBL_MAX))
    {
	__math_set_errno(ERANGE);
	__fp_raise_except(FE_OVERFLOW);
	Retval = copysignl(HUGE_VALL, x);
    }
    else
    {
	signgam = ((x >= 0) ? 1 : ((int)q & 1) ? 1 : -1);

	if (q > 13.0L)
	{
	    Retval = stirf(q);

	    if (x < 0.0L)
	    {
		z = q - p;
		if (z > 0.5L)
		{
		    --z;
		}
		z = q * sinl(PIL * z);	\
		z = fabsl(z) * Retval;

		Retval = PIL / z;
	    }
	    Retval = (signgam * Retval);
	}
	else				/* fabsl(x) <= 13 */
	{
	    z = 1.0L;
	    while (x >= 3.0L)
	    {
		--x;
		z *= x;
	    }

	    while (x < -0.03125L)	/* x < -1/32 */
	    {
		z /= x;
		++x;
	    }

	    if (x <= 0.03125L)
	    {
		if (x < 0.0L)
		{
		    x = -x;
		    Retval = z / (x * polevll(x, SN, 8));
		}
		else
		{
		    Retval = z / (x * polevll(x, S, 8));
		}
	    }
	    else
	    {
		while (x < 2.0L)
		{
		    z /= x;
		    ++x;
		}

		if (x == 2.0L)
		{
		    Retval = (z);
		}
		else
		{
		    x -= 2.0L;
		    p = polevll(x, P, 7);
		    q = polevll(x, Q, 8);
		    Retval = z * p / q;
		}
	    }
	}
    }
    return Retval;
}
