/*	tgammaf.c
 *
 *	Gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * float x, y, tgammaf();
 * extern int signgam;
 *
 * y = tgammaf( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns gamma function of the argument.  The result is
 * correctly signed, and the sign (+1 or -1) is also
 * returned in a global (extern) variable named signgam.
 * This same variable is also filled in by the logarithmic
 * gamma function lgam().
 *
 * Arguments between 0 and 10 are reduced by recurrence and the
 * function is approximated by a polynomial function covering
 * the interval (2,3).  Large arguments are handled by Stirling's
 * formula. Negative arguments are made positive using
 * a reflection formula.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE       0,-33      100,000     5.7e-7      1.0e-7
 *    IEEE       -33,0      100,000     6.1e-7      1.2e-7
 *
 */

/*
Cephes Math Library Release 2.7:  July, 1998
Copyright 1984, 1987, 1989, 1992, 1998 by Stephen L. Moshier
*/
// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, February 2004


#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

/* Stirling's formula for the gamma function
 * gamma(x) = sqrt(2 pi) x^(x-.5) exp(-x) ( 1 + 1/x P(1/x) )
 * .028 < 1/x < .1
 * relative error < 1.9e-11
 */

/* *INDENT-OFF* */
static float STIR[] = 
{
-2.705194986674176E-003,
 3.473255786154910E-003,
 8.333331788340907E-002,
};
/* *INDENT-ON* */

static	float	MAXGAM     = 34.84425627277176174;
static	float	MAXSTIR    = 26.77;
static	float	SQTFLT_MAX = 2.50662827463100050242;	/* sqrt( 2 pi ) */

//int signgam = 0;
extern int signgam;

static float polevlf(float, float *, int);

static float
polevlf(float xx, float *coef, int N)
{
    float   ans, x;
    float  *p;
    int     i;

    x = xx;
    p = coef;
    ans = *p++;

    i = N;
    do
	ans = ans * x + *p++;
    while (--i);

    return (ans);
}

/* Gamma function computed by Stirling's formula,
 * sqrt(2 pi) x^(x-.5) exp(-x) (1 + 1/x P(1/x))
 * The polynomial STIR is valid for 33 <= x <= 172.
 */
static float
stirf(float xx)
{
    float   x, y, w, v;

    x = xx;
    w = 1.0 / x;
    w = 1.0 + w * polevlf(w, STIR, 2);
    y = expf(-x);
    if (x >MAXSTIR)
    {					/* Avoid overflow in pow() */
	v = powf(x, 0.5 * x - 0.25);
	y *= v;
	y *= v;
    }
    else
    {
	y = powf(x, x - 0.5) * y;
    }
    y = SQTFLT_MAX * y * w;
    return (y);
}


/* *INDENT-OFF* */
/* gamma(x+2), 0 < x < 1 */
static float P[] = {
 1.536830450601906E-003,
 5.397581592950993E-003,
 4.130370201859976E-003,
 7.232307985516519E-002,
 8.203960091619193E-002,
 4.117857447645796E-001,
 4.227867745131584E-001,
 9.999999822945073E-001,
};
/* *INDENT-ON* */

float
tgammaf(float xx)
{
    float   p, q, x, z, nz;
    int     i, direction, negative;

    x = xx;
    signgam = 1;
    negative = 0;
    nz = 0.0;
    if ((isnanf)(x))
    {
	return (x);
    }
    if ((isinff)(x))
    {
	if (x > 0.0)
	{
	    return (x);
	}
	else				// -Inf
	{
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	    return (NAN);
	}
    }
    q = fabsf(x), p = floorf(q);

    if ((q < 1.0 / FLT_MAX) || (q > MAXGAM))
    {
	__math_set_errno(ERANGE);
	__fp_raise_except(FE_OVERFLOW);
	return copysign(HUGE_VALF, x);
    }
    if (x < 0.0)
    {
	negative = 1;
	if (p == q)
	{		
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	    return NAN;
	}
	i = p;
	if ((i & 1) == 0)
	    signgam = -1;
	nz = q - p;
	if (nz > 0.5)
	{
	    p += 1.0;
	    nz = q - p;
	}
	nz = q * sinf(PI * nz);
	if (nz == 0.0)
	{
	    __math_set_errno(ERANGE);
	    __fp_raise_except(FE_OVERFLOW);
	    return (signgam * HUGE_VALF);
	}
	if (nz < 0)
	    nz = -nz;
	x = q;
    }
    if (x >= 10.0)
    {
	z = stirf(x);
    }
    if (x < 2.0)
	direction = 1;
    else
	direction = 0;
    z = 1.0;
    while (x >= 3.0)
    {
	x -= 1.0;
	z *= x;
    }
    while (x < 2.0)
    {
	if (x < 1.e-4)
	    goto small;
	z *= x;
	x += 1.0;
    }

    if (direction)
	z = 1.0 / z;

    if (x == 2.0)
	return (z);

    x -= 2.0;
    p = z * polevlf(x, P, 7);

gdone:

    if (negative)
    {
	p = signgam * PI / (nz * p);
    }
    return (p);

small:
    if (x == 0.0)
    {
	__math_set_errno(EDOM);
	__fp_raise_except(FE_DIVBYZERO);
	return (copysignf(HUGE_VALF, x));
    }
    else
    {
	p = z / ((1.0 + 0.5772156649015329 * x) * x);
	goto gdone;
    }
}
