/*							tanhl.c
 *
 *	Hyperbolic tangent, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, tanhl();
 *
 * y = tanhl( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns hyperbolic tangent of argument in the range MINLOGL to
 * MAXLOGL.
 *
 * A rational function is used for |x| < 0.625.	 The form
 * x + x**3 P(x)/Q(x) of Cody _& Waite is employed.
 * Otherwise,
 *    tanh(x) = sinh(x)/cosh(x) = 1  -	2/(exp(2x) + 1).
 *
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE	-2,2	    30000	1.3e-19	    2.4e-20
 *
 */

/*
Cephes Math Library Release 2.7:  May, 1998
Copyright 1984, 1987, 1989, 1998 by Stephen L. Moshier
Modified for DJGPP/GCC by KB Williams, kbwms@aol.com,
December 2001
*/

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

#
static long double P[] =
{
    -6.8473739392677100872869E-5L,
    -9.5658283111794641589011E-1L,
    -8.4053568599672284488465E1L,
    -1.3080425704712825945553E3L,
};
static long double Q[] =
{
/* 1.0000000000000000000000E0L,*/
    9.6259501838840336946872E1L,
    1.8218117903645559060232E3L,
    3.9241277114138477845780E3L,
};

# if defined __STDC__
long double tanhl(long double x)
# else
long double tanhl(x)
long double x;
# endif
{
    long double p, q, s, z;

    if (x == 0.0L || isnanl(x))
    {
	z = x;
    }
    else if (fpclassifyl(x) == FP_SUBNORMAL)
    {
	z = x;
	__math_set_errno(ERANGE);
	__fp_raise_except(FE_UNDERFLOW);
    }        		       
    else
    {
	z = fabsl(x);
	if (z > 0.5L * MAXLOGL)
	{
	    z = copysignl(1.0L, x);
	}
	else if (z >= 0.625L)
	{
	    s = expl(2.0 * z);
	    z = 1.0L - 2.0 / (s + 1.0L);
	    if (x < 0)
		z = -z;
	}
	else
	{
	    s = x * x;

	    p = P[3] + s * (P[2] + s * (P[1] + s * P[0]));
	    q = Q[2] + s * (Q[1] + s * (Q[0] + s));

	    z = p / q;
	    z = x * s * z;
	    z = x + z;
	}
    }
    return (z);
}
