// -----------
// slmpyldbl.c
// -----------
//
// Function SlMpyLdbl
//
// Multiplies two single-length long double precision arguments to produce
// an extra-precision result.
//
// Returns the double-length long double result in space provided in the
// calling sequence.
//
// void SlMpyLdbl(long double ArgA, long double ArgB,
//              long double *HeadC, long double *TailC);
//
// Procedure is:
//
// 1. Save rounding mode, reset to round toward zero (truncate).
//
// 2. Calculate product C = A * B as a double-length product
//
//    a. Calculate Head of C
//
//       *HeadC = ArgA * ArgB;
//
//    b. Calculate Tail of C
//
//       Get fraction bits of ArgA and ArgB as 64-bit unsigned integers
//
//	 Multiply the two numbers to get the lower 64 bits of the product.
//
//	 Adjust the product as necessary to scale properly.
//
//	 Convert and store tail of product in *TailC
//
// 4. Restore rounding mode

#include <fdlibml.h>
#include <fenv.h>

void
SlMpyLdbl(long double ArgA, long double ArgB,
    long double *HeadC, long double *TailC)
{
    int     CrntRndMode;
    int     ExpA, ExpB, ExpC;//, Unused;
    ULLONG  FracBitsA, FracBitsB, UTailC;


// Save rounding mode, reset to round toward zero (truncate).

    CrntRndMode = fegetround();
    fesetround(FE_TOWARDZERO);

// Compute upper 64 bits of product

    *HeadC = ArgA * ArgB;

// Extract exponents from each argument and their product

    (void)frexpl(ArgA,   &ExpA);
    (void)frexpl(ArgB,   &ExpB);
    (void)frexpl(*HeadC, &ExpC);

// Get significand bits of ArgA and ArgB as 64-bit unsigned integers.

    GET_LDOUBLE_SIG(FracBitsA, ArgA);
    GET_LDOUBLE_SIG(FracBitsB, ArgB);

//  Multiply to get lower 64 bits of the product, adjust as necessary.

    UTailC = (FracBitsA * FracBitsB) << (ExpA + ExpB > ExpC);

//  Convert to correctly-signed long double and store as tail of product

    *TailC = copysignl(ldexpl((LDBL) UTailC, +ExpC - 64 - 64), *HeadC);

    fesetround(CrntRndMode);
}
