// ---------
// remquo.c
// ---------
//
// Function remquo()
//
// Synopsis
//
//	#include <math.h>
//	double remquo(double x, double y, int *quo);
//
// Description
//
// Function remquo computes the same remainder as function
// remainder.  In the object pointed to by quo, remquo
// stores the signed lowest three bits (range -7 to +7) of the
// integer value closest to the quotient x/y . This partial
// quotient can be used in some argument reduction algorithms.
//
// Returns
//
// Function remquo returns x REM y (as required by IEC 60559).
// Also, in the object pointed to by quo, function remquo stores
// the signed lowest three bits of the integer value closest to
// the quotient x/y.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, October 2003
//
#include <errno.h>
#include <fenv.h>
#include <float.h>
#include <fdlibml.h>

double
remquo(double x, double y, int *quo)
{
    double Retval;

    *quo   = 0;

    if (isnand(x) || isnand(y))
    {
    	Retval = NAN;
    }
    else if (isinfd(x) || (y == 0))
    {
       	__math_set_errno(EDOM);
    	Retval = NAN;
	__fp_raise_except(FE_INVALID);
    }
    else if (isinfd(y))
    {
    	Retval = x;
    }
    else
    {
	double	NewX;
	int	CrntRndDir, IntPart, Exponent;

        (void)frexp(y, &Exponent);

        if (Exponent < (DBL_MAX - 2))		// Reduce x
	    NewX = fmod(x, 8 * y);		// x < 8 * y
        else
            NewX = x;

        CrntRndDir = fegetround();
        fesetround(FE_TONEAREST);

        IntPart = rint(NewX / y);		// x/y rounded to int

        fesetround(CrntRndDir);

        Retval = NewX - IntPart * y;

	if (Retval == 0)
	    Retval = copysign(Retval, x);

	*quo = (IntPart >= 0) ? IntPart & 7	// get lowest 3 bits
			      : -((-IntPart) & 7);
    }

    return  Retval;
}
