/* @(#)e_pow.c 5.1 93/09/24 */
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */

/* Expansions and modifications for long double precision
   contributed by Stephen L. Moshier <moshier@na-net.ornl.gov> */

/* pow(x,y) return x**y
 *
 *		      n
 * Method:  Let x =  2	 * (1+f)
 *	1. Compute and return log2(x) in two pieces:
 *		log2(x) = w1 + w2,
 *	   where w1 has 53-24 = 29 bit trailing zeros.
 *	2. Perform y*log2(x) = n+y' by simulating muti-precision
 *	   arithmetic, where |y'|<=0.5.
 *	3. Return x**y = 2**n*exp(y'*log2)
 *
 * Special cases:
 *	1.  (anything) ** 0  is 1
 *	2.  (anything) ** 1  is itself
 *	3.  (anything except 1) ** NAN is NAN
 *	4.  NAN ** (anything except 0) is NAN
 *	5.  +-(|x| > 1) **  +INF is +INF
 *	6.  +-(|x| > 1) **  -INF is +0
 *	7.  +-(|x| < 1) **  +INF is +0
 *	8.  +-(|x| < 1) **  -INF is +INF
 *	9.  +-1		** +-INF is +1
 *	    +1 ** (anything)     is +1
 *	10. +0 ** (+anything except 0, NAN)		  is +0
 *	11. -0 ** (+anything except 0, NAN, odd integer)  is +0
 *	12. +0 ** (-anything except 0, NAN)		  is +INF
 *	13. -0 ** (-anything except 0, NAN, odd integer)  is +INF
 *	14. -0 ** (odd integer) = -( +0 ** (odd integer) )
 *	15. +INF ** (+anything except 0,NAN) is +INF
 *	16. +INF ** (-anything except 0,NAN) is +0
 *	17. -INF ** (anything)	= -0 ** (-anything)
 *	18. (-anything) ** (integer) is (-1)**(integer)*(+anything**integer)
 *	19. (-anything except 0 and inf) ** (non-integer) is NAN
 *

C9X rules:
   -- pow(x, +-0) returns 1 for any x, even a NaN.
   -- pow(x, +inf) returns +inf for |x|>1.
   -- pow(x, +inf) returns +0 for |x|<1.
   -- pow(x, -inf) returns +0 for |x|>1.
   -- pow(x, -inf) returns +inf for |x|<1.
   -- pow(+inf, y) returns +inf for y>0.
   -- pow(+inf, y) returns +0 for y<0.
   -- pow(-inf, y) returns -inf for y an odd integer > 0.
   -- pow(-inf, y) returns +inf for y>0 and not an odd integer.
   -- pow(-inf, y) returns -0 for y an odd integer < 0.
   -- pow(-inf, y) returns +0 for y<0 and not an odd integer.
   -- pow(+-1,	+-inf)	returns	 a  NaN	 and  raises  the  invalid
	    exception.
   -- pow(x,  y)  returns  a  NaN  and	raises	the   invalid
	    exception for finite x<0 and finite non-integer y.
   -- pow(+-0,	y)  returns  +-inf  and	 raises the divide-by-zero
	    exception for y an odd integer < 0.
   -- pow(+-0, y) returns  +inf	 and  raises  the  divide-by-zero
	    exception for y<0 and not an odd integer.
   -- pow(+-0, y) returns +-0 for y an odd integer > 0.
   -- pow(+-0, y) returns +0 for y>0 and not an odd integer.

 * Accuracy:
 *	pow(x,y) returns x**y nearly rounded. In particular
 *			pow(integer,integer)
 *	always returns the correct integer provided it is
 *	representable.
 *
 */
/*
Modified for DJGPP/GCC by KB Williams,
kbwms@aol.com, December 2001
*/
#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>
static long double bp[] =
{
    1.0L,
    1.5L,
};

/* log_2(1.5) */
static long double dp_h[] =
{
    0.0,
    5.849624983966350555419921875E-1L
};

/* Low part of log_2(1.5) */
static long double dp_l[] = {
    0.0,
    2.3245211259117467564478165087598144076925E-9L
};

static long double zero = 0.0L, one = 1.0L, two = 2.0L,
  /* two53 = 9007199254740992.0L, */
    two64 = 1.8446744073709551616e19L,
    huge  = 1.0e3000L,
    tiny  = 1.0e-3000L,
    third = 0.33333333333333333333333333333333333333L;

/* 3/2 log x = 3 z + z^3 + z^3 (z^2 R(z^2))
   z = (x-1)/(x+1)
   1 <= x <= 1.25
   Peak relative error 4.1e-24 */
static long double L[] =
{
    6.0000000000000007169547191663855889491040E-1L,
    4.2857142857132763806701255649289162379800E-1L,
    3.3333333338738505409595007515712554280000E-1L,
    2.7272725804001041082311989184109244489036E-1L,
    2.3077145913915231227801687059313748810780E-1L,
    1.9980884602415972761639642101602574102351E-1L,
    1.8511538985136111886620516443876540497628E-1L
};

/* exp(x) = 1 + x - x / (1 - 2 / (x - x^2 R(x^2)))
   0 <= x <= 0.5
   Peak relative error 4.8e-23	*/

static long double P[] =
{
    1.666666666666666664382962387166536461319E-1L,
    -2.777777777777759227224746992835215024890E-3L,
    6.613756613701664005813566424838927765351E-5L,
    -1.653439145389401684254028056730766195090E-6L,
    4.175344919663847321134018383127756053825E-8L,
    -1.056546116151985479218653485492915080642E-9L,
    2.607290471950409562201869799828791565119E-11L
};

static long double
  /* ln 2 */
    lg2	  = 6.9314718055994530941723212145817656807550E-1L,
    lg2_h = 6.931471787393093109130859375E-1L,
    lg2_l = 1.8206359985041461839581765680755001343603E-9L,
    ovt	  = 8.0085662595372944372e-0017L,
  /* 2/(3*log(2)) */
    cp	 = 9.6179669392597560490661645400126142495110E-1L,
    cp_h = 9.61796693503856658935546875E-1L,
    cp_l = 4.2211894597106957900126142495109730276866E-10L,
  /* 1 / log(2) */
    ivln2   = 1.4426950408889634073599246810018921374266E0L,
    ivln2_h = 1.44269503653049468994140625L,
    ivln2_l = 4.3584687174185184310018921374266459541530E-9L;

// ----
// NANL
// ----
static
union
    {
	unsigned short		U_Value[5];
	long double		D_Value;
    }
    LDFU = { {0,0,0,0xc000,0x7fff} };

static
long double powlx(long double, long double);

long double powl(long double x, long double y)
{
    long double Retval;

    if (x == one)
    {
    	Retval = x;
    }
    else if ((x == -one) && isinfl(y))
    {
         Retval = one;
    }
    else
    {
	Retval = powlx(x,y);
	if (isnanl(Retval))
	{
	     Retval = LDFU.D_Value;
	}
    }
    return Retval;
}

static
long double powlx(long double x, long double y)
{
    long double z, ax, z_h, z_l, p_h, p_l;
    long double Y1, t1, t2, r, s, t, u, v, w;

    int	    i, j, k, yisint, n;
    UINT    se, ix, ix0, ix1, iy, iy0, iy1;
    UINT    hax, iax0, iax1;
    int	    hx, hy;

    GET_LDOUBLE_WORDS(se, ix0, ix1, x);
    hx = (se << 16) | (ix0 >> 16);
    ix = hx & 0x7fffffff;

    GET_LDOUBLE_WORDS(se, iy0, iy1, y);
    hy = (se << 16) | (iy0 >> 16);
    iy = hy & 0x7fffffff;

    /* y==zero: x**0 = 1 */
    if ((iy | iy0 | iy1) == 0)
	return one;

    /* +-NaN return x+y */
    if ((ix > 0x7fff8000)
	|| ((ix == 0x7fff8000) && (((ix0 & 0xffff) | ix1) != 0))
	|| (iy > 0x7fff8000)
	|| ((iy == 0x7fff8000) && (((iy0 & 0xffff) | iy1) != 0)))
	return x + y;

    /* determine if y is an odd int when x < 0
     * yisint = 0	... y is not an integer
     * yisint = 1	... y is an odd int
     * yisint = 2	... y is an even int
     */
    yisint = 0;
    if (hx < 0)
    {
	if (iy >= 0x403f8000) /* 2^64 */
	    yisint = 2;			/* even integer y */
	else if (iy >= 0x3fff8000)	/* 1.0 */
	{
	    if (floorl(y) == y)
	    {
		z = 0.5 * y;
		if (floorl(z) == z)
		    yisint = 2;
		else
		    yisint = 1;
	    }
	}
    }

    /* special value of x */
    if ((x == 0.0L) && (y < 0.0L))
    {
	__math_set_errno(ERANGE);
	z = HUGE_VALL;

	if (yisint == 1)
	{
	    z = copysignl(z, x);
	}

	__fp_raise_except(FE_DIVBYZERO);
	return z;
    }

    /* special value of y */
    if (((iy0 & 0x7fff) | iy1) == 0)
    {
	if (iy == 0x7fff8000)		/* y is +-inf */
	{
	    if (((ix - 0x3fff8000) | (ix0 & 0x7fff) | ix1) == 0)
		return y - y;		/* inf**+-1 is NaN */
	    else if (ix >= 0x3fff8000)	/* (|x|>1)**+-inf = inf,0 */
		return (hy >= 0) ? y : zero;
	    else			/* (|x|<1)**-,+inf = inf,0 */
		return (hy < 0) ? -y : zero;
	}
	if (iy == 0x3fff8000)
	{				/* y is	 +-1 */
	    if (hy < 0)
		return one / x;
	    else
		return x;
	}
	if (hy == 0x40008000)
	    return x * x;		/* y is	 2 */
	if (hy == 0x3ffe8000)
	{				/* y is	 0.5 */
	    if (hx >= 0)		/* x >= +0 */
		/*	return __ieee754_sqrtl (x); */
		return sqrtl(x);
	}
    }

    ax = fabsl(x);

    /* special value of x */
    if (((ix0 & 0xffff) | ix1) == 0)
    {
	if (ix == 0x7fff8000 || ix == 0 || ix == 0x3fff8000)
	{
	    z = ax;			/*x is +-0,+-inf,+-1 */
	    if (hy < 0)
		z = one / z;		/* z = (1/|x|) */
	    if (hx < 0)
	    {
		if (((ix - 0x3fff8000) | yisint) == 0)
		{
		    __math_set_errno(EDOM);
		    __fp_raise_except(FE_INVALID);
		    z = (z - z) / (z - z);	/* (-1)**non-int is NaN */
		}
		else if (yisint == 1)
		    z = -z;		/* (x<0)**odd = -(|x|**odd) */
	    }
	    return z;
	}
    }

    /* (x<0)**(non-int) is NaN */
    if (((((UINT) hx >> 31) - 1) | yisint) == 0)
    {
	__math_set_errno(EDOM);
	__fp_raise_except(FE_INVALID);
	return (x - x) / (x - x);
    }
    /* |y| is huge */
    /* if (1 - 1/65536)^|y| underflows, |y| > 7.47e8 */
    if (iy > 0x401cb21d)		/* 7.47061248e8 */
    {
	/* if (1 - 2^-64)^|y| underflows, |y| > 2.1028e23 must o/uflow */
	if (iy > 0x404cb21d)		/*  2.10279047382233606258688e23 */
	{
	    if (ix <= 0x3ffeffff)	/* 9.999847412109375 */
		return (hy < 0) ? huge * huge : tiny * tiny;
	    if (ix >= 0x3fff8000)	/* 1.0 */
		return (hy > 0) ? huge * huge : tiny * tiny;
	}

	/* over/underflow if x is not close to one */
	if (ix < 0x3ffeffff)
	    return (hy < 0) ? huge * huge : tiny * tiny;
	if (ix > 0x3fff8000)
	    return (hy > 0) ? huge * huge : tiny * tiny;

	/* now |1-x| is tiny <= 2**-15, suffice to compute
	   log(x) by x-x^2/2+x^3/3-x^4/4 +x^5/5 */

	t = ax - 1;			/* t has 32 trailing zeros */
	w = (t * t) * (0.5L - t * (third - t * (0.25 - t * 0.2)));

	u = ivln2_h * t;		/* ivln2_h has 28 sig. bits */
	v = t * ivln2_l - w * ivln2;
	t1 = u + v;

	/*	SET_LOW_WORD (t1, 0); */
	GET_LDOUBLE_WORDS(hax, iax0, iax1, t1);
	iax1 = 0;
	iax0 &= 0xfffffff0;
	SET_LDOUBLE_WORDS(t1, hax, iax0, iax1);
	t2 = v - (t1 - u);
    }
    else
    {
	long double s2, s_h, s_l, t_h, t_l;
	n = 0;

	/* take care subnormal number */
	if (ix < 0x00010000)
	{
	    ax *= two64;
	    n -= 64;
	    GET_LDOUBLE_WORDS(hax, iax0, iax1, ax);
	    ix = (hax << 16) | (iax0 >> 16);
	}
	n += ((ix) >> 16) - 0x3fff;
	j = ix & 0x0000ffff;

	/* determine interval */
	ix = j | 0x3fff0000;		/* normalize ix */
	if (j <= 0x9cc4)		/* 1.2247314453125 */
	    k = 0;			/* |x|<sqrt(3/2) */
	else if (j < 0xddb3)		/* 1.732025146484375 */
	    k = 1;			/* |x|<sqrt(3)	 */
	else
	{
	    k = 0;
	    n += 1;
	    ix -= 0x00010000;
	}


	GET_LDOUBLE_WORDS(hax, iax0, iax1, ax);
	SET_LDOUBLE_WORDS(ax, (ix >> 16), iax0, iax1);

	/* compute s = s_h+s_l = (x-1)/(x+1) or (x-1.5)/(x+1.5) */
	u = ax - bp[k];			/* bp[0]=1.0, bp[1]=1.5 */
	v = one / (ax + bp[k]);
	s = u * v;
	s_h = s;

	/* SET_LOW_WORD (s_h, 0); */
	GET_LDOUBLE_WORDS(hax, iax0, iax1, s_h);
	iax1 = 0;
	iax0 &= 0xfffffff0;
	SET_LDOUBLE_WORDS(s_h, hax, iax0, iax1);

	/* t_h=ax+bp[k] High */
	t_h = ax + bp[k];
	GET_LDOUBLE_WORDS(hax, iax0, iax1, t_h);
	iax1 = 0;
	iax0 &= 0xfffffff0;
	SET_LDOUBLE_WORDS(t_h, hax, iax0, iax1);
	t_l = ax - (t_h - bp[k]);
	s_l = v * ((u - s_h * t_h) - s_h * t_l);

	/* compute log(ax) */
	s2 = s * s;
	r = s2 * s2 * (L[0] + s2 * (L[1] + s2 * (L[2] + s2 * (L[3]
			+ s2 * (L[4] + s2 * (L[5] + s2 * L[6]))))));
	r += s_l * (s_h + s);
	s2 = s_h * s_h;
	t_h = 3.0 + s2 + r;

	/*SET_LOW_WORD (t_h, 0); */
	GET_LDOUBLE_WORDS(hax, iax0, iax1, t_h);
	iax1 = 0;
	iax0 &= 0xfffffff0;
	SET_LDOUBLE_WORDS(t_h, hax, iax0, iax1);
	t_l = r - ((t_h - 3.0) - s2);

	/* u+v = s*(1+...) */
	u = s_h * t_h;
	v = s_l * t_h + t_l * s;

	/* 2/(3log2)*(s+...) */
	p_h = u + v;

	/* SET_LOW_WORD (p_h, 0); */
	GET_LDOUBLE_WORDS(hax, iax0, iax1, p_h);
	iax1 = 0;
	iax0 &= 0xfffffff0;
	SET_LDOUBLE_WORDS(p_h, hax, iax0, iax1);
	p_l = v - (p_h - u);
	z_h = cp_h * p_h;		/* cp_h+cp_l = 2/(3*log2) */
	z_l = cp_l * p_h + p_l * cp + dp_l[k];

	/* log2(ax) = (s+..)*2/(3*log2) = n + dp_h + z_h + z_l */
	t = (long double)n;
	t1 = (((z_h + z_l) + dp_h[k]) + t);

	/* SET_LOW_WORD (t1, 0); */
	GET_LDOUBLE_WORDS(hax, iax0, iax1, t1);
	iax1 = 0;
	iax0 &= 0xfffffff0;
	SET_LDOUBLE_WORDS(t1, hax, iax0, iax1);
	t2 = z_l - (((t1 - t) - dp_h[k]) - z_h);
    }

    /* s (sign of result -ve**odd) = -1 else = 1 */
    s = one;
    if (((((UINT) hx >> 31) - 1) | (yisint - 1)) == 0)
	s = -one;			/* (-ve)**(odd int) */

    /* split up y into Y1+y2 and compute (Y1+y2)*(t1+t2) */
    Y1 = y;

    /* SET_LOW_WORD (Y1, 0); */
    GET_LDOUBLE_WORDS(hax, iax0, iax1, Y1);
    iax1 = 0;
    iax0 &= 0xfffffff0;
    SET_LDOUBLE_WORDS(Y1, hax, iax0, iax1);

    p_l = (y - Y1) * t1 + y * t2;
    p_h = Y1 * t1;
    z = p_l + p_h;

    /* EXTRACT_WORDS (j, i, z); */
    GET_LDOUBLE_WORDS(hax, iax0, iax1, z);
    j = (hax << 16) | (iax0 >> 16);
    if (j >= 0x400d8000)
    {					/* z >= 16384 */
	if (((j - 0x400d8000) | (iax0 & 0xffff) | iax1) != 0)	/* if z > 16384 */
	    return s * huge * huge;	/* overflow */
	else
	{
	    if (p_l + ovt > z - p_h)
		return s * huge * huge; /* overflow */
	}
    }
    else if ((j & 0x7fffffff) >= 0x400d807c)	/* z <= -16446 */
    {
	if (((j - 0xc00d807c) | (iax0 & 0xffff) | iax1) != 0)	/* z < -16446 */
	    return s * tiny * tiny;	/* underflow */
	else
	{
	    if (p_l <= z - p_h)
		return s * tiny * tiny; /* underflow */
	}
    }
    /*
     * compute 2**(p_h+p_l)
     */
    i = j & 0x7fffffff;
    k = (i >> 16) - 0x3fff;
    n = 0;
    if (i > 0x3ffe8000)
    {					/* if |z| > 0.5, set n = [z+0.5] */
	n = floorl(z + 0.5L);
	t = n;
	p_h -= t;
    }
    t = p_l + p_h;

    /* SET_LOW_WORD (t, 0); */
    GET_LDOUBLE_WORDS(hax, iax0, iax1, t);
    iax1 = 0;
    iax0 &= 0xfffffff0;
    SET_LDOUBLE_WORDS(t, hax, iax0, iax1);
    u = t * lg2_h;
    v = (p_l - (t - p_h)) * lg2 + t * lg2_l;
    z = u + v;
    w = v - (z - u);

    /* expl(z); */
    t = z * z;
    t1 = z - t * (P[0] + t * (P[1] + t * (P[2] + t * (P[3]
		    + t * (P[4] + t * (P[5] + t * P[6]))))));
    r = (z * t1) / (t1 - two) - (w + z * w);
    z = one - (r - z);

    /* GET_HIGH_WORD (j, z); */
    GET_LDOUBLE_WORDS(hax, iax0, iax1, z);
    j = (hax << 16) | (iax0 >> 16);
    j += (n << 16);
    if ((j >> 16) <= 0)
	z = scalbnl(z, n);		/* subnormal output */
    else
    {
	/* SET_HIGH_WORD (z, j); */
	GET_LDOUBLE_WORDS(hax, iax0, iax1, z);
	hax = j >> 16;
	SET_LDOUBLE_WORDS(z, hax, iax0, iax1);
    }
    return s * z;
}
