// ------------
// lrint.c
// ------------
//
// From ISO C99:
//
// Function lrint
//
// Synopsis
//
//	#include <math.h>
//	long int lrint(double x);
//
// Description
//
// The lrint function rounds its argument to the nearest
// integer value, rounding according to the current rounding
// direction. If the rounded value is outside the range of
// the return type, the numeric result is unspecified.	A
// range error may occur if the magnitude of x is too large.
// In this case, global variable errno is set to ERANGE.
//
// Returns
//
// The lrint function returns the rounded integer value.
//
// Remarks
//
// The lrint function provides floating-to-integer conversion
// as prescribed by IEC 60559. It rounds according to the current
// rounding direction.	If the rounded value is outside the range
// of the return type, the numeric result is undefined and the
// invalid exception is raised.	 When no other exception is raised
// during the conversion and the result differs from the argument,
// the inexact exception is raised.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, September 2003
//
#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long int lrint(double Arg)
{
    int		CrntExcepts, Exponent;
    double	DblIntVal;

    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    ROUND_TO_INT(Arg, DblIntVal);

    (void)frexp(DblIntVal, &Exponent);

    if ((Exponent > 31) || !isfinited(Arg))
    {
	__math_set_errno(EDOM);
	DblIntVal = -LONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != DblIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long int) DblIntVal;
}
