/*							logl.c
 *
 *	Natural logarithm, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, logl();
 *
 * y = logl( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the base e (2.718...) logarithm of x.
 *
 * The argument is separated into its exponent and fractional
 * parts.  If the exponent is between -1 and +1, the logarithm
 * of the fraction is approximated by
 *
 *     log(1+x) = x - 0.5 x**2 + x**3 P(x)/Q(x).
 *
 * Otherwise, setting  z = 2(x-1)/(x+1),
 *
 *     log(x) = z + z**3 P(z)/Q(z).
 *
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE	0.5, 2.0    150000	8.71e-20    2.75e-20
 *    IEEE     exp(+-10000) 100000	5.39e-20    2.34e-20
 *
 * In the tests over the interval exp(+-10000), the logarithms
 * of the random arguments were uniformly distributed over
 * [-10000, +10000].
 *
 * ERROR MESSAGES:
 *
 * log singularity:  x = 0; returns -HUGE_VALL 
 * log domain:	     x < 0; returns NaN
 * 
 * 
 *  */

/*
Cephes Math Library Release 2.7:  May, 1998
Copyright 1984, 1990, 1998 by Stephen L. Moshier
Modified for DJGPP/GCC by KB Williams, 
kbwms@aol.com, December 2001 & October 2003
*/

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

/* Coefficients for log(1+x) = x - x**2/2 + x**3 P(x)/Q(x)
 * 1/sqrt(2) <= x < sqrt(2)
 * Theoretical peak relative error = 2.32e-20
 */

/* *INDENT-OFF* */

static long double P[] =
{
    4.5270000862445199635215E-5L,
    4.9854102823193375972212E-1L,
    6.5787325942061044846969E0L,
    2.9911919328553073277375E1L,
    6.0949667980987787057556E1L,
    5.7112963590585538103336E1L,
    2.0039553499201281259648E1L,
};
static long double Q[] = {
 /* 1.0000000000000000000000E0,*/
    1.5062909083469192043167E1L,
    8.3047565967967209469434E1L,
    2.2176239823732856465394E2L,
    3.0909872225312059774938E2L,
    2.1642788614495947685003E2L,
    6.0118660497603843919306E1L,
};

/* Coefficients for log(x) = z + z^3 P(z^2)/Q(z^2),
 * where z = 2(x-1)/(x+1)
 * 1/sqrt(2) <= x < sqrt(2)
 * Theoretical peak relative error = 6.16e-22
 */

static long double R[4] =
{
    1.9757429581415468984296E-3L,
    -7.1990767473014147232598E-1L,
    1.0777257190312272158094E1L,
    -3.5717684488096787370998E1L,
};
static long double S[4] =
{
  /* 1.00000000000000000000E0L,*/
    -2.6201045551331104417768E1L,
    1.9361891836232102174846E2L,
    -4.2861221385716144629696E2L,
};
static long double C1 = 6.9314575195312500000000E-1L;
static long double C2 = 1.4286068203094172321215E-6L;

#if defined __STDC__
extern long double frexpl(long double, int *);
extern long double ldexpl(long double, int);
extern int isfinitel(long double);
#else
long double frexpl(), ldexpl(), isfinitel();
#endif

long double logl(long double);

/* *INDENT-ON* */

# if defined __STDC__
long double logl(long double x)
# else
long double logl(x)
long double x;
# endif
{
    long double p, q, r, s, y, z, Retval;
    int	    e;

    /* Test for domain */

    if (x == 1.0L)
    {
	Retval = 0.0L;
    }
    else if (x <= 0.0L)
    {
	if (x == 0.0L)
	{
	    __math_set_errno(ERANGE);
	    __fp_raise_except(FE_DIVBYZERO);
	    Retval = -HUGE_VALL;
	}
	else
	{
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	    Retval = NAN;
	}
    }
    else if (!isfinitel(x))
    {
	Retval = x;
    }
    else
    {
	/* separate mantissa from exponent */

	/* Note, frexp is used so that denormal numbers
	 * will be handled properly.
	 */

	x = frexpl(x, &e);

	/* logarithm using log(x) = z + z**3 P(z)/Q(z),
	 * where z = 2(x-1)/x+1)
	 */
	if ((e > 2) || (e < -2))
	{
	    if (x < SQRTH)
	    {				/* 2( 2x-1 )/( 2x+1 ) */
		e -= 1;
		z = x - 0.5L;
		y = 0.5L * z + 0.5L;
	    }
	    else
	    {				/*  2 (x-1)/(x+1)   */
		z = x - 0.5L;
		z -= 0.5L;
		y = 0.5L * x + 0.5L;
	    }
	    x = z / y;
	    z = x * x;
	    r = z * (R[3] + z * (R[2] + z * (R[1] + z * R[0])));
	    s = S[2] + z * (S[1] + z * (S[0] + z));
	    z = x * r / s;
	    z = z + e * C2;
	    z = z + x;
	    z = z + e * C1;
	    Retval = (z);
	}

	else
	{
	    /* logarithm using log(1+x) = x - .5x**2 + x**3 P(x)/Q(x) */

	    if (x < SQRTH)
	    {
		e -= 1;
		x = ldexpl(x, 1) - 1.0L;	/*  2x - 1  */
	    }
	    else
	    {
		x = x - 1.0L;
	    }
	    z = x * x;
/* *INDENT-OFF* */
	    p = P[6]+x*(P[5]+x*(P[4]+x*(P[3]+x*(P[2]+x*(P[1]+x*P[0])))));
	    q = Q[5]+x*(Q[4]+x*(Q[3]+x*(Q[2]+x*(Q[1]+x*(Q[0]+x)))));
/* *INDENT-ON* */

	    y = x * z * p/q;
	    y = y + e * C2;
	    z = y - ldexpl(z, -1);	/*  y - 0.5 * z	 */

	    /* Note, the sum of above terms does not exceed x/4,
	     * so it contributes at most about 1/4 lsb to the error.
	     */
	    z = z + x;
	    z = z + e * C1;		/* This sum has an error of 1/2 lsb. */
	    Retval = (z);
	}
    }
    return Retval;
}
