/*							log10l.c
 *
 *	Common logarithm, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, log10l();
 *
 * y = log10l( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the base 10 logarithm of x.
 *
 * The argument is separated into its exponent and fractional
 * parts.  If the exponent is between -1 and +1, the logarithm
 * of the fraction is approximated by
 *
 *     log(1+x) = x - 0.5 x**2 + x**3 P(x)/Q(x).
 *
 * Otherwise, setting  z = 2(x-1)/x+1),
 *
 *     log(x) = z + z**3 P(z)/Q(z).
 *
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE	0.5, 2.0     30000	9.0e-20	    2.6e-20
 *    IEEE     exp(+-10000)  30000	6.0e-20	    2.3e-20
 *
 * In the tests over the interval exp(+-10000), the logarithms
 * of the random arguments were uniformly distributed over
 * [-10000, +10000].
 *
 * ERROR MESSAGES:
 *
 * log singularity:  x = 0; returns MINLOG
 * log domain:	     x < 0; returns MINLOG
 */

/*
Cephes Math Library Release 2.2:  January, 1991
Copyright 1984, 1991 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
Modified for DJGPP/GCC by KB Williams, kbwms@aol.com,
December 2001
*/

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

/* Coefficients for log(1+x) = x - x**2/2 + x**3 P(x)/Q(x)
 * 1/sqrt(2) <= x < sqrt(2)
 * Theoretical peak relative error = 6.2e-22
 */

/* *INDENT-OFF* */
static long double P[] =
{
    4.9962495940332550844739E-1L,
    1.0767376367209449010438E1L,
    7.7671073698359539859595E1L,
    2.5620629828144409632571E2L,
    4.2401812743503691187826E2L,
    3.4258224542413922935104E2L,
    1.0747524399916215149070E2L,
};
static long double Q[] =
{
/* 1.0000000000000000000000E0,*/
    2.3479774160285863271658E1L,
    1.9444210022760132894510E2L,
    7.7952888181207260646090E2L,
    1.6911722418503949084863E3L,
    2.0307734695595183428202E3L,
    1.2695660352705325274404E3L,
    3.2242573199748645407652E2L,
};

/* Coefficients for log(x) = z + z^3 P(z^2)/Q(z^2),
 * where z = 2(x-1)/(x+1)
 * 1/sqrt(2) <= x < sqrt(2)
 * Theoretical peak relative error = 6.16e-22
 */

static long double R[4] =
{
    1.9757429581415468984296E-3L,
    -7.1990767473014147232598E-1L,
    1.0777257190312272158094E1L,
    -3.5717684488096787370998E1L,
};
static long double S[4] =
{
/* 1.00000000000000000000E0L,*/
    -2.6201045551331104417768E1L,
    1.9361891836232102174846E2L,
    -4.2861221385716144629696E2L,
};
/* log10(2) */
#define L102A 0.3125L
#define L102B -1.1470004336018804786261e-2L
/* log10(e) */
#define L10EA 0.5L
#define L10EB -6.5705518096748172348871e-2L

/* *INDENT-ON* */

# if defined __STDC__
long double log10l(long double x)
# else
long double log10l(x)
long double x;
# endif
{
    long double p, q, r, s, y;
    volatile long double z;
    int	    e;

    if (x == 1.0L)
	return 0.0L;

    if (isnanl(x))
	return (x);

	/* Test for domain */
    if (x <= 0.0L)
    {
	if (x == 0.0L)
	{
	    __math_set_errno(ERANGE);
	    __fp_raise_except(FE_DIVBYZERO);
	    return -HUGE_VALL;
	}
	else
	{
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	    return NAN;
	}
    }
    if (isinfl(x))
	return (x);
	/* separate mantissa from exponent */

	/* Note, frexp is used so that denormal numbers
	 * will be handled properly.
	 */
    x = frexpl(x, &e);


	/* logarithm using log(x) = z + z**3 P(z)/Q(z),
	 * where z = 2(x-1)/x+1)
	 */
    if ((e > 2) || (e < -2))
    {
	if (x < SQRTH)
	{				/* 2( 2x-1 )/( 2x+1 ) */
	    e -= 1;
	    z = x - 0.5L;
	    y = 0.5L * z + 0.5L;
	}
	else
	{				/*  2 (x-1)/(x+1)   */
	    z = x - 0.5L;
	    z -= 0.5L;
	    y = 0.5L * x + 0.5L;
	}
	x = z / y;
	z = x * x;

	//y = x * (z * polevll(z, R, 3) / p1evll(z, S, 3));

	r = z * (R[3] + z * (R[2] + z * (R[1] + z * R[0])));
	s = S[2] + z * (S[1] + z * (S[0] + z));
	y = x * r / s;
	goto done;
    }


	/* logarithm using log(1+x) = x - .5x**2 + x**3 P(x)/Q(x) */

    if (x < SQRTH)
    {
	e -= 1;
	x = ldexpl(x, 1) - 1.0L;	/*  2x - 1  */
    }
    else
    {
	x = x - 1.0L;
    }
    z = x * x;

    //y = x * (z * polevll(x, P, 6) / p1evll(x, Q, 7));

/* *INDENT-OFF* */
    p = P[6]+x*(P[5]+x*(P[4]+x*(P[3]+x*(P[2]+x*(P[1]+x*P[0])))));
    q = Q[6]+x*(Q[5]+x*(Q[4]+x*(Q[3]+x*(Q[2]+x*(Q[1]+x*(Q[0]+x))))));
/* *INDENT-ON* */

    y = x * z * p/q;
    y = y - ldexpl(z, -1);		/* -0.5x^2 + ... */

  done:

/* Multiply log of fraction by log10(e)
 * and base 2 exponent by log10(2).
 *
 * ***CAUTION***
 *
 * This sequence of operations is critical and it may
 * be horribly defeated by some compiler optimizers.
 */
    z = y * (L10EB);
    z += x * (L10EB);
    z += e * (L102B);
    z += y * (L10EA);
    z += x * (L10EA);
    z += e * (L102A);

    return (z);
}
