// ------------
// llrint.c
// ------------
//
// From ISO C99:
//
// Function llrint
//
// Synopsis
//
//	#include <math.h>
//	long long int llrint(double x);
//
// Description
//
// Using the current rounding direction, function llrint
// rounds its argument to the nearest integer value and
// returns a numeric value of type long long int. If the
// rounded value is outside the range of type long long
// int, the value of the result is undefined and global
// variable errno is set to ERANGE.
//
// Returns
//
// The llrint function returns the rounded integer value.
//
// Remarks (C99 paragraph 7.9.6.5)
//
// The llrint function provides floating-to-integer conversion
// as prescribed by IEC 60559. It rounds according to the current
// rounding direction.	If the rounded value is outside the range
// of the return type, the numeric result is undefined and the
// invalid exception is raised.	 When no other exception is raised
// during the conversion and the result differs from the argument,
// the inexact exception is raised.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, September 2003
//

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long long llrint(double Arg)
{
    int		CrntExcepts, Exponent;
    double	DblIntVal;

    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    ROUND_TO_INT(Arg, DblIntVal);

    (void)frexp(DblIntVal, &Exponent);

    if ((Exponent > 63) || ! isfinited(Arg))
    {
	__math_set_errno(EDOM);
	DblIntVal = -LLONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != DblIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long long) DblIntVal;
}
