// ------
// fmal.c
// ------
//
// Name
//
// Function fmal - long double precision floating-point multiply-add
//
// Synopsis:
//
// #include <math.h>
//
// long double fmal(long double x, long double y, long double z);
//
// Purpose:
//
// Computes (x*y) + z rounded as one ternary operation.  The product-sum
// is computed as though to infinite precision and rounded once according
// to the current rounding mode.
//
// The procedure is:
//
// 1. Examine parameters in calling sequence for legitimacy:
//
//    o If either x or y is a NaN, a NaN is returned
//    o If either x or y is an infinity and the other is zero:
//
//	1. set errno to EDOM;
//	2. set invalid floating-point exception;
//	3. return a NaN.
//
//    o If neither x nor y is an infinity and the other zero,
//	and z is a NaN, return a NaN.
//
// 2. Calculate product-sum
//
//    If the final result is not finite (whether a NaN or Inf):
//
//	1. set errno to EDOM;
//	2. set invalid floating-point exception;
//	3. return a NaN.
//

#include <errno.h>
#include </gputil/fdlibml.h>
#include <fenv.h>
#include <stdio.h>

long double fmal(long double x, long double y, long double z)
{
    long double Retval;

    if (isnanl(x) || isnanl(y))
    {
	Retval = NAN;
    }
    else if ((isinfl(x) && (y == 0.0L))
    	 ||  (isinfl(y) && (x == 0.0L)))
    {
    	__math_set_errno(EDOM);
	__fp_raise_except(FE_INVALID);
	Retval = NAN;
    }
    else if (isnanl(z))
    {
	Retval = NAN;
    }
    else				// Calculate Product-Sum
    {
    	long double	HeadP, TailP, HeadZ, TailZ;

	feclearexcept(FE_OVERFLOW | FE_UNDERFLOW);

	SlMpyLdbl(x, y, &HeadP, &TailP);
	DlAddLdbl(HeadP, TailP, z, 0.0L, &HeadZ, &TailZ);

	Retval = HeadZ + TailZ;

//Retval = x*y+z;
	if (fetestexcept(FE_OVERFLOW | FE_UNDERFLOW))
	{
	    __math_set_errno(ERANGE);
	    Retval = NAN;
	}
	else if (!isfinitel(Retval))
	{
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	    Retval = NAN;
	}
    }

    return Retval;
}
