/*							exp2l.c
 *
 *	Base 2 exponential function, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, exp2l();
 *
 * y = exp2l( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns 2 raised to the x power.
 *
 * Range reduction is accomplished by separating the argument
 * into an integer k and fraction f such that
 *     x    k  f
 *    2	 = 2  2.
 *
 * A Pade' form
 *
 *   1 + 2x P(x**2) / (Q(x**2) - x P(x**2) )
 *
 * approximates 2**x in the basic range [-0.5, 0.5].
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE	+-16300	    300000	9.1e-20	    2.6e-20
 *
 *
 * See exp.c for comments on error amplification.
 *
 *
 * ERROR MESSAGES:
 *
 *   message	     condition	    value returned
 * exp2l underflow   x < -16382	       0.0
 * exp2l overflow    x >= 16384	      MAXNUM
 *
 */
// 
// Cephes Math Library Release 2.7:  May, 1998
// Copyright 1984, 1991, 1998 by Stephen L. Moshier
// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, December 2001 & October 2003
//


#include "errno.h"
#include <fdlibml.h>
#include <fenv.h>

static long double P[] = 
{
    6.0614853552242266094567E1L,
    3.0286971917562792508623E4L,
    2.0803843631901852422887E6L,
};
static long double Q[] = 
{
/* 1.0000000000000000000000E0,*/
    1.7492876999891839021063E3L,
    3.2772515434906797273099E5L,
    6.0027204078348487957118E6L,
};

long double
exp2l(long double x)
{
    long double p, q, px, xx;
    int     n;

    if (isnanl(x))
    {		
	return (x);
    }
    if (x == 0.0)
    {
	return (1.0);
    }
    if (x >= LDBL_MAX_EXP)
    {
	if (isinfl(x))
	{
	    return (x);
	}
	__math_set_errno(ERANGE);
	__fp_raise_except(FE_OVERFLOW);
	return (HUGE_VALL);
    }

    if (x < LDBL_MIN_EXP-1)
    {
    	if (isinfl(x))
	{
	    return (0.0L);
	}
    	__math_set_errno(ERANGE);
	__fp_raise_except(FE_UNDERFLOW);    		       
    }

    xx = x;				/* save x */
/* separate into integer and fractional parts */
    px = floorl(x + 0.5L);
    n = px;
    x = x - px;

/* rational approximation
 * exp2(x) = 1.0 +  2xP(xx)/(Q(xx) - P(xx))
 * where xx = x**2
 */
    xx = x * x;
    p = P[2] + xx * (P[1] + xx * P[0]);
    q = Q[2] + xx * (Q[1] + xx * (Q[0] + xx));
    px = p * x;

    x = px / (q - px);
    x = 1.0L + ldexpl(x, 1);

/* scale by power of 2 */
    x = ldexpl(x, n);
    return (x);
}
